package com.xdja.pki.ldap.config;

import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.keystore.utils.GMSSLKeyStoreUtils;
import com.xdja.pki.ldap.X509Utils;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.io.File;
import java.io.InputStream;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.Security;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.List;

@Component
@ConfigurationProperties(prefix = "ldap.server")
public class LDAPConfiguration {
    private Logger logger = LoggerFactory.getLogger(LDAPConfiguration.class);
    private String caCert;
    private String signCert;
    private String signKey;
    private int privateKeyIndex;
    private String privateKeyPassword;
    private String cryptoType;


    public String getCryptoType() {
        if (StringUtils.isEmpty(cryptoType)) {
            this.cryptoType = "YUNHSM";
        }
        return cryptoType;
    }

    public void setCryptoType(String cryptoType) throws Exception {
        this.cryptoType = cryptoType;
        if (!("YUNHSM".equalsIgnoreCase(this.cryptoType) ||
                "PCIE".equalsIgnoreCase(this.cryptoType) ||
                "BC".equalsIgnoreCase(this.cryptoType))) {
            logger.error("请配置正确的签名验签方式");
            throw new Exception("请配置正确的签名验签方式");
        }
        logger.info("签名验签方式为  " + this.cryptoType);
    }


    public int getPrivateKeyIndex() {
        return privateKeyIndex;
    }

    public String getPrivateKeyPassword() {
        return privateKeyPassword;
    }

    public void setPrivateKeyIndex(int privateKeyIndex) {
        this.privateKeyIndex = privateKeyIndex;
    }

    public void setPrivateKeyPassword(String privateKeyPassword) {
        this.privateKeyPassword = privateKeyPassword;
    }
//
//    public List<X509Certificate> getCaCert() throws Exception {
//        return X509Utils.getX509CertificateFromPEM(caCert);
//    }
//
//    public void setCaCert(String caCert) {
//        this.caCert = caCert;
//    }

    public List<X509Certificate> getCaCerts() throws Exception {
        String caPem = getCaPem();
        if(isCaPemExist()){
            return X509Utils.getX509CertificateListFromFile(caPem);
        } else {
            X509Utils.convertP7BToCaPemJson(caCert, caPem);
            return X509Utils.getX509CertificateListFromP7B(caCert);//支持p7b文件部署
        }
    }

    public void setCaCert(String caCert) {
        this.caCert = caCert;
    }

    public String getCaPem() {
        String name = "ca.pem";
        int lastSlashPos = caCert.lastIndexOf("/" );
        return caCert.substring(0,lastSlashPos) + "/" + name;
    }

    private boolean isCaPemExist(){
        String caPem = getCaPem();
        File file = new File(caPem);
        return file.exists();
    }

    public void updateCaCert(X509Certificate certificate) throws Exception {
        String caPem = getCaPem();
        if(!isCaPemExist()){
            X509Utils.convertP7BToCaPemJson(caCert, caPem);
        }
        X509Utils.appendCertificateToPem(caPem, certificate);
    }

    public X509Certificate getSignCert() throws Exception {
        if (cryptoType.equalsIgnoreCase("YUNHSM")) {
            return X509Utils.getX509CertificateFromPEM(signCert);
        }
        String sign = "sign";
        KeyStore keyStore = readKeyStoreFromPath(signKey, "password".toCharArray());
        return (X509Certificate) GMSSLKeyStoreUtils.readCertificateFromKeyStore(keyStore, sign);
    }

    public void setSignCert(String signCert) {
        this.signCert = signCert;
    }

    public KeyPair getSignKey() throws Exception {
        String sign = "sign";
        KeyStore keyStore = readKeyStoreFromPath(signKey, "password".toCharArray());
        PrivateKey privateKey = GMSSLKeyStoreUtils.readPrivateKeyFromKeyStore(keyStore, "password".toCharArray(), sign);
        Certificate certificate = GMSSLKeyStoreUtils.readCertificateFromKeyStore(keyStore, sign);
        return new KeyPair(certificate.getPublicKey(), privateKey);
    }

    public void setSignKey(String signKey) {
        this.signKey = signKey;
    }

    public static KeyStore readKeyStoreFromPath(String filename, char[] pw) throws Exception {
        Security.addProvider(new BouncyCastleProvider());
        boolean flag;
        try {
            InputStream in = GMSSLX509Utils.readInputStreamFromPath(filename);
            KeyStore keyStore = KeyStore.getInstance("JKS");
            keyStore.load(in, pw);
            return keyStore;
        } catch (Exception e) {
            flag = false;
        }
        System.out.println(flag);
        if (!flag) {
            InputStream in = GMSSLX509Utils.readInputStreamFromPath(filename);
            KeyStore keyStore = KeyStore.getInstance("BKS", BouncyCastleProvider.PROVIDER_NAME);
            keyStore.load(in, pw);
            return keyStore;
        }
        return null;
    }
}
