package com.xdja.pki.ldap.config;

import com.xdja.pki.gmssl.core.utils.GMSSLECUtils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.crypto.utils.GMSSLECKeyUtils;
import com.xdja.pki.gmssl.keystore.utils.GMSSLKeyStoreUtils;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLCryptoType;
import com.xdja.pki.ldap.X509Utils;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPrivateKey;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPublicKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.io.File;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.security.spec.ECPoint;
import java.util.*;

@Component
@ConfigurationProperties(prefix = "ldap.server")
public class LDAPConfiguration {
    private Logger logger = LoggerFactory.getLogger(LDAPConfiguration.class);
    private String caCert;
    private String signCert;
    private String signKey;
    private int privateKeyIndex;
    private String privateKeyPassword;
    private String cryptoType;
    private String privateKey;


    public String getCryptoType() {
        if (StringUtils.isEmpty(cryptoType)) {
            this.cryptoType = "YUNHSM";
        }
        return cryptoType;
    }

    public void setCryptoType(String cryptoType) throws Exception {
        this.cryptoType = cryptoType;
        if (!("YUNHSM".equalsIgnoreCase(this.cryptoType) ||
                "PCIE".equalsIgnoreCase(this.cryptoType) ||
                "BC".equalsIgnoreCase(this.cryptoType))) {
            logger.error("请配置正确的签名验签方式");
            throw new Exception("请配置正确的签名验签方式");
        }
        logger.info("签名验签方式为  " + this.cryptoType);
    }

    public int getPrivateKeyIndex() {
        return privateKeyIndex;
    }

    public String getPrivateKeyPassword() {
        return privateKeyPassword;
    }

    public void setPrivateKeyIndex(int privateKeyIndex) {
        this.privateKeyIndex = privateKeyIndex;
    }

    public void setPrivateKeyPassword(String privateKeyPassword) {
        this.privateKeyPassword = privateKeyPassword;
    }

    public List<String> getPrivateKey() {
        String[] indexPwd = privateKey.split(",");
        return Arrays.asList(indexPwd);
    }

    public void setPrivateKey(String privateKey) {
        this.privateKey = privateKey;
    }

    //
//    public List<X509Certificate> getCaCert() throws Exception {
//        return X509Utils.getX509CertificateFromPEM(caCert);
//    }
//
//    public void setCaCert(String caCert) {
//        this.caCert = caCert;
//    }

    public List<X509Certificate> getCaCerts() throws Exception {
        String caPem = getCaPem();
        if(isCaPemExist()){
            return X509Utils.getX509CertificateListFromFile(caPem);
        } else {
            X509Utils.convertP7BToCaPemJson(caCert, caPem);
            return X509Utils.convertNestListToList(X509Utils.getX509CertificateListFromAllP7B(caCert));//支持p7b文件部署
    }
    }

    public void setCaCert(String caCert) {
        this.caCert = caCert;
    }

    /*public String getCaPem() {
        String name = "ca.pem";
        int lastSlashPos = caCert.lastIndexOf("/" );
        return caCert.substring(0,lastSlashPos) + "/" + name;
    }*/

    public String getCaPem() {
        String name = "ca.pem";
        return this.caCert + "/" + name;
    }

    private boolean isCaPemExist(){
        String caPem = getCaPem();
        File file = new File(caPem);
        return file.exists();
    }

    public void updateCaCert(X509Certificate certificate) throws Exception {
        String caPem = getCaPem();
        if(!isCaPemExist()){
            X509Utils.convertP7BToCaPemJson(caCert, caPem);
        }
        X509Utils.appendCertificateToPem(caPem, certificate);
    }

    public void updateCaCert(X509Certificate certificate, List<X509Certificate> caCerts) throws Exception {
        String caPemPath = getCaPemPathByCaCerts(caCerts, caCert);
        List<X509Certificate> pemCertList = X509Utils.getX509CertificateListFromFile(caPemPath);
        if(!pemCertList.contains(certificate)){
            X509Utils.appendCertificateToPem(caPemPath, certificate);
        }
    }

    public X509Certificate getSignCert() throws Exception {
        initInstance();
        if (cryptoType.equalsIgnoreCase("YUNHSM") && this.privateKeyPassword != null) {
            return X509Utils.getX509CertificateFromPEM(signCert);
        }
        String sign = "sign";
        KeyStore keyStore = X509Utils.readKeyStoreFromPath(signKey, "password".toCharArray());
        return (X509Certificate) GMSSLKeyStoreUtils.readCertificateFromKeyStore(keyStore, sign);
    }

    public List<X509Certificate> getAllSignCert(String cryptoType) throws Exception {
        initInstance(cryptoType);
        List<X509Certificate> certificateList = new ArrayList<>();
        certificateList = X509Utils.getX509CertificateFromAllPEM(signCert);

        String sign = "sign";
        List<KeyStore> keyStores = X509Utils.readAllKeyStoreFromPath(signKey, "password".toCharArray());
        if (keyStores != null && keyStores.size() > 0) {
            for (int i = 0; i < keyStores.size(); i++) {
                certificateList.add((X509Certificate) GMSSLKeyStoreUtils.readCertificateFromKeyStore(keyStores.get(i), sign));
            }
        }
        return certificateList;
    }

    public void setSignCert(String signCert) {
        this.signCert = signCert;
    }

    public KeyPair getSignKey() throws Exception {
        String sign = "sign";
        KeyStore keyStore = X509Utils.readKeyStoreFromPath(signKey, "password".toCharArray());
        PrivateKey privateKey = GMSSLKeyStoreUtils.readPrivateKeyFromKeyStore(keyStore, "password".toCharArray(), sign);
        Certificate certificate = GMSSLKeyStoreUtils.readCertificateFromKeyStore(keyStore, sign);
        return new KeyPair(certificate.getPublicKey(), privateKey);
    }

    public List<KeyPair> getAllSignKey() throws Exception {
        List<KeyPair> keyPairList = new ArrayList<>();
        String sign = "sign";
        List<KeyStore> keyStores = X509Utils.readAllKeyStoreFromPath(signKey, "password".toCharArray());
        if (keyStores != null && keyStores.size() > 0) {
            for (int i = 0; i < keyStores.size(); i++) {
                PrivateKey privateKey = GMSSLKeyStoreUtils.readPrivateKeyFromKeyStore(keyStores.get(i), "password".toCharArray(), sign);
                Certificate certificate = GMSSLKeyStoreUtils.readCertificateFromKeyStore(keyStores.get(i), sign);
                KeyPair keyPair = new KeyPair(certificate.getPublicKey(), privateKey);
                keyPairList.add(keyPair);
            }
        }
        return keyPairList;
    }

    public void setSignKey(String signKey) {
        this.signKey = signKey;
    }


    public void initInstance(){
        if("YUNHSM".equalsIgnoreCase(this.cryptoType)){
            if (this.privateKeyPassword == null){
                try{
                    GMSSLPkiCryptoInit.getSancHsmInstance();
                } catch (Exception e){
                    logger.error("获取Sanc hsm instance异常");
                }

            } else {
                GMSSLPkiCryptoInit.getXdjaYunHsmInstance();
            }
        }
        if("PCIE".equalsIgnoreCase(this.cryptoType)){
            GMSSLPkiCryptoInit.getPcieInstance();
        }
        if("BC".equalsIgnoreCase(this.cryptoType)){
            GMSSLPkiCryptoInit.getBCInstance();
        }
    }

    public void initInstance(String cryptoType){
        if("YUNHSM".equalsIgnoreCase(cryptoType)){
            if (this.privateKeyPassword == null){
                try{
                    GMSSLPkiCryptoInit.getSancHsmInstance();
                } catch (Exception e){
                    logger.error("获取Sanc hsm instance异常");
                }

            } else {
                GMSSLPkiCryptoInit.getXdjaYunHsmInstance();
            }
        }
        if("PCIE".equalsIgnoreCase(cryptoType)){
            GMSSLPkiCryptoInit.getPcieInstance();
        }
        if("BC".equalsIgnoreCase(cryptoType)){
            GMSSLPkiCryptoInit.getBCInstance();
        }
    }

    public Map<String, Object> getCAToLDAPMap(String cryptoType) throws Exception {
        Map<String, Object> caLdapMap = new HashMap<>();
        List<List<X509Certificate>> caCertsList = getAllCAListFromPath(caCert);
        List<X509Certificate> signCertList = this.getAllSignCert(cryptoType);
        List<KeyPair> keyPairs = this.getAllSignKey();
        List<String> privateKeys = this.getPrivateKey();
        for (int i = 0; i < caCertsList.size(); i++) {
            List<Object> tempCALDAPList = new ArrayList<>();
            List<X509Certificate> x509Certificates = caCertsList.get(i);
            //list列表，大小为3； 0 为ca证书列表，1为服务器证书，2为keypair或者私钥索引
            tempCALDAPList.add(x509Certificates);
            String keyStr = X509Utils.getKeyFromCertificate(x509Certificates.get(x509Certificates.size()-1));
            for (int m = x509Certificates.size()-1 ; m >= 0; m--) {
                X509Certificate x509Cert = x509Certificates.get(m);
                String signAlg = x509Cert.getSigAlgName();
                //根据服务器证书和Ca证书签发依赖性
                Iterator<X509Certificate> iterator = signCertList.iterator();
                //for (int j = 0; j < signCertList.size(); j++) {
                while(iterator.hasNext()){
                    X509Certificate serverCert = iterator.next();
                    if (X509Utils.verifyCert(x509Certificates, serverCert)) {
                        tempCALDAPList.add(serverCert);
                        //根据服务器证书和公私钥添加映射
                        if ("BC".equalsIgnoreCase(cryptoType)) {
                            for (int k = 0; k < keyPairs.size(); k++) {
                                KeyPair keyPair = keyPairs.get(k);
                                if (serverCert.getPublicKey().equals(keyPair.getPublic())) {
                                    tempCALDAPList.add(keyPair);
                                    iterator.remove();
                                    continue;
                                }
                            }
                        }
                        //比对硬算法
                        if ("YUNHSM".equalsIgnoreCase(cryptoType) && signAlg.contains("SM2") && serverCert.getSigAlgName().contains("SM2")) {
                            BCECPublicKey bcecPublicKeyFromCert = (BCECPublicKey) serverCert.getPublicKey();
                            for (int l = 0; l < privateKeys.size(); l++) {
                                String[] privateKey = privateKeys.get(l).split("-");
                                if (privateKey.length != 2) {
                                    continue;
                                }
                                BCECPublicKey bcecPublicKeyFromHsm = (BCECPublicKey) GMSSLECKeyUtils.getSignPublicKeyByYunhsm(Integer.parseInt(privateKey[0]), GMSSLECUtils.SM2p256);
                                if (bcecPublicKeyFromCert.equals(bcecPublicKeyFromHsm)) {
                                    tempCALDAPList.add(privateKey);
                                    iterator.remove();
                                    continue;
                                }
                            }
                        }
                    }
                }
                if(tempCALDAPList.size() != 3){
                    continue;
                }
                caLdapMap.put(keyStr, tempCALDAPList);
            }
        }
        return caLdapMap;
    }

    /**
     * @MethodName: getAllCAListFromPath
     * @Description: 获取指定路径下所有的证书列表
     * @Param: path
     * @Return: java.util.List<java.util.List<java.security.cert.X509Certificate>>
     **/
    private List<List<X509Certificate>> getAllCAListFromPath(String path) throws Exception {
        String[] allCaFiles = new File(path).list();
        List<List<X509Certificate>> caCerts = new ArrayList<>();
        for (int i = 0; i < allCaFiles.length; i++) {
            List<X509Certificate> singleCaCerts = new ArrayList<>();
            String singleCa = allCaFiles[i];
            String[] singleCaFiles = new File(path + "/" + singleCa).list();
            if(singleCaFiles.length == 1){
                String p7bFilePath = path + "/" + singleCa + "/" + singleCaFiles[0];
                String pemFilePath = path + "/" + singleCa + "/" + "ca.pem";
                X509Utils.convertP7BToCaPemJson(p7bFilePath, pemFilePath);
                singleCaCerts = X509Utils.getX509CertificateListFromP7B(p7bFilePath);
            } else {
                String pemFilePath = path + "/" + singleCa + "/" + "ca.pem";
                List<X509Certificate> oldSingleCaCerts = X509Utils.getX509CertificateListFromFile(pemFilePath);
                List<X509Certificate> tempCertList = new ArrayList<>();
                for (int j = 0; j < singleCaFiles.length; j++) {
                    String p7bFilePath = null;
                    if(!"ca.pem".equalsIgnoreCase(singleCaFiles[j])){
                        p7bFilePath = path + "/" + singleCa + "/" + singleCaFiles[j];
                        tempCertList = X509Utils.getX509CertificateListFromP7B(p7bFilePath);
                    }
                    if(!oldSingleCaCerts.containsAll(tempCertList)){
                        for (int k = 0; k < tempCertList.size(); k++) {
                            X509Utils.appendCertificateToPem(pemFilePath, tempCertList.get(k));
                        }
                    }
                }
                singleCaCerts = X509Utils.getX509CertificateListFromFile(pemFilePath);
            }
            caCerts.add(singleCaCerts);
        }
        return caCerts;
    }

    private String getCaPemPathByCaCerts(List<X509Certificate> caCertList, String path) throws Exception {
        String[] allCaFiles = new File(path).list();
        String pemFilePath = null;
        for (int i = 0; i < allCaFiles.length; i++) {
            List<X509Certificate> singleCaCerts = new ArrayList<>();
            String singleCa = allCaFiles[i];
            String[] singleCaFiles = new File(path + "/" + singleCa).list();
            if (singleCaFiles.length == 1) {
                String p7bFilePath = path + "/" + singleCa + "/" + singleCaFiles[0];
                pemFilePath = path + "/" + singleCa + "/" + "ca.pem";
                X509Utils.convertP7BToCaPemJson(p7bFilePath, pemFilePath);
                singleCaCerts = X509Utils.getX509CertificateListFromP7B(p7bFilePath);
            } else {
                pemFilePath = path + "/" + singleCa + "/" + "ca.pem";
                List<X509Certificate> oldSingleCaCerts = X509Utils.getX509CertificateListFromFile(pemFilePath);
                List<X509Certificate> tempCertList = new ArrayList<>();
                for (int j = 0; j < singleCaFiles.length; j++) {
                    String p7bFilePath = null;
                    if (!"ca.pem".equalsIgnoreCase(singleCaFiles[j])) {
                        p7bFilePath = path + "/" + singleCa + "/" + singleCaFiles[j];
                        tempCertList = X509Utils.getX509CertificateListFromP7B(p7bFilePath);
                    }
                    if (!oldSingleCaCerts.containsAll(tempCertList)) {
                        X509Utils.convertP7BToCaPemJson(p7bFilePath, pemFilePath);
                    }
                }
                singleCaCerts = X509Utils.getX509CertificateListFromFile(pemFilePath);
            }
            if(singleCaCerts.containsAll(caCertList)) {
                return pemFilePath;
            }
        }
        return null;
    }
}
