package com.xdja.pki.auditlog.dao;

import com.xdja.pki.auditlog.dao.model.AuditLogListDO;
import com.xdja.pki.core.bean.PageInfo;
import com.xdja.pki.auditlog.dao.model.AuditLogDO;
import com.xdja.pki.core.exception.DAOException;
import com.xdja.pki.dao.BaseJdbcDao;
import org.apache.commons.lang3.StringUtils;
import org.nutz.dao.Cnd;
import org.springframework.stereotype.Repository;

import java.util.ArrayList;
import java.util.List;

/**
 * 操作日志表 - 数据访问层
 *
 * @author syg
 */
@Repository
public class AuditLogDao extends BaseJdbcDao {
    /**
     * 存储审计日志对象
     *
     * @param logDO 审计日志对象
     * @return 审计日志对象
     */
    public AuditLogDO saveLog(AuditLogDO logDO) {
        return this.daoTemplate.insert(logDO);
    }

    public AuditLogDO getLogById(int id) {
        Cnd cnd = Cnd.where("id", "=", id);
        cnd.orderBy("gmt_create", "desc");
        try {
            List<AuditLogDO> list = this.daoTemplate.query(AuditLogDO.class, cnd);
            if (list.size() != 1) {
                throw new DAOException("根据ID获取到:" + list.size() + "个数据!");
            }
            return list.get(0);
        } catch (Exception e) {
            throw new DAOException("根据id查询审计日志数据库异常", e);
        }
    }

    /**
     * 如果更新成功，返回 1，否则，返回 0
     */
    public int updateLog(AuditLogDO logDO) {
        return this.daoTemplate.update(logDO);
    }

    /**
     * 根据 id 查询 审计日志 对象
     *
     * @param id 序号
     * @return 日志信息
     */
    public AuditLogDO get(int id) {
        Cnd cnd = Cnd.where("id", "=", id);
        cnd.orderBy("gmt_create", "desc");

        try {
            List<AuditLogDO> list = this.daoTemplate.query(AuditLogDO.class, cnd);
            if (list.size() != 1) {
                throw new DAOException("根据ID获取到:" + list.size() + "个数据!");
            }
            return list.get(0);
        } catch (Exception e) {
            throw new DAOException("根据id查询审计日志数据库异常", e);
        }
    }

    /**
     * 读取 审计日志列表 分页数据
     *
     * @param pageNo          页码
     * @param pageSize        每页个数
     * @param operatorSubject 操作员证书主体 过滤项
     * @param operatorType    操作类型 过滤项
     * @param isAudit         是否审核 过滤项
     * @param startTime       开始时间 过滤项
     * @param endTime         结束时间 过滤项
     * @return 日志列表信息
     */
    public Object getLogList(Integer pageNo, Integer pageSize, String operatorSubject, Integer operatorType, Integer isAudit, String startTime, String endTime) {
        Cnd cnd = Cnd.NEW();
        if (StringUtils.isNotBlank(operatorSubject)) {
            cnd.and("operator_subject", "LIKE", "%"+operatorSubject+"%");
        }
        if (null != operatorType) {
            cnd.and("operator_type", "=", operatorType);
        }
        if (null != isAudit) {
            cnd.and("is_audit", "=", isAudit);
        }

        if (StringUtils.isNotBlank(startTime) && StringUtils.isNotBlank(endTime)) {
            cnd.and("gmt_create", ">=", startTime + " 00:00:00");
            cnd.and("gmt_create", "<=", endTime + " 23:59:59");
        }
        //id 降序
        cnd.orderBy("id", "desc");

        try {
            if (pageNo == 0 && pageSize == 0) {
                return this.daoTemplate.query(AuditLogListDO.class, cnd);
            }
            PageInfo pageInfo = new PageInfo(pageNo, pageSize, this.daoTemplate.count(AuditLogListDO.class, cnd));
            pageInfo.setDatas(this.daoTemplate.query(AuditLogListDO.class, cnd, createPager(pageNo, pageSize)));
            logger.debug("list page info {} ", pageInfo);
            return pageInfo;
        } catch (Exception e) {
            throw new DAOException("分页查询模板信息列表时数据库异常", e);
        }
    }

    public List<AuditLogDO> getAuditLogListbyPeriod( String startTime, String endTime) {
        Cnd cnd = Cnd.NEW();
        if (StringUtils.isNotBlank(startTime) && StringUtils.isNotBlank(endTime)) {
            cnd.and("gmt_create", ">=", startTime);
            cnd.and("gmt_create", "<=", endTime);
        }
        //id 升序
        cnd.orderBy("id", "desc");

        try {
            return this.daoTemplate.query(AuditLogDO.class, cnd);
        } catch (Exception e) {
            throw new DAOException("分页查询模板信息列表时数据库异常", e);
        }
    }


    /**
     * @MethodName: archiveList
     * @Description: 审计日志列表,归档数据
     * @Param: operatorSubject
     * @Param: operatorType
     * @Param: isAudit
     * @Param: startTime
     * @Param: endTime
     * @Return: java.lang.Object
     * @Author: songxuetao
     **/
    public Object archiveList(String operatorSubject, Integer operatorType, Integer isAudit, String startTime, String endTime) {
        Cnd cnd = Cnd.NEW();

        if (!com.alibaba.druid.util.StringUtils.isEmpty(operatorSubject)) {
            cnd.and("operator_subject", "LIKE", "%" + operatorSubject + "%");
        }

        if (null != operatorType) {
            cnd.and("operator_type", "=", operatorType);
        }

        if (null != isAudit) {
            cnd.and("is_audit", "=", isAudit);
        }

        if (!com.alibaba.druid.util.StringUtils.isEmpty(startTime) && !com.alibaba.druid.util.StringUtils.isEmpty(endTime)) {
            cnd.and("gmt_create", ">", startTime + " 00:00:00");
            cnd.and("gmt_create", "<", endTime + " 23:59:59");
        }
        //归档数据不需要排序
        //cnd.orderBy("gmt_create", "desc");

        try {
            return this.daoTemplate.query(AuditLogDO.class, cnd);
        } catch (Exception e) {
            throw new DAOException("手动归档查询审计日志列表时数据库异常", e);
        }
    }

    /**
     * @MethodName: getTotalRecords
     * @Description: 获取操作日志的记录总数
     * @Param:
     * @Return: int
     * @Author: songxuetao
     **/
    public int getTotalRecords(){
        return this.daoTemplate.count(AuditLogDO.class);
    }

    public int getOverAuditRecordNum(String compareTime){
        Cnd cnd = Cnd.NEW();
        //针对未审核的日志
        cnd.and("is_audit", "=", 1);
        //小于compareTime的属于超期日志
        cnd.and("gmt_create", "<", compareTime + " 00:00:00");
        return this.daoTemplate.count(AuditLogDO.class, cnd);
    }

    public Object autoArchiveList(String compareTime){
        Cnd cnd = Cnd.NEW();
        //小于compareTime的属于自动归档需要归档的日志
        cnd.and("gmt_create", "<", compareTime);
        try {
            return this.daoTemplate.query(AuditLogDO.class, cnd);
        } catch (Exception e) {
            throw new DAOException("自动归档查询审计日志列表时数据库异常", e);
        }
    }

    public int deleteAuditLogList(List<AuditLogDO> auditLogDOS){
        return this.daoTemplate.delete(auditLogDOS);
    }
}