package com.xdja.pki.monitor.service;

import com.xdja.pki.core.bean.CoreResult;
import com.xdja.pki.core.utils.DateUtils;
import com.xdja.pki.monitor.bean.*;
import com.xdja.pki.monitor.collector.CollectCallBack;
import com.xdja.pki.monitor.collector.Collector;
import com.xdja.pki.monitor.common.ErrorEnum;
import com.xdja.pki.monitor.dao.MonitorCpuDao;
import com.xdja.pki.monitor.dao.model.MonitorCpuDO;
import com.xdja.pki.monitor.factory.CollectorFactory;
import com.xdja.pki.monitor.service.bean.MonitorCpuVO;
import com.xdja.pki.monitor.util.MonitorUtil;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 监控接口实现
 *
 * @author yangmenghao
 * @date 2020/06/30 14:19:36
 */
@Service
public class MonitorServiceImpl implements MonitorService, InitializingBean {
	private final static Logger log = LoggerFactory.getLogger(MonitorServiceImpl.class);
	@Resource
	private MonitorCpuDao monitorCpuDao;
	private Collector<CpuData> cpuCollector;
	private Collector<DiskData> diskCollector;
	private Collector<MemoryData> memoryCollector;
	private Collector<NicData> nickCollector;
	private Collector<SystemRuntimeData> systemRunTimeCollector;

	@Override
	public void afterPropertiesSet() throws Exception {
		cpuCollector = CollectorFactory.createCollector(CpuData.class);
		diskCollector = CollectorFactory.createCollector(DiskData.class);
		memoryCollector = CollectorFactory.createCollector(MemoryData.class);
		nickCollector = CollectorFactory.createCollector(NicData.class);
		systemRunTimeCollector = CollectorFactory.createCollector(SystemRuntimeData.class);
	}

	@Override
	public CoreResult monitorMemoryData(long current) {
		MemoryData[] memoryData = {new MemoryData()};
		memoryCollector.doCollect(new CollectCallBack<MemoryData>() {
			@Override
			public void onSuccess(MemoryData data) {
				memoryData[0] = data;
			}

			@Override
			public void onError(String errorMsg) {
				log.warn("[MonitorServiceImpl#monitorMemoryData]{}| call CollectCallBack fail,{}.", current, errorMsg);
				memoryData[0] = null;
			}
		});
		if (memoryData[0] == null) {
			log.info("[MonitorServiceImpl#monitorMemoryData]{}| call AbstractCollector.process error .", current);
			return CoreResult.failure(ErrorEnum.SERVER_INTERNAL_EXCEPTION.resp());
		}
		return CoreResult.success(memoryData[0]);
	}


	@Override
	public CoreResult monitorDiskData(long current) {
		DiskData[] diskData = {new DiskData()};
		diskCollector.doCollect(new CollectCallBack<DiskData>() {

			@Override
			public void onSuccess(DiskData data) {
				diskData[0] = data;
			}

			@Override
			public void onError(String errorMsg) {
				log.info("[MonitorServiceImpl#monitorDiskData]{}| call AbstractCollector.process error {}.", current, errorMsg);
				diskData[0] = null;
			}
		});
		if (diskData[0] == null) {
			log.info("[MonitorServiceImpl#monitorDiskData]{}| call AbstractCollector.process error .", current);
			return CoreResult.failure(ErrorEnum.SERVER_INTERNAL_EXCEPTION.resp());
		}
		return CoreResult.success(diskData[0]);
	}

	@Override
	public CoreResult getCpuData(long current) {
		List<MonitorCpuDO> monitorCpuList = monitorCpuDao.queryRecentThreeHourData(current);
		if (CollectionUtils.isEmpty(monitorCpuList)) {
			return CoreResult.success(CollectionUtils.EMPTY_COLLECTION);
		}
		List<MonitorCpuVO> dataList = convertData(monitorCpuList);
		CoreResult result = CoreResult.success(dataList);
		log.info("[MonitorServiceImpl#getCpuData]{}| result-->{}.", current, result);
		return result;
	}

	@Override
	public CoreResult monitorCpu(long current) {
		CpuData[] cpuData = {new CpuData()};
		cpuCollector.doCollect(new CollectCallBack<CpuData>() {
			@Override
			public void onSuccess(CpuData data) {
				cpuData[0] = data;
			}

			@Override
			public void onError(String errorMsg) {
				log.warn("[MonitorServiceImpl#monitorCpu]{}| call CollectCallBack fail,{}.", current, errorMsg);
				cpuData[0] = null;
			}
		});
		log.info("[MonitorServiceImpl#monitorCpu]{}|{}.", current, cpuData[0]);
		return CoreResult.success(cpuData[0]);
	}

	@Override
	public CoreResult monitorSystemRunTimeData(long current) {
		SystemRuntimeData[] systemRuntimeData = {new SystemRuntimeData()};
		systemRunTimeCollector.doCollect(new CollectCallBack<SystemRuntimeData>() {
			@Override
			public void onSuccess(SystemRuntimeData data) {
				systemRuntimeData[0] = data;
			}

			@Override
			public void onError(String errorMsg) {
				log.warn("[MonitorServiceImpl#monitorSystemRunTimeData]{}| call AbstractCollector.process error {}.", current, errorMsg);
				systemRuntimeData[0] = null;
			}
		});
		if (systemRuntimeData[0] == null) {
			log.info("[MonitorServiceImpl#monitorSystemRunTimeData]{}| call AbstractCollector.process error .", current);
			return CoreResult.failure(ErrorEnum.SERVER_INTERNAL_EXCEPTION.resp());
		}
		Map<String, String> resultMap = new HashMap<>(4);
		resultMap.put("runTime", MonitorUtil.secondToDate(systemRuntimeData[0].getRuntime()));
		resultMap.put("lastCollectTime", DateUtils.longToStr(current));
		return CoreResult.success(resultMap);
	}

	@Override
	public CoreResult monitorNickData(long current) {
		NicData[] nicData = {new NicData()};
		nickCollector.doCollect(new CollectCallBack<NicData>() {
			@Override
			public void onSuccess(NicData data) {
				nicData[0] = data;
			}

			@Override
			public void onError(String errorMsg) {
				log.warn("[MonitorServiceImpl#monitorNickData]{}| call AbstractCollector.process error {}.", current, errorMsg);
				nicData[0] = null;
			}
		});
		if (nicData[0] == null) {
			log.info("[MonitorServiceImpl#monitorNickData]{}| call AbstractCollector.process error .", current);
			return CoreResult.failure(ErrorEnum.SERVER_INTERNAL_EXCEPTION.resp());
		}
		return CoreResult.success(nicData[0]);
	}

	private List<MonitorCpuVO> convertData(List<MonitorCpuDO> monitorCpuList) {
		List<MonitorCpuVO> dataList = new ArrayList<>(monitorCpuList.size());
		for (MonitorCpuDO monitorCpuDO : monitorCpuList) {
			dataList.add(new MonitorCpuVO(monitorCpuDO.getUtilization().setScale(2, BigDecimal.ROUND_HALF_UP),
					DateUtils.dateToString(monitorCpuDO.getGmtCreate(), DateUtils.FORMAT_ONE)));
		}
		return dataList;
	}

}
