package com.xdja.pki.monitor.util;

import java.math.BigDecimal;

/**
 * 提供精确的浮点数运算(包括加 、 减 、 乘 、 除 、 四舍五入)工具类
 *
 * @author yangmenghao
 * @date 2020/07/02
 */
public class BigDecimalUtil {

	/**
	 * 除法运算默认精度
	 */
	private static final int DEF_DIV_SCALE = 5;

	/**
	 * 精确加法
	 */
	public static double add(double value1, double value2) {
		return add(BigDecimal.valueOf(value1), BigDecimal.valueOf(value2)).doubleValue();
	}

	/**
	 * 精确加法
	 */
	public static double add(String value1, String value2) {
		return add(new BigDecimal(value1), new BigDecimal(value2)).doubleValue();
	}

	public static BigDecimal add(BigDecimal value1, BigDecimal value2) {
		return value1.add(value2);
	}

	/**
	 * 精确减法
	 */
	public static double sub(double value1, double value2) {
		return subtract(BigDecimal.valueOf(value1), BigDecimal.valueOf(value2)).doubleValue();
	}

	/**
	 * 精确减法
	 */
	public static double sub(String value1, String value2) {
		return subtract(new BigDecimal(value1), new BigDecimal(value2)).doubleValue();
	}

	public static BigDecimal subtract(BigDecimal value1, BigDecimal value2) {
		return value1.subtract(value2);
	}

	/**
	 * 精确乘法
	 */
	public static double mul(double value1, double value2) {
		return multiply(new BigDecimal(value1), new BigDecimal(value2)).doubleValue();
	}

	/**
	 * 精确乘法
	 */
	public static double mul(String value1, String value2) {
		return multiply(new BigDecimal(value1), new BigDecimal(value2)).doubleValue();
	}

	public static BigDecimal multiply(BigDecimal value1, BigDecimal value2) {
		return value1.multiply(value2);
	}

	/**
	 * 精确除法 使用默认精度
	 */
	public static double div(double value1, double value2) throws IllegalAccessException {
		return divide(BigDecimal.valueOf(value1), BigDecimal.valueOf(value2), DEF_DIV_SCALE).doubleValue();
	}

	public static BigDecimal div(long value1, long value2) {
		try {
			return divide(BigDecimal.valueOf(value1), BigDecimal.valueOf(value2), DEF_DIV_SCALE);
		} catch (IllegalAccessException e) {
			return new BigDecimal(0);
		}
	}

	/**
	 * 精确除法 使用默认精度
	 */
	public static double div(String value1, String value2) throws IllegalAccessException {
		return divide(new BigDecimal(value1), new BigDecimal(value2), DEF_DIV_SCALE).doubleValue();
	}

	/**
	 * 精确除法 使用默认精度
	 */
	public static BigDecimal div(BigDecimal value1, BigDecimal value2) throws IllegalAccessException {
		return divide(value1, value2, DEF_DIV_SCALE);
	}

	/**
	 * 精确除法
	 *
	 * @param scale 精度
	 */
	public static BigDecimal divide(BigDecimal value1, BigDecimal value2, int scale) throws IllegalAccessException {
		if (scale < 0) {
			throw new IllegalAccessException("精确度不能小于0");
		}
		return value1.divide(value2, scale, BigDecimal.ROUND_HALF_UP);
	}


	/**
	 * 四舍五入
	 *
	 * @param scale 小数点后保留几位
	 */
	public static double round(double v, int scale) {
		try {
			return divide(BigDecimal.valueOf(v), new BigDecimal(1), scale).doubleValue();
		} catch (IllegalAccessException e) {
			return 0.0;
		}
	}

	/**
	 * 四舍五入
	 *
	 * @param scale 小数点后保留几位
	 */
	public static double round(String v, int scale) throws IllegalAccessException {
		return divide(new BigDecimal(v), new BigDecimal(1), scale).doubleValue();
	}

	/**
	 * 比较大小
	 */
	public static boolean equalTo(BigDecimal b1, BigDecimal b2) {
		if (b1 == null || b2 == null) {
			return false;
		}
		return 0 == b1.compareTo(b2);
	}

}
