package com.xdja.pki.monitor.util;

import com.sun.management.OperatingSystemMXBean;
import com.xdja.pki.monitor.bean.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.lang.management.ManagementFactory;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 监控采集工具类
 *
 * @author yangmenghao
 * @date 2020-07-02 15:15:16
 */
public class MonitorUtil {
	private final static Logger log = LoggerFactory.getLogger(MonitorUtil.class);
	private final static String SYMBOL_COMMA = ",";
	private final static String SYMBOL_COLON = ":";
	private final static String SYMBOL_SPOT = ".";
	private final static String SYMBOL_SPACE = " ";
	private final static String LINE_REED = "\n";
	private final static String CMD_MONITOR_CPU = "top -b -n 1 | sed -n '3p'";
	private final static String FILE_NET = "/proc/net/dev";
	private final static String FILE_UPTIME = "/proc/uptime";
	/*private static SystemInfo systemInfo = new SystemInfo();*/
	private static OperatingSystemMXBean osmxb = (OperatingSystemMXBean) ManagementFactory.getOperatingSystemMXBean();

	public static SystemRuntimeData monitorSystemRuntime(int osType) {
		long uptime = -1;
		if (osType == 1) {
			// java 启动时间
			long startTime = ManagementFactory.getRuntimeMXBean().getStartTime();
			uptime = (System.currentTimeMillis() - startTime) / 1000;
		} else {
			String uptimeStr = readFile(FILE_UPTIME);
			uptime = Long.parseLong(getUptimeSecond(uptimeStr));
		}
		return new SystemRuntimeData(uptime);
	}


	public static String secondToDate(long seconds) {
		long days = seconds / 86400;
		seconds = seconds % 86400;
		long hours = seconds / 3600;
		seconds = seconds % 3600;
		long minutes = seconds / 60;
		seconds = seconds % 60;
		long second = seconds % 60;
		return days + ":" + hours + ":" + minutes + ":" + second;
	}


	public static NicData monitorNetWork() {
		NicData nicData = new NicData();
		String netWorks = readFile(FILE_NET);
		if (isEmpty(netWorks)) {
			return null;
		}
		String[] netWorkArr = netWorks.split(LINE_REED);
		if (isEmpty(netWorkArr)) {
			return null;
		}
		List<NicDTO> netDataList = new ArrayList<>(netWorkArr.length);
		for (String netWork : netWorkArr) {
			NicDTO nicDTO = parserNetWork(netWork);
			if (nicDTO != null) {
				netDataList.add(nicDTO);
			}
		}
		nicData.setList(netDataList);
		return nicData;
	}

	private static NicDTO parserNetWork(String netDataStr) {
		long kb = 1024L;
		if (isEmpty(netDataStr)) {
			return null;
		}
		String[] metric = netDataStr.split(SYMBOL_COLON);
		if (isEmpty(metric) || metric.length == 1) {
			return null;
		}
		String netName = metric[0];
		if (netName.contains("lo")) {
			return null;
		}
		String dateStr = metric[1].trim();
		String[] dataArr = dateStr.split("\\s+");
		long receive = Long.parseLong(dataArr[0]) / kb;
		long transmit = Long.parseLong(dataArr[8]) / kb;
		return new NicDTO(netName, transmit, receive);
	}


	public static MemoryData monitorMemory() {
		long kb = 1024L;
		OperatingSystemMXBean osmxb = (OperatingSystemMXBean) ManagementFactory.getOperatingSystemMXBean();
		// 总的物理内存
		long totalMemorySize = osmxb.getTotalPhysicalMemorySize() / kb;
		// 剩余的物理内存
		long freePhysicalMemorySize = osmxb.getFreePhysicalMemorySize() / kb;
		// 已使用的物理内存
		long usedMemory = totalMemorySize - freePhysicalMemorySize;
		BigDecimal usage = BigDecimalUtil.div(usedMemory, totalMemorySize);
		MemoryData memoryData = new MemoryData();
		memoryData.setTotal(totalMemorySize);
		memoryData.setUse(usedMemory);
		memoryData.setFree(freePhysicalMemorySize);
		memoryData.setUsage(usage.doubleValue());
		return memoryData;
	}

	public static CpuData monitorCpuByJdk() {
		double cpuLoad = osmxb.getSystemCpuLoad();
		try {
			TimeUnit.SECONDS.sleep(1);
		} catch (InterruptedException e) {
			log.info("[MonitorUtil#monitorCpu] error!", e);
		}
		return new CpuData(osmxb.getSystemCpuLoad(), System.currentTimeMillis());
	}

	public static DiskData monitorDiskData() {
		long kb = 1024L;
		File[] files = File.listRoots();
		long total = 0L;
		long free = 0L;
		long used = 0L;
		for (File file : files) {
			total += file.getTotalSpace();
			free += file.getUsableSpace();
			used += file.getTotalSpace() - file.getUsableSpace();
		}
		return new DiskData(total / kb, used / kb, free / kb);
	}
	public static BigDecimal monitorCpuRateForLinux() {
		BigDecimal cpuUsage = null;
		BigDecimal idleUsage;
		Runtime runtime = Runtime.getRuntime();
		String[] cmdMonitorCpu = {"/bin/sh", "-c", "top -b -n 1 | sed -n '3p'"};
		try (BufferedReader buffer = new BufferedReader(new InputStreamReader(runtime.exec(cmdMonitorCpu).getInputStream()))) {
			String cpuInfo = buffer.readLine();
			String[] split = cpuInfo.substring(0, cpuInfo.indexOf("id")).split(SYMBOL_COMMA, -1);
			idleUsage = new BigDecimal(split[split.length - 1].trim());
			cpuUsage = new BigDecimal(100).subtract(idleUsage);
		} catch (IOException e) {
			System.out.println("[MonitorUtil#monitorCpuRateForLinux] fail," + e.getMessage());
		}
		return cpuUsage;
	}

	/**
	 * 返回运行时间的秒-更精确
	 *
	 * @return
	 */
	private static String getUptimeSecond(String str) {
		String result = "0";
		if (str != null && str.length() != 0) {
			if (str.contains(SYMBOL_SPACE)) {
				String[] re = str.split(SYMBOL_SPACE);
				if (re.length > 0) {
					String first = re[0];
					if (first.contains(SYMBOL_SPOT)) {
						result = first.substring(0, first.indexOf(SYMBOL_SPOT));
					} else {
						result = first;
					}
				}
			} else {
				if (str.contains(SYMBOL_SPOT)) {
					result = str.substring(0, str.indexOf(SYMBOL_SPOT));
				} else {
					result = str;
				}
			}
		}
		return result;
	}

	/**
	 * 执行系统命令
	 *
	 * @param cmd 命令
	 * @return 字符串结果
	 */
	private static String runCommand(String cmd) {
		StringBuilder info = new StringBuilder();
		Runtime runtime = Runtime.getRuntime();
		try {
			Process pos = runtime.exec(cmd);
			InputStreamReader isr = new InputStreamReader(pos.getInputStream());
			LineNumberReader lnr = new LineNumberReader(isr);
			String line;
			while ((line = lnr.readLine()) != null) {
				info.append(line).append(LINE_REED);
			}
			if (info.length() > 1) {
				info.setLength(info.length() - 1);
			}
		} catch (Exception e) {
			log.info("[MonitorUtil#runCommand] cmd {} Exception!", cmd, e);
			return null;
		}
		return info.toString();
	}

	private static String readFile(String fileName) {
		StringBuilder info = new StringBuilder();
		try (InputStream in = new FileInputStream(new File(fileName));
		     LineNumberReader reader = new LineNumberReader(new InputStreamReader(in));
		) {

			String line;
			while ((line = reader.readLine()) != null) {
				info.append(line).append(LINE_REED);
			}
			if (info.length() > 1) {
				info.setLength(info.length() - 1);
			}
		} catch (Exception e) {
			log.info("[MonitorUtil#readFile] fileName {} Exception!", fileName, e);
			return null;
		}
		return info.toString();
	}

	public static boolean isEmpty(final CharSequence cs) {
		int strLen;
		if (cs == null || (strLen = cs.length()) == 0) {
			return true;
		}
		for (int i = 0; i < strLen; i++) {
			if (!Character.isWhitespace(cs.charAt(i))) {
				return false;
			}
		}
		return true;
	}

	public static boolean isEmpty(final Collection<?> coll) {
		return coll == null || coll.isEmpty();
	}

	public static boolean isEmpty(final Object[] array) {
		return array == null || array.length == 0;
	}
}
