package com.xdja.ra.utils;

import org.bouncycastle.asn1.ASN1Encoding;
import org.bouncycastle.asn1.cms.ContentInfo;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaCertStore;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.cms.CMSProcessableByteArray;
import org.bouncycastle.cms.CMSSignedData;
import org.bouncycastle.cms.CMSSignedDataGenerator;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.Store;
import org.bouncycastle.util.encoders.Base64;
import org.bouncycastle.util.io.pem.PemObject;
import org.bouncycastle.util.io.pem.PemReader;
import org.bouncycastle.util.io.pem.PemWriter;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringWriter;
import java.security.Security;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

/**
 * @program: ca-server
 * @description: 封装与解析符合PKCS#7的证书证书链格式，符合0010 signedData规范
 * @author: ssh
 * @create: 2019-12-23 19:24
 */
public class SignedDataUtils {

    public static final String PKCS7_TYPE = "PKCS7";

    static {
        Security.addProvider(new BouncyCastleProvider());
    }

    /**
     * 进行p7b的pem格式转换
     * @param cmsSignedData
     * @author ssh
     * @return
     * @throws Exception
     */
    public static String writeP7bPem(CMSSignedData cmsSignedData) throws Exception {
        PemWriter pemWriter = null;
        try {
            PemObject pemObject = new PemObject(SignedDataUtils.PKCS7_TYPE,cmsSignedData.toASN1Structure().getEncoded(ASN1Encoding.DER));
            StringWriter stringWriter = new StringWriter();
            pemWriter = new PemWriter(stringWriter);
            pemWriter.writeObject(pemObject);
            pemWriter.flush();
            return stringWriter.toString();
        } catch (Exception e) {
            throw new Exception("将p7b对象转换为Pem格式异常",e);
        } /*finally {
            if (null == pemWriter) {
                pemWriter.close();
            }
        }*/
    }

    /**
     * 进行p7b的pem格式转换
     * @param contentInfo
     * @author ssh
     * @return
     * @throws Exception
     */
    public static String writeP7bPem(ContentInfo contentInfo) throws Exception {
        PemWriter pemWriter = null;
        try {
            PemObject pemObject = new PemObject(SignedDataUtils.PKCS7_TYPE,contentInfo.getEncoded(ASN1Encoding.DER));
            StringWriter stringWriter = new StringWriter();
            pemWriter = new PemWriter(stringWriter);
            pemWriter.writeObject(pemObject);
            pemWriter.flush();
            return stringWriter.toString();
        } catch (Exception e) {
            throw new Exception("将p7b对象转换为Pem格式异常",e);
        } /*finally {
            if (null == pemWriter) {
                pemWriter.close();
            }
        }*/
    }

    /**
     * 生成符合PKCS7 singedData格式的pem证书链
     * @param certificateList
     * @author ssh
     * @return
     * @throws Exception
     */
    public static String createCertChainByCerts(List<X509Certificate> certificateList) throws Exception {

        CMSSignedDataGenerator gen = new CMSSignedDataGenerator();
        try {
            CMSProcessableByteArray msg = new CMSProcessableByteArray("".getBytes());
            JcaCertStore jcaCertStore = new JcaCertStore(certificateList);
            gen.addCertificates(jcaCertStore);
            CMSSignedData cmsSignedData = gen.generate(msg);
            return SignedDataUtils.writeP7bPem(cmsSignedData.toASN1Structure());
        } catch (Exception e) {
            throw new Exception("创建证书链异常",e);
        }
    }

    /**
     * 解析base64编码或者pem编码的证书链
     * @param signedData
     * @return
     * @throws Exception
     */
    public static List<X509Certificate> resolveCertChain(byte[] signedData) throws Exception {
        InputStream inputStream = new ByteArrayInputStream(signedData);
        try {
            return resolvePemCertChain(inputStream);
        } catch (Exception e) {
            try {
                return resolveBase64CertChain(inputStream);
            } catch (Exception e1) {
                return resolveByteCertChain(signedData);
            }
        } finally {
            if (null != inputStream) {
                inputStream.close();
            }
        }
    }

    /**
     * 将字符流转换byte数据
     * @param in
     * @return
     * @throws Exception
     */
    public static byte[] convertBytesFromStream(FileInputStream in) throws Exception {
        try {
            int count = in.available();
            byte[] b = new byte[count];
            in.read(b);
            return b;
        } catch (Exception e) {
            e.printStackTrace();
            throw new Exception("流读取异常",e);
        }
    }

    /**
     * 解析pme格式的证书链
     * @param in
     * @author ssh
     * @return
     * @throws Exception
     */
    public static List<X509Certificate> resolvePemCertChain(InputStream in) throws Exception {

        try {
            Reader reader = new InputStreamReader(in);
            PemReader pemReader = new PemReader(reader);
            PemObject pemObject = pemReader.readPemObject();

            CMSSignedData cmsSignedDataResolve = new CMSSignedData(pemObject.getContent());
            return SignedDataUtils.getX509CertificateListFromSignedData(cmsSignedDataResolve);
        }catch (Exception e) {
            throw new Exception("转换pem格式证书链异常",e);
        } /*finally {
            if (null != in) {
                in.close();
            }
        }*/
    }

    /**
     * 解析base64格式的证书链
     * @param in
     * @author ssh
     * @return
     * @throws Exception
     */
    public static List<X509Certificate> resolveBase64CertChain(InputStream in) throws Exception {

        try {
            String p7b = SignedDataUtils.readBase64FromFile(in);
            CMSSignedData cmsSignedData = new CMSSignedData(Base64.decode(p7b));
            return SignedDataUtils.getX509CertificateListFromSignedData(cmsSignedData);
        } catch (Exception e) {
            throw new Exception("转换base64格式的证书链异常",e);
        }
    }

    /**
     * 解析DER格式的证书链
     * @param in
     * @author ssh
     * @return
     * @throws Exception
     */
    public static List<X509Certificate> resolveByteCertChain(byte[] in) throws Exception {

        try {
            CMSSignedData cmsSignedData = new CMSSignedData(in);
            return SignedDataUtils.getX509CertificateListFromSignedData(cmsSignedData);
        } catch (Exception e) {
            throw new Exception("解析DER格式的证书链",e);
        }
    }

    /**
     * 从cmsSigendData获取证书链
     * @param cmsSignedData
     * @return
     */
    public static List<X509Certificate> getX509CertificateListFromSignedData(CMSSignedData cmsSignedData) throws Exception {
        List<X509Certificate> certificateList = new ArrayList<X509Certificate>();
        try {
            Store<X509CertificateHolder> store = cmsSignedData.getCertificates();

            Collection collection = store.getMatches(null);
            Iterator<X509CertificateHolder> it = collection.iterator();
            while (it.hasNext()) {
                X509CertificateHolder x509CertificateHolder = it.next();
                X509Certificate x509Certificate = new JcaX509CertificateConverter().setProvider("BC").getCertificate(x509CertificateHolder);
                certificateList.add(x509Certificate);
            }
            return certificateList;
        } catch (Exception e) {
            throw new Exception("从cmsSignedData中获取证书链异常",e);
        }
    }


    /**
     * 从文件中读取base64字符串
     * @param in
     * @return
     */
    public static String readBase64FromFile(InputStream in) throws Exception {
        StringBuffer stringBuffer = new StringBuffer();
        try {
            BufferedReader br = null;
            String pemStr = "";
            br = new BufferedReader(new InputStreamReader(in));
            while((pemStr = br.readLine()) != null) {
                stringBuffer.append(pemStr);
            }
            return stringBuffer.toString();
        } catch (Exception e) {
            throw new Exception("从文件中解析base64字符串",e);
        } /*finally {
            if (null != in) {
                in.close();
            }
        }*/
    }
}
