package com.xdja.ra.helper;

import com.alibaba.fastjson.JSON;
import com.xdja.pki.gmssl.core.utils.GMSSLBCSignUtils;
import com.xdja.pki.gmssl.crypto.utils.GMSSLRSASignUtils;
import com.xdja.pki.gmssl.crypto.utils.GMSSLSM2SignUtils;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLSignatureAlgorithm;
import com.xdja.ra.asn1.*;
import com.xdja.ra.bean.*;
import com.xdja.ra.constant.SdkCommonVariable;
import com.xdja.ra.constant.SdkConstants;
import com.xdja.ra.error.ErrorEnum;
import com.xdja.ra.sdk.SDKService;
import com.xdja.ra.utils.DnUtil;
import com.xdja.ra.utils.GMSSLHttpReqUtils;
import com.xdja.ra.utils.SdkApacheClientUtils;
import com.xdja.ra.utils.SdkBCUtils;
import com.xdja.ra.utils.SdkCertUtils;
import com.xdja.ra.vo.FreeText;
import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.asn1.*;
import org.bouncycastle.asn1.cmp.*;
import org.bouncycastle.asn1.cms.*;
import org.bouncycastle.asn1.crmf.*;
import org.bouncycastle.asn1.nist.NISTNamedCurves;
import org.bouncycastle.asn1.sec.SECObjectIdentifiers;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.style.RFC4519Style;
import org.bouncycastle.asn1.x509.*;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.jce.X509KeyUsage;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.BigIntegers;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.security.*;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPrivateKey;
import java.security.spec.RSAPublicKeySpec;
import java.util.*;

public class PKIMessageHelper {

    private static Logger logger = LoggerFactory.getLogger(PKIMessageHelper.class);

    public static final String CERT_HEAD = "-----BEGIN CERTIFICATE-----";
    public static final String CERT_TAIL = "-----END CERTIFICATE-----";

    /**
     * 检查CMP响应  检查消息头
     *
     * @return
     */
    public static Result checkCmpHeaderAndSign(byte[] pkiMessage, String transId, byte[] senderNonce) {
        PKIMessage respPkiMessage = PKIMessage.getInstance(pkiMessage);
        if (respPkiMessage == null) {
            logger.info(" =================== No pkiMessage response message.");
            return Result.failure(ErrorEnum.NO_PKI_MESSAGE_RESP_MESSAGE);
        }

        PKIHeader header = respPkiMessage.getHeader();
        if (header == null) {
            logger.info(" =================== No header in response message.");
            return Result.failure(ErrorEnum.NO_HEADER_IN_RESPONSE_MESSAGE);
        }
        // 检测发送者的nonce的长度
        if (header.getRecipNonce().getOctets().length != 16) {
            logger.info("检查消息头和签名 ====== Wrong length of received recip nonce (made up by server). Is " + header.getRecipNonce().getOctets().length + " byte but should be 16.");
            return Result.failure(ErrorEnum.WRONG_LEN_OF_RECEIVED_RECIP_NONCE);
        }

        //第三方唯一标识
        String recipient = header.getRecipient().getName().toString();
        logger.info("第三方唯一标识normal ========== " + recipient);

        // 检测RA返回的transId是否和请求的一致
        String s = new String(header.getTransactionID().getOctets());
        if (!transId.equalsIgnoreCase(s)) {
            logger.info(" =================== transid is not the same as the one we sent");
            return Result.failure(ErrorEnum.TRANS_ID_IS_NOT_THE_SAME_AS_WE_SENT);
        }

        // 判断RA返回的nonce的长度
        if (header.getSenderNonce().getOctets().length != 16) {
            logger.info(" =================== Wrong length of received sender nonce (made up by server). Is " + header.getSenderNonce().getOctets().length + " byte but should be 16.");
            return Result.failure(ErrorEnum.WRONG_LEN_OF_RECEIVED_SENDER_NONCE);
        }
        // 检测senderNonce一致性 十六字节数组
        if (!Arrays.equals(header.getSenderNonce().getOctets(), senderNonce)) {
            logger.info(" =================== recipient nonce not the same as we sent away as the sender nonce. Sent: " + Arrays.toString(senderNonce) + " Received: " + Arrays.toString(header.getRecipNonce().getOctets()));
            return Result.failure(ErrorEnum.RECIPIENT_NONCE_NOT_THE_SAME_AS_WE_SENT);
        }

        // 获取消息体保护算法
        AlgorithmIdentifier algId = header.getProtectionAlg();
        String protectionAlgId = algId.getAlgorithm().getId();
        if (algId == null || algId.getAlgorithm() == null || algId.getAlgorithm().getId() == null) {
            logger.info("检查消息头和签名 ======  Not possible to get algorithm.");
            return Result.failure(ErrorEnum.NO_PROTECTION_ALG_IN_PKI_HEADER);
        }

        // 验证CertRepMessage中的签名值
        logger.debug("检查消息头和签名 ====== 校验签名值");
        // 获取验证证书  RA的服务器证书的签名证书
        CMPCertificate[] extraCerts = respPkiMessage.getExtraCerts();
        X509Certificate certificate = null;
        try {
            certificate = PKIMessageHelper.convertDerCertToCert(extraCerts[0].getEncoded());
        } catch (IOException e) {
            logger.error("验证证书转换异常", e);
            return Result.failure(ErrorEnum.VERIFY_CERT_CONVERSION_ERROR);
        }
        // 验证消息保护
        boolean verifyResult = false;
        // 验证PKIMessage消息体签名值
        try {
            verifyResult = SdkBCUtils.verifyCertByBC(protectionAlgId, certificate.getPublicKey(), PKIMessageHelper.getProtectedBytes(header, respPkiMessage.getBody()), respPkiMessage.getProtection().getBytes());
            logger.info("验签结果 ====== " + verifyResult);
        } catch (Exception e) {
            logger.error("验证resp消息异常", e);
            return Result.failure(ErrorEnum.VERIFY_RESP_MESSAGE_ERROR);
        }

        if (!verifyResult) {
            logger.info("检查消息头和签名 ====== verify_pki_message_protection_error");
            return Result.failure(ErrorEnum.VERIFY_PKI_MESSAGE_PROTECTION_ERROR);
        }

        return Result.success(null);
    }

    /**
     * 封装并发送ErrorMsg消息
     *
     * @param errMsg
     * @param errCode
     * @param recipNonce
     * @param senderNonce
     * @param transId
     * @param url
     * @return
     */
    public static Result genErrorPKIMsg(String normalName, PrivateKey normalPrivateKey, X509Certificate[] trustCaCert, X509Certificate normalCert, String errMsg, int errCode, byte[] recipNonce, byte[] senderNonce, String transId, String url, ASN1ObjectIdentifier protectAlgName) {

        Result sdkResult = new Result();
        // 封装ErrorMsgContent返回给RA
        logger.info("发送错误消息 ======== 1.封装ErrorMsgContent结构体");
        ErrorMsgContent errorMsgContent;
        try {
            errorMsgContent = PKIMessageHelper.genErrorMsgContent(PKIStatus.rejection, errMsg, errCode);
        } catch (Exception e) {
            logger.error("=============== 封装ErrorMsgContent异常{}", e);
            return Result.failure(ErrorEnum.ERROR_MSG_CONTENT_EXCEPTION);
        }
        FreeText freeText = new FreeText();
        freeText.setSignSn(SDKService.config.getUserCertSn().toLowerCase());
        logger.debug("发送错误消息 ======== 2.封装PKIMesage结构体");
        PKIMessage errorPKIMessage = null;
        try {
            errorPKIMessage = PKIMessageHelper.genPKIMessage(normalPrivateKey, normalName, PKIBody.TYPE_ERROR, recipNonce, senderNonce, transId, errorMsgContent, protectAlgName, JSON.toJSONString(freeText), normalCert);
        } catch (Exception e) {
            logger.error("=============== 封装ErrorMsg的PKIMessage异常{}", e);
            return Result.failure(ErrorEnum.MAKE_PKI_MESSAGE_EXCEPTION);
        }

        logger.debug("发送错误消息 ======== 3.发送证书错误消息");
        try {
            Result postResult = SdkApacheClientUtils.sendApacheClientRequest(errorPKIMessage.getEncoded(),null,url,"application/pkixcmp", SDKService.config,null, "post");
            if (!postResult.isSuccess()) {
                sdkResult.setErrorMsg(postResult.getErrorMsg());
                return sdkResult;
            }
        } catch (Exception e) {
            logger.error(" ============= 发送证书错误Http请求异常:{}", e);
            return Result.failure(ErrorEnum.SEND_HTTP_MESSAGE_EXCEPTION);
        }

        logger.debug("发送错误消息 ========正常结束");
        return sdkResult;
    }

    /**
     * 获取RA返回消息 解析各种响应体
     *
     * @param pkiInfo
     * @param transId
     * @return
     */
    public static Result resolveVarietyRepMessage(byte[] pkiInfo, String transId,int keyFormat) {
        Result sdkResult = new Result();
        PKIMessage respPkiMessage = PKIMessage.getInstance(pkiInfo);
        PKIBody body = respPkiMessage.getBody();
        logger.debug("CMP返回body体的tagNo:" + body.getType());

        // 处理1，3-CertRepMessage结构体
        if (body.getType() == PKIBody.TYPE_INIT_REP || body.getType() == PKIBody.TYPE_CERT_REP) {
            Result certRepSdkResult = null;
            try {
                certRepSdkResult = PKIMessageHelper.resolveCertRepMessage(body, transId);
            } catch (Exception e) {
                logger.error("解析CertRepMessage异常{}", e);
                return Result.failure(ErrorEnum.RESOLVE_CERT_REP_MESSAGE_EXCEPTION);
            }
            if (!certRepSdkResult.isSuccess()) {
                sdkResult.setErrorMsg(certRepSdkResult.getErrorMsg());
                return sdkResult;
            }
            sdkResult.setInfo(certRepSdkResult.getInfo());
            return sdkResult;

        } else if (body.getType() == PKIBody.TYPE_KEY_RECOVERY_REP) {
            // 处理 10-KeyRecRepContent结构体
            Result recRepSdkResult = null;
            try {
                recRepSdkResult = PKIMessageHelper.resolveKeyRecRepContent(body);
            } catch (Exception e) {
                logger.error("解析KeyRecRepContent异常", e);
                return Result.failure(ErrorEnum.RESOLVE_REV_REP_CONTENT_EXCEPTION);
            }
            sdkResult.setInfo(recRepSdkResult.getInfo());
        }
        // 处理12-RevRepContent结构体
        else if (body.getType() == PKIBody.TYPE_REVOCATION_REP) {
            Result revRepSdkResult = null;
            try {
                revRepSdkResult = PKIMessageHelper.resolveRevRepContent(body);
            } catch (Exception e) {
                logger.error("解析RevRepContent异常{}", e);
                return Result.failure(ErrorEnum.RESOLVE_REV_REP_CONTENT_EXCEPTION);
            }
            sdkResult.setInfo(revRepSdkResult.getInfo());
        } else {
            logger.debug("Cert body tag is:" + body.getType());
            return Result.failure(ErrorEnum.RA_NOT_SUPPORT_THIS_CERT_BODY_TAG);
        }

        return sdkResult;
    }

    /**
     * 解析恢复响应结构体
     *
     * @param body PKIBody
     * @return
     */
    private static Result resolveKeyRecRepContent(PKIBody body) {
        Result sdkResult = new Result();
        KeyRecRepContent keyRecRepContent = (KeyRecRepContent) body.getContent();
        PKIStatusInfo pkiStatusInfo = keyRecRepContent.getStatus();
        int status = PKIStatus.GRANTED;
        UserCertRep userCertInfo = new UserCertRep();
        if (status == pkiStatusInfo.getStatus().intValue()) {
            logger.info("==========RA返回的恢复成功=========");
            logger.trace("==========提取签名证书=========");
            CMPCertificate newSigCert = keyRecRepContent.getNewSigCert();
            Result checkResult = PKIMessageHelper.checkCMPCert(newSigCert);
            if (!checkResult.isSuccess()) {
                sdkResult.setErrorMsg(checkResult.getErrorMsg());
                return sdkResult;
            }
            X509Certificate signCert = (X509Certificate) checkResult.getInfo();
            String base64SignCert = SdkCertUtils.certToFullB64(signCert);
            userCertInfo.setSignCert(base64SignCert);
            logger.trace("==========提取加密证书=========");
            CertifiedKeyPair[] keyPairHist = keyRecRepContent.getKeyPairHist();
            CertOrEncCert oldEncCert = keyPairHist[0].getCertOrEncCert();
            if (oldEncCert == null) {
                logger.info("No CertOrEncCert for certificate received.");
                return Result.failure(ErrorEnum.NO_CERT_OR_ENC_CERT_FOR_RECEIVED);
            }
            CMPCertificate cmpCertificate = oldEncCert.getCertificate();
            checkResult = PKIMessageHelper.checkCMPCert(cmpCertificate);
            if (!checkResult.isSuccess()) {
                sdkResult.setErrorMsg(checkResult.getErrorMsg());
                return sdkResult;
            }
            // 获取明文加密证书
            X509Certificate encCert = (X509Certificate) checkResult.getInfo();
            String base64EncCert = SdkCertUtils.certToFullB64(encCert);
            userCertInfo.setEncCert(base64EncCert);
            EncryptedValue encPprKey = keyPairHist[0].getPrivateKey();
            // 获取CertifiedKeyPair结构体
            if (encPprKey == null) {
                logger.info("No encPprKey for certificate received.");
                return Result.failure(ErrorEnum.NO_CERTIFIED_KEY_PAIR_FOR_RECEIVED);
            }
            /*// 获取私钥信封
            try {
                byte[] encoded = encPprKey.getEncoded();
                userCertInfo.setEncPriKey(Base64.toBase64String(encoded));
            } catch (IOException e) {
                logger.info("No encPprKey for certificate received.");
                return Result.failure(ErrorEnum.NO_CERTIFIED_KEY_PAIR_FOR_RECEIVED);
            }*/
            if (encPprKey != null) {
                SignedAndEnvelopedData signedAndEnvelopedData = null;
                try {
                    signedAndEnvelopedData = buildSignedAndEnvelopedData(signCert, encPprKey, SDKService.config.getUserCertSn());
                } catch (Exception e) {
                    logger.error("Build SignedAndEnvelopedData From Encryptedvalue Exception.", e);
                    return Result.failure(ErrorEnum.NO_CERT_OR_ENC_CERT_FOR_RECEIVED);
                }
                userCertInfo.setEncPriKey(Base64.toBase64String(signedAndEnvelopedData.getDEREncoded()));
            }
            sdkResult.setInfo(userCertInfo);
        } else {
            PKIFreeText pkiFreeTexts = pkiStatusInfo.getStatusString();
            String failInfo = pkiFreeTexts.getStringAt(0).toString();
            logger.debug("RA返回的恢复失败,原因:{}", failInfo);
            sdkResult.setInfo("恢复失败");
        }
        return sdkResult;
    }

    /**
     * 解析撤销响应结构体
     *
     * @param body
     * @return
     */
    public static Result resolveRevRepContent(PKIBody body) {
        Result sdkResult = new Result();
        RevRepContent revRepContent = (RevRepContent) body.getContent();
        PKIStatusInfo[] pkiStatusInfos = revRepContent.getStatus();
        int status = PKIStatus.GRANTED;
        if (status == pkiStatusInfos[0].getStatus().intValue()) {
            logger.debug("==========RA返回的撤销成功=========");
            sdkResult.setInfo("撤销成功");
        } else {
            PKIFreeText pkiFreeTexts = pkiStatusInfos[0].getStatusString();
            String failInfo = pkiFreeTexts.getStringAt(0).toString();
            logger.debug("RA返回的撤销失败,原因:" + failInfo);
            sdkResult.setInfo("撤销失败");
        }
        return sdkResult;
    }

    /**
     * 校验RA返回的证书内容
     *
     * @param cmpCertificate
     * @return
     */
    public static Result checkCMPCert(CMPCertificate cmpCertificate) {
        if (cmpCertificate == null) {
            logger.debug("No X509CertificateStructure for certificate received.");
            return Result.failure(ErrorEnum.NO_X509_CERT_FOR_RECEIVED);
        }

        // 获取返回中的证书结构体
        byte encoded[] = new byte[0];
        try {
            encoded = cmpCertificate.getEncoded();
            if (encoded == null || encoded.length <= 0) {
                logger.debug("No encoded certificate received");
                return Result.failure(ErrorEnum.NO_ENCODE_CERT_FOR_RECEIVED);
            }
        } catch (IOException e) {
            logger.debug("CMPCertificate Encode Exception.{}", e);
            return Result.failure(ErrorEnum.CMP_CERT_ENCODE_EXCEPTION);
        }

        X509Certificate cert = null;
        try {
            cert = SdkCertUtils.convertDerCertToCert(encoded);
        } catch (Exception e) {
            logger.error("Not possible to create certificate.{}", e);
            return Result.failure(ErrorEnum.NOT_POSSIBLE_TO_CREATE_CERT);
        }

        return Result.success(cert);
    }


    /**
     * 解析CertRepMessage结构体
     *
     * @param body
     * @return
     */
    public static Result resolveCertRepMessage(PKIBody body, String transId) throws IOException {
        Result sdkResult = new Result();
        // 获取certRepMessage结构体
        CertRepMessage certRepMessage = (CertRepMessage) body.getContent();
        if (certRepMessage == null) {
            logger.debug("============== No CertRepMessage for certificate received.");
            return Result.failure(ErrorEnum.NO_CERT_REQ_MESSAGE_RECEIVED);
        }
        // 获取证书链
        CertResponse[] certResponses = certRepMessage.getResponse();
        UserCertRep userCertInfo = new UserCertRep();
        X509Certificate signCert = null;
        for (CertResponse resp : certResponses) {
            // 依次获取证书返回CertResponse
            if (resp == null) {
                logger.debug("============== No CertResponse for certificate received.");
                return Result.failure(ErrorEnum.NO_CERT_RESPONSE_MESSAGE_RECEIVED);
            }

            // 获取返回对象中的PKIStatusInfo结构体
            final PKIStatusInfo info = resp.getStatus();
            if (info == null) {
                logger.debug("No PKIStatusInfo for certificate received.");
                return Result.failure(ErrorEnum.NO_PKI_STATUS_INFO_FOR_RECEIVE);
            }

            // 获取返回的PkiStatus内容
            int pkiStatus = info.getStatus().intValue();
            if (pkiStatus != 0) {
                logger.debug("Received Status is " + pkiStatus + " but should be 0  ");
                //错误原因
                DERUTF8String stringAt = info.getStatusString().getStringAt(0);
                //错误码
                int failInfo = info.getFailInfo().intValue();
                if (SdkConstants.ERROR_CODE_300.equals(String.valueOf(failInfo).substring(0, 3))) {
                    return Result.failure(ErrorEnum.RA_OPEN_API_INNER_EXCEPTION);
                } else if (getHashSet().contains(failInfo)) {
                    return Result.failure(ErrorEnum.RA_SERVICE_INNER_EXCEPTION);
                } else if (failInfo >= 20502 && failInfo <= 20506) {
                    return Result.failure(ErrorEnum.CA_RETURN_USER_CERT_INFO_ERROR);
                } else {
                    return Result.failure(failInfo, stringAt.toString());
                }
            }

            // 获取CertifiedKeyPair结构体
            final CertifiedKeyPair certifiedKeyPair = resp.getCertifiedKeyPair();
            if (certifiedKeyPair == null) {
                logger.debug("No CertifiedKeyPair for certificate received.");
                return Result.failure(ErrorEnum.NO_CERTIFIED_KEY_PAIR_FOR_RECEIVED);
            }

            // 获取CertOrEncCert结构体
            final CertOrEncCert certOrEncCert = certifiedKeyPair.getCertOrEncCert();
            if (certOrEncCert == null) {
                logger.debug("No CertOrEncCert for certificate received.");
                return Result.failure(ErrorEnum.NO_CERT_OR_ENC_CERT_FOR_RECEIVED);
            }
            CMPCertificate cmpCertificate = certOrEncCert.getCertificate();
            // 说明该CertificateKeyPair中包含了明文加密证书和加密私钥信封

            // 与请求体中的requestId对应匹配
            Map<String, Object> headerMap = SdkCommonVariable.getHeaderMap();
            BaseCMPInfo baseCMPInfo = (BaseCMPInfo) headerMap.get(transId);
            if (baseCMPInfo == null) {
                logger.debug(" ============= No ra send transId.");
                return Result.failure(ErrorEnum.NO_RA_SEND_TRANS_ID);
            }
            boolean isSignCert = false;
            long requestId = baseCMPInfo.getRequestId();
            long receiveCertReqId = resp.getCertReqId().getValue().longValue();
            if (receiveCertReqId == requestId) {
                isSignCert = true;
            } else if (receiveCertReqId == -1) {
                isSignCert = false;
            } else {
                String errMsg = "=============== Received CertReqId is " + receiveCertReqId + " but should be " + requestId;
                logger.debug(errMsg);
                return Result.failure(ErrorEnum.RA_RECEIVED_CERT_REQ_ID_IS_ERROR);
            }

            Result checkResult = PKIMessageHelper.checkCMPCert(cmpCertificate);
            if (!checkResult.isSuccess()) {
                sdkResult.setErrorMsg(checkResult.getErrorMsg());
                return sdkResult;
            }
            if (isSignCert) {
                // 获取签名证书
                signCert = (X509Certificate) checkResult.getInfo();
                String base64SignCert = SdkCertUtils.certToFullB64(signCert);
                userCertInfo.setSignCert(base64SignCert);
            } else {
                // 获取明文加密证书
                X509Certificate encCert = (X509Certificate) checkResult.getInfo();
                String base64EncCert = SdkCertUtils.certToFullB64(encCert);
                userCertInfo.setEncCert(base64EncCert);
            }

            EncryptedValue privateKey = certifiedKeyPair.getPrivateKey();
            // 获取私钥信封
            if (privateKey != null) {
                ASN1OctetString priKey = ASN1OctetString.getInstance(ASN1Sequence.getInstance(resp.getEncoded()).getObjectAt(3));
                userCertInfo.setEncPriKey(new String(priKey.getOctets()));
            }
//            // 获取私钥信封  不在进行客户端的私钥格式转换
//            if (privateKey != null) {
////                    ASN1OctetString priKey = ASN1OctetString.getInstance(ASN1Sequence.getInstance(resp.getEncoded()).getObjectAt(3));
////                    userCertInfo.setEncPriKey(new String(priKey.getOctets()));
//                if (signCert == null) {
//                    CertResponse signCertResp = certResponses[1];
//                    try {
//                        signCert = (X509Certificate) SdkCertUtils.convertDerCertToCert(signCertResp.getCertifiedKeyPair().getCertOrEncCert().getCertificate().getEncoded());
//                    } catch (Exception e) {
//                        logger.error("Not possible to create certificate.{}", e);
//                        sdkResult.failure(ErrorEnum.NOT_POSSIBLE_TO_CREATE_CERT);
//                        return sdkResult;
//                    }
//                }
//                SignedAndEnvelopedData signedAndEnvelopedData = null;
//                try {
//                    signedAndEnvelopedData = PKIMessageHelper.buildSignedAndEnvelopedData(signCert, privateKey, SDKService.config.getUserCertSn());
//                } catch (Exception e) {
//                    logger.error("Build SignedAndEnvelopedData From Encryptedvalue Exception.", e);
//                    return sdkResult.failure(ErrorEnum.BUILD_SIGNEDANDENVLOPEDDATA_FROM_ENCRYPTEDVALUE_EXCPTION);
//                }
//                userCertInfo.setEncPriKey(Base64.toBase64String(signedAndEnvelopedData.getDEREncoded()));
//            }
        }

        return Result.success(userCertInfo);
    }

    private static HashSet getHashSet() {
        HashSet hashSet = new HashSet();
        hashSet.add(20339);
        hashSet.add(20340);
        hashSet.add(20317);
        hashSet.add(20403);
        hashSet.add(20334);
        hashSet.add(20328);

        return hashSet;
    }

    public static void main(String[] args) {
        //错误码
        int failInfo = 100001;
        if (SdkConstants.ERROR_CODE_300.equals(String.valueOf(failInfo).substring(0, 3))) {
            System.out.println(1111);
        } else if (getHashSet().contains(failInfo)) {
            System.out.println(2222);
        } else if (failInfo >= 20502 && failInfo <= 20506) {
            System.out.println(3333);
        } else {
            System.out.println(4444);
        }
    }

    /**
     * 生成证书请求对象CertRequest
     *
     * @param certReqId
     * @param certType
     * @return
     * @throws IOException
     */
    public static CertRequest genCertRequest(byte[] publicKey, ASN1ObjectIdentifier signAlg, long certReqId, int certType) throws Exception {

        // 封装证书模板
        final CertTemplateBuilder myCertTemplate = new CertTemplateBuilder();
        myCertTemplate.setVersion(1);
        myCertTemplate.setSigningAlg(new AlgorithmIdentifier(signAlg));
        SubjectPublicKeyInfo keyInfo = null;
        if (publicKey != null) {
            if (SM2ObjectIdentifiers.sm2SignWithSm3.equals(signAlg)) {
                if (certType == SdkConstants.CERT_TYPE_SIGN_2) {
                    keyInfo = SubjectPublicKeyInfo.getInstance(publicKey);
                } else {
                    PublicKey pubKey = SdkCertUtils.convertSM2PublicKey(Base64.toBase64String(publicKey));
                    keyInfo = SubjectPublicKeyInfo.getInstance(pubKey.getEncoded());
                }
            } else if (NISTObjectIdentifiers.nistSignAlgorithm.equals(signAlg)) {
                if (certType == SdkConstants.CERT_TYPE_SIGN_2) {
                    try {
                        keyInfo = SubjectPublicKeyInfo.getInstance(publicKey);
                    } catch (Exception e) {
                        PublicKey pubKey = SdkCertUtils.convertECPublicKey(Base64.toBase64String(publicKey), NISTNamedCurves.getName(SECObjectIdentifiers.secp256r1));
                        keyInfo = SubjectPublicKeyInfo.getInstance(pubKey.getEncoded());
                    }
                }

            } else {
                if (certType == SdkConstants.CERT_TYPE_SIGN_2) {
                    keyInfo = SubjectPublicKeyInfo.getInstance(publicKey);
                } else {
                    KeyFactory kf = KeyFactory.getInstance("RSA", new BouncyCastleProvider());
                    RSAPublicKeySpec rsaPublicKeySpec = new RSAPublicKeySpec(BigIntegers.fromUnsignedByteArray(publicKey), BigInteger.valueOf(65537L));
                    PublicKey pubKey = kf.generatePublic(rsaPublicKeySpec);
                    keyInfo = SubjectPublicKeyInfo.getInstance(pubKey.getEncoded());
                }
            }
        }

        myCertTemplate.setPublicKey(keyInfo);
        // 信息都由CA侧根据申请信息进行补充
        ExtensionsGenerator extgen = new ExtensionsGenerator();
        try {
            int bcku = 0;
            if (certType == SdkConstants.CERT_TYPE_SIGN_2) {
                // KeyUsage
                bcku = KeyUsage.digitalSignature | KeyUsage.nonRepudiation;
            } else if (certType == SdkConstants.CERT_TYPE_ENC_3) {
                bcku = KeyUsage.keyAgreement | KeyUsage.keyEncipherment | KeyUsage.dataEncipherment;
            }
            final X509KeyUsage ku = new X509KeyUsage(bcku);
            extgen.addExtension(Extension.keyUsage, false, ku);
        } catch (IOException e) {
            logger.info("封装CertRequest的扩展信息异常{}", e);
            throw new IOException();
        }
        myCertTemplate.setExtensions(extgen.generate());
        return new CertRequest(new ASN1Integer(certReqId), myCertTemplate.build(), null);
    }


    /**
     * 生成CertReqMsg
     *
     * @param certRequests
     * @return
     * @throws Exception
     */
    public static CertReqMessages genCertReqMessages(CertRequest[] certRequests) {
        CertReqMsg[] certReqMsgs = new CertReqMsg[certRequests.length];
        for (int i = 0; i < certRequests.length; i++) {
            if (certRequests[i] != null) {
                CertReqMsg certReqMsg = new CertReqMsg(certRequests[i], null, null);
                certReqMsgs[i] = certReqMsg;
            }
        }
        CertReqMessages myCertReqMessages = new CertReqMessages(certReqMsgs);
        return myCertReqMessages;
    }

    /**
     * 封装PKIMessage (1)
     *
     * @param normalPrivateKey
     * @param sender
     * @param tagNo
     * @param recipNonce
     * @param senderNonce
     * @param asn1Encodable    certReqMessages ErrorMsgContent
     * @return
     * @throws Exception
     */
    public static PKIMessage genPKIMessage(PrivateKey normalPrivateKey, String sender, int tagNo, byte[] recipNonce, byte[] senderNonce, String transId, ASN1Encodable asn1Encodable, ASN1ObjectIdentifier protectionAlg, String freeText, X509Certificate normalCert) throws Exception {
        PKIBody myPKIBody = new PKIBody(tagNo, asn1Encodable);        // initialization request
        GeneralName normalName = new GeneralName(GeneralName.uniformResourceIdentifier, sender);
        GeneralName raName = new GeneralName(GeneralName.uniformResourceIdentifier, "RA");
        final PKIHeaderBuilder myPKIHeader = new PKIHeaderBuilder(PKIHeader.CMP_1999, normalName, raName);
        myPKIHeader.setMessageTime(new ASN1GeneralizedTime(new Date()));
        myPKIHeader.setSenderNonce(new DEROctetString(senderNonce));
        myPKIHeader.setRecipNonce(new DEROctetString(recipNonce));
        myPKIHeader.setTransactionID(transId.getBytes());
        myPKIHeader.setProtectionAlg(new AlgorithmIdentifier(protectionAlg));
        if (freeText != null) {
            // 用于接收者可读的消息
            myPKIHeader.setFreeText(new PKIFreeText(freeText));
        }
        PKIHeader myHeader = myPKIHeader.build();
        if (normalPrivateKey == null) {
            throw new Exception("第三方签名私钥为空");
        }
        // 使用第三方私钥文件的签名私钥对pkiMessage签名
        PKIMessage pkiMessage;
        String singValue;
        logger.info("被签名计算的原始数据：" + Base64.toBase64String(PKIMessageHelper.getProtectedBytes(myHeader, myPKIBody)));
        if (protectionAlg.getId().equalsIgnoreCase(SM2ObjectIdentifiers.sm2SignWithSm3.getId())) {
            long begin = System.currentTimeMillis();
            singValue = GMSSLSM2SignUtils.signByBC(normalPrivateKey, Base64.toBase64String(PKIMessageHelper.getProtectedBytes(myHeader, myPKIBody)));
            System.out.println(System.currentTimeMillis() - begin);
        } else if (protectionAlg.getId().equalsIgnoreCase(RsaObjectIdentifiers.sha256WithRSA.getId())) {
            singValue = GMSSLRSASignUtils.signByBC(GMSSLSignatureAlgorithm.SHA256_WITH_RSA.getSigAlgName(), normalPrivateKey, Base64.toBase64String(PKIMessageHelper.getProtectedBytes(myHeader, myPKIBody)));
        } else if (protectionAlg.getId().equalsIgnoreCase(RsaObjectIdentifiers.sha1WithRSA.getId())) {
            singValue = GMSSLRSASignUtils.signByBC(GMSSLSignatureAlgorithm.SHA1_WITH_RSA.getSigAlgName(), normalPrivateKey, Base64.toBase64String(PKIMessageHelper.getProtectedBytes(myHeader, myPKIBody)));
        } else {
            singValue = Base64.toBase64String(GMSSLBCSignUtils.generateSignature(GMSSLSignatureAlgorithm.SHA256_WITH_ECDSA.getSigAlgName(), normalPrivateKey, PKIMessageHelper.getProtectedBytes(myHeader, myPKIBody)));
        }
        if (StringUtils.isBlank(singValue)) {
            throw new Exception("使用BC签名失败");
        }
        logger.info("签名值：" + singValue);

        //验证证书 放第三方系统读取的证书  放不放都可以
        CMPCertificate[] cmpCert = PKIMessageHelper.getCMPCert(normalCert);
        pkiMessage = new PKIMessage(myHeader, myPKIBody, new DERBitString(Base64.decode(singValue)), cmpCert);
        return pkiMessage;
    }

    /**
     * 获取header和body组合的字节数组
     *
     * @param header
     * @param body
     * @return
     */
    public static byte[] getProtectedBytes(PKIHeader header, PKIBody body) {
        byte[] res = null;
        ASN1EncodableVector v = new ASN1EncodableVector();
        v.add(header);
        v.add(body);
        ASN1Encodable protectedPart = new DERSequence(v);
        try {
            ByteArrayOutputStream bao = new ByteArrayOutputStream();
            DEROutputStream out = new DEROutputStream(bao);
            out.writeObject(protectedPart);
            res = bao.toByteArray();
        } catch (Exception ex) {
            logger.error(ex.getLocalizedMessage(), ex);
        }
        return res;
    }


    /**
     * 生成证书确认PKIMessage
     *
     * @param hash
     * @return
     */
    public static CertConfirmContent genCertConfirmContent(String hash, long certReqId) {
        CertStatus certStatus = new CertStatus(hash.getBytes(), new BigInteger(String.valueOf(certReqId)));
        ASN1EncodableVector v = new ASN1EncodableVector();
        v.add(certStatus);
        CertConfirmContent certConfirmContent = CertConfirmContent.getInstance(new DERSequence(v));
        return certConfirmContent;
    }

    /**
     * 将x509证书转换为CMP证书
     *
     * @param cert
     * @return
     * @throws CertificateEncodingException
     * @throws IOException
     */
    private static CMPCertificate[] getCMPCert(Certificate cert) throws CertificateEncodingException, IOException {
        ASN1InputStream ins = new ASN1InputStream(cert.getEncoded());
        try {
            ASN1Primitive pcert = ins.readObject();
            org.bouncycastle.asn1.x509.Certificate c = org.bouncycastle.asn1.x509.Certificate.getInstance(pcert.toASN1Primitive());
            CMPCertificate[] res = {new CMPCertificate(c)};
            return res;
        } finally {
            ins.close();
        }
    }

    /**
     * 生成ErrorMsgContent
     *
     * @param pkiStatus
     * @param errorDetails
     * @param errorCode
     * @return
     */
    public static ErrorMsgContent genErrorMsgContent(PKIStatus pkiStatus, String errorDetails, int errorCode) {
        PKIFreeText pkiFreeText = new PKIFreeText(errorDetails);
        PKIStatusInfo pkiStatusInfo = new PKIStatusInfo(pkiStatus);
        ErrorMsgContent errorMessage = new ErrorMsgContent(pkiStatusInfo, new ASN1Integer(errorCode), pkiFreeText);
        return errorMessage;
    }

    /**
     * 通过密码机生成随机数
     *
     * @return
     */
    public static byte[] genRandomByHsm(int length) throws SdfSDKException {
        //  String randomByHsm = GMSSLRandomUtils.generateRandomByYunhsm(length);
        byte[] normalSenderNonce = new byte[16];
        SecureRandom secureRandom = new SecureRandom();
        secureRandom.nextBytes(normalSenderNonce);
        //  byte[] random = Base64.decode(randomByHsm);
        return normalSenderNonce;
    }

    /**
     * DER编码的证书成为证书对象
     *
     * @param certder
     * @return
     */
    public static X509Certificate convertDerCertToCert(byte[] certder) {
        // 解析二进制证书流
        CertificateFactory cf = null;
        X509Certificate x509Cert = null;
        try {
            cf = CertificateFactory.getInstance("X.509", new BouncyCastleProvider());
            InputStream inStream = new ByteArrayInputStream(certder);
            x509Cert = (X509Certificate) cf.generateCertificate(inStream);
        } catch (Exception e) {
            e.printStackTrace();
        }
        if (null != x509Cert) {
            return x509Cert;
        }

        String str = new String(certder);
        str = str.replace(CERT_HEAD, "").replace(CERT_TAIL, "");
        str = str.replace("\r", "").replace("\n", "");
        str = str.replace("\\r", "").replace("\\n", "");

        // 解析base64编码的证书流
        x509Cert = getCertFromB64(str);
        if (x509Cert == null) {
            // 解析16进制编码的证书流
            x509Cert = getCertFromStr16(str);
        }
        return x509Cert;
    }

    /**
     * 从Base64进制字符串获取证书对象
     *
     * @param b64
     * @return
     */
    public synchronized static X509Certificate getCertFromB64(String b64) {
        CertificateFactory cf;
        X509Certificate x509Cert;
        try {
            cf = CertificateFactory.getInstance("X.509", new BouncyCastleProvider());
            byte[] bsCert = Base64.decode(b64);
            InputStream inStream = new ByteArrayInputStream(bsCert);
            x509Cert = (X509Certificate) cf.generateCertificate(inStream);
            return x509Cert;
        } catch (Exception e) {
            System.err.println("getCertFromB64 error: " + e.toString());
        }
        return null;
    }


    /**
     * 从16进制字符串获取证书对象
     *
     * @param str
     * @return
     */
    public static X509Certificate getCertFromStr16(String str) {
        byte[] bs = hex2byte(str);
        CertificateFactory cf;
        X509Certificate x509Cert;
        try {
            cf = CertificateFactory.getInstance("X.509", new BouncyCastleProvider());
            InputStream inStream = new ByteArrayInputStream(bs);
            x509Cert = (X509Certificate) cf.generateCertificate(inStream);
            return x509Cert;
        } catch (Exception e) {
            System.err.println("getCertFromFullStr error: " + e.toString());
        }
        return null;
    }

    /**
     * 16进制字符串转换为二进制
     *
     * @param str String 类型参数
     * @return
     */
    public static byte[] hex2byte(String str) {
        if (null == str || str.equals("")) {
            return null;
        }
        str = str.trim();
        StringBuffer sb = new StringBuffer(str);
        int len = sb.length();
        if (len == 0 || len % 2 == 1) {
            return null;
        }
        byte[] b = new byte[len / 2];
        try {
            for (int i = 0; i < len; i += 2) {
                b[i / 2] = (byte) Integer.decode("0x" + sb.substring(i, i + 2)).intValue();
            }
            return b;
        } catch (Exception e) {
            return null;
        }
    }

    private static SignedAndEnvelopedData buildSignedAndEnvelopedData(X509Certificate signCert, EncryptedValue encryptedValue, String raSignSn) throws Exception {
        // 创建SignedAndEnvelopedData一级元素
        String raSignCertDN = signCert.getSubjectX500Principal().getName();
        String caAlg = signCert.getSigAlgName();
        ASN1Integer version = new ASN1Integer(1);
        ASN1Set digestAlgorithms = null;
        if ("SHA-1WithRSA".equalsIgnoreCase(caAlg) || "SHA1WithRSA".equalsIgnoreCase(caAlg) || "SHA256WithRSA".equalsIgnoreCase(caAlg)) {
            digestAlgorithms = new DERSet(new AlgorithmIdentifier(DigestObjectIdentifiers.sha1));
        } else {
            digestAlgorithms = new DERSet(new AlgorithmIdentifier(DigestObjectIdentifiers.sm3));
        }

        // 组装
        RecipientIdentifier rid = new RecipientIdentifier(new IssuerAndSerialNumber(DnUtil.getRFC4519X500Name(raSignCertDN), new BigInteger(raSignSn, 16)));
        KeyTransRecipientInfo keyTransRecipientInfo = new KeyTransRecipientInfo(rid, encryptedValue.getKeyAlg(), new DEROctetString(encryptedValue.getEncSymmKey().getOctets()));
        ASN1Set recipientInfos = new DERSet(keyTransRecipientInfo);
        //SymmetryObjectIdentifiers.sm4表示SM4的OID，这里应该是ContentType，oid应该为1.2.156.10197.6.1.4.2.1
        EncryptedContentInfo encryptedContentInfo = new EncryptedContentInfo(SymmetryObjectIdentifiers.contentType, encryptedValue.getSymmAlg(), new DEROctetString(encryptedValue.getEncValue()));

        byte[] mergeData = SdkCertUtils.byteMergerAll(version.getEncoded(), recipientInfos.getEncoded(), digestAlgorithms.getEncoded(), encryptedContentInfo.getEncoded());
        ASN1Set signerInfos = makeSignerInfos(raSignCertDN, new BigInteger(raSignSn, 16), mergeData);
        SignedAndEnvelopedData envelopedData = new SignedAndEnvelopedData(version, recipientInfos, digestAlgorithms, encryptedContentInfo, null, null, signerInfos);
        return envelopedData;
    }

    public static ASN1Set makeSignerInfos(String keyGeneraterCertIssuerDN, BigInteger keyGeneraterCertSN, byte[] structData)
            throws Exception {
        PrivateKey privateKey = SDKService.config.getPrivateKey();
        SignerIdentifier sid = new SignerIdentifier(new IssuerAndSerialNumber(DnUtil.getRFC4519X500Name(keyGeneraterCertIssuerDN), keyGeneraterCertSN));
        AlgorithmIdentifier digAlgorithm = null;
        AlgorithmIdentifier digEncryptionAlgorithm = null;
        ASN1OctetString encryptedDigest = null;

        if (privateKey instanceof RSAPrivateKey) {
            digAlgorithm = new AlgorithmIdentifier(DigestObjectIdentifiers.sha1);
            digEncryptionAlgorithm = new AlgorithmIdentifier(RsaObjectIdentifiers.rsaEncryption);

            Signature signature = Signature.getInstance("SHA1WithRSAEncryption", "BC");
            signature.initSign(privateKey);
            signature.update(structData);
            encryptedDigest = new DEROctetString(signature.sign());
        } else {
            digAlgorithm = new AlgorithmIdentifier(DigestObjectIdentifiers.sm3);
            digEncryptionAlgorithm = new AlgorithmIdentifier(SM2ObjectIdentifiers.sm2256_sign);

            Signature signature = Signature.getInstance("SM3withSM2", "BC");
            signature.initSign(privateKey);
            signature.update(structData);
            encryptedDigest = new DEROctetString(signature.sign());
        }
        // 组装
        /**
         *         SignerIdentifier        sid,
         *         AlgorithmIdentifier     digAlgorithm,
         *         ASN1Set                 authenticatedAttributes,
         *         AlgorithmIdentifier     digEncryptionAlgorithm,
         *         ASN1OctetString         encryptedDigest,
         *         ASN1Set                 unauthenticatedAttributes)
         */
        ASN1OctetString encryptedDigestString = ASN1OctetString.getInstance(encryptedDigest);
        ASN1Set asn1Set = null;
        SignerInfo signerInfo = new SignerInfo(sid, digAlgorithm, asn1Set, digEncryptionAlgorithm, encryptedDigestString, null);
        return new DERSet(signerInfo);
    }
}
