package com.xdja.ra.utils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.io.*;

/**
 * 文件上传下载工具类
 * syg
 */
public class FileUtils {

    protected static transient final Logger logger = LoggerFactory.getLogger(FileUtils.class.getClass());

    /**
     * 读取文件内容
     *
     * @param path 完整文件路径
     * @return 文件内容字符串
     * @throws Exception
     */
    public static String read(String path) throws Exception {
        StringBuilder content = new StringBuilder();
        BufferedReader reader = null;
        try {
            File file = new File(path);
        } catch (Exception e) {
            e.printStackTrace();
        }
        try {
            reader = new BufferedReader(new FileReader(new File(path)));

            String temp;

            while (null != (temp = reader.readLine())) {
                content.append(temp);
            }
            return content.toString();
        } catch (Exception e) {
            logger.error("读取文件异常", e);
            throw e;
        } finally {
            if (null != reader) {
                try {
                    reader.close();
                } catch (IOException e) {
                    logger.error("读取文件时流关闭异常", e);
                }
            }
        }
    }

    /**
     * 以字节为单位读取文件，通常用于读取二进制文件，如图片
     *
     * @param path
     * @return
     */
    public static String readByBytes(String path) {
        String content = null;
        try {
            InputStream inputStream = new FileInputStream(path);
            StringBuffer sb = new StringBuffer();
            int c = 0;
            byte[] bytes = new byte[1024];
            while ((c = inputStream.read(bytes)) != -1) {
                sb.append(new String(bytes, 0, c, "utf-8"));
            }
            content = sb.toString();
            inputStream.close();
        } catch (Exception e) {
            logger.error("以字节为单位读取文件", e.getMessage());
//            throw new RuntimeException("以字节为单位读取文件异常" + e.getMessage(), e);
        }
        return content;
    }

    /**
     * 以行为单位读取文件，常用于读取面向行的格式化文件
     *
     * @param path
     * @return
     */
    public static String readByLines(String path) {
        String content = null;
        try {
            File file = new File(path);
            if (!file.exists()) {
                file.createNewFile();
            }
            BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(new FileInputStream(path), "utf-8"));
            StringBuffer sb = new StringBuffer();
            String temp = null;

            while ((temp = bufferedReader.readLine()) != null) {
                sb.append(temp);
            }
            content = sb.toString();
            bufferedReader.close();
        } catch (Exception e) {
            logger.error("以行为单位读取文件", e.getMessage());
            throw new RuntimeException("以行为单位读取文件异常" + e.getMessage(), e);
        }
        return content;
    }

    /**
     * 以字符为单位读取文件，常用于读取文本文件
     *
     * @param path
     * @return
     */
    public static String readByChars(String path) {
        String content = null;

        try {
            Reader reader = new InputStreamReader(new FileInputStream(path), "utf-8");
            StringBuffer sb = new StringBuffer();
            char[] tempchars = new char[1024];
            while (reader.read(tempchars) != -1) {
                sb.append(tempchars);
            }
            content = sb.toString();
            reader.close();
        } catch (Exception e) {
            logger.error("以字符为单位读取文件", e.getMessage());
            throw new RuntimeException(" 以字符为单位读取文件异常" + e.getMessage(), e);
        }
        return content;
    }

    /**
     * 向文件保存内容 （字符串）
     *
     * @param content 待保存内容
     * @param path    文件全路径
     * @throws IOException
     */
    public static void saveFile(String content, String path) {
        File file = new File(path);
        if (!file.getParentFile().exists()) {
            boolean mkdir = file.getParentFile().mkdirs();
            if (!mkdir) {
                throw new RuntimeException("创建目标文件所在目录失败！");
            }
        }
        try {
            if(!file.exists()){
                boolean mkfile =file.createNewFile();
            }
        } catch (IOException e) {
            throw new RuntimeException("创建文件失败",e);
        }
        FileWriter fw = null;
        try {
            synchronized (FileUtils.class) {
                fw = new FileWriter(new File(path), false);
                if (content != null) {
                    fw.write(content);
                }
                fw.flush();
            }
        } catch (IOException e) {
            logger.error("写入配置文件异常", e.getMessage());
            throw new RuntimeException("写入配置文件异常" + e.getMessage(), e);
        } finally {
            if (fw != null) {
                try {
                    fw.close();
                } catch (IOException e) {
                    logger.error("关闭文件写入流异常", e.getMessage());
                }
            }
        }
    }

    /**
     * 向文件保存内容 (字节)
     *
     * @param content 待保存内容
     * @param path    文件全路径
     * @throws IOException
     */
    public static void saveFile(byte[] content, String path) {
        File file = new File(path);
        if (!file.getParentFile().exists()) {
            boolean mkdir = file.getParentFile().mkdirs();
            if (!mkdir) {
                throw new RuntimeException("创建目标文件所在目录失败！");
            }
        }
        OutputStream out = null;
        try {
            synchronized (FileUtils.class) {
                out = new FileOutputStream(new File(path), false);
                InputStream is = new ByteArrayInputStream(content);
                byte[] buff = new byte[1024];
                int len = 0;
                while ((len = is.read(buff)) != -1) {
                    out.write(buff, 0, len);
                }
                is.close();
                out.close();
            }
        } catch (IOException e) {
            logger.error("写入配置文件异常", e.getMessage());
            throw new RuntimeException("写入配置文件异常" + e.getMessage(), e);
        } finally {
            if (out != null) {
                try {
                    out.close();
                } catch (IOException e) {
                    logger.error("关闭文件写入流异常", e.getMessage());
                }
            }
        }
    }

    /**
     * 下载文件
     *
     * @param fileFullPath
     * @param fileInfo
     * @return 文件下载路径
     */
    public static String downloadFile(String fileFullPath, byte[] fileInfo) {
        try {
            File file = new File(fileFullPath);
            if (file.exists()) {
                file.delete();
            }
            if (!file.getParentFile().exists()) {
                boolean mkdir = file.getParentFile().mkdirs();
                if (!mkdir) {
                    throw new RuntimeException("创建目标文件所在目录失败！");
                }

            }
            file.createNewFile();
            FileOutputStream fileWriter = new FileOutputStream(file);
            fileWriter.write(fileInfo);
            fileWriter.close();
        } catch (IOException e) {
            logger.error("写入文件，返回下载路径异常", e.getMessage());
            throw new RuntimeException("写入文件，返回下载路径异常" + e.getMessage(), e);
        }
        return fileFullPath;
    }

    /**
     * 保存证书相关的文件到指定目录下
     *
     * @param fileFullPath
     * @return 返回文件所在全路径
     */
    public static String saveCertInfoFile(String fileFullPath, String fileInfo, boolean isClearDir) {
        FileWriter fw = null;
        try {
            File file = new File(fileFullPath);
            if (file.getParentFile().exists() && isClearDir) {
                FileUtils.deleteFile(file.getParentFile());
            }
            if (!file.getParentFile().exists()) {
                boolean mkdir = file.getParentFile().mkdirs();
                if (!mkdir) {
                    throw new RuntimeException("创建目标文件所在目录失败！");
                }
            }
            synchronized (FileUtils.class) {
                fw = new FileWriter(new File(fileFullPath), false);
                if (fileInfo != null) {
                    fw.write(fileInfo);
                }
                fw.flush();
            }
        } catch (IOException e) {
            logger.error("写入文件，返回下载路径异常", e.getMessage());
            throw new RuntimeException("写入文件，返回下载路径异常" + e.getMessage(), e);
        } finally {
            if (fw != null) {
                try {
                    fw.close();
                } catch (IOException e) {
                    logger.error("关闭文件写入流异常", e.getMessage());
                }
            }
        }
        return fileFullPath;
    }

    /**
     * 保存证书相关的文件到指定目录下
     *
     * @param fileFullPath
     * @return 返回文件所在全路径
     */
    public static String saveByteCertInfoFile(String fileFullPath, byte[] fileInfo, boolean isClearDir) {
        FileOutputStream fw = null;
        try {
            File file = new File(fileFullPath);
            if (file.getParentFile().exists() && isClearDir) {
                FileUtils.deleteFile(file.getParentFile());
            }
            if (!file.getParentFile().exists()) {
                boolean mkdir = file.getParentFile().mkdirs();
                if (!mkdir) {
                    throw new RuntimeException("创建目标文件所在目录失败！");
                }
            }
            synchronized (FileUtils.class) {
                fw = new FileOutputStream(new File(fileFullPath), false);
                if (fileInfo != null) {
                    fw.write(fileInfo);
                }
                fw.flush();
            }
        } catch (IOException e) {
            logger.error("写入文件，返回下载路径异常", e.getMessage());
            throw new RuntimeException("写入文件，返回下载路径异常" + e.getMessage(), e);
        } finally {
            if (fw != null) {
                try {
                    fw.close();
                } catch (IOException e) {
                    logger.error("关闭文件写入流异常", e.getMessage());
                }
            }
        }
        return fileFullPath;
    }

    /*
     * 如果是文件 ==》直接删除
     * 如果是目录 ==》必须先删除里面每一层目录里的所有文件，最后才能删除外层的目录
     *              原因：不为空的话 删不了
     */
    public static void deleteFile(File file) {
        if (file.exists()) {//判断路径是否存在
            if (file.isFile()) {//boolean isFile():测试此抽象路径名表示的文件是否是一个标准文件。
                file.delete();
            } else {//不是文件，对于文件夹的操作
                //保存 路径D:/1/新建文件夹2  下的所有的文件和文件夹到listFiles数组中
                File[] listFiles = file.listFiles();//listFiles方法：返回file路径下所有文件和文件夹的绝对路径
                for (File file2 : listFiles) {
                    /*
                     * 递归作用：由外到内先一层一层删除里面的文件 再从最内层 反过来删除文件夹
                     *    注意：此时的文件夹在上一步的操作之后，里面的文件内容已全部删除
                     *         所以每一层的文件夹都是空的  ==》最后就可以直接删除了
                     */
                    deleteFile(file2);
                }
            }
            file.delete();
        } else {
            System.out.println("该file路径不存在！！");
        }
    }

    /**
     * 读取文件二进制内容
     *
     * @param path
     * @return
     * @throws IOException
     */
    public static byte[] readByBinary(String path) {
        InputStream inputStream = null;
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try {
            inputStream = new FileInputStream(path);
            int c = 0;
            byte[] bytes = new byte[1024];
            while ((c = inputStream.read(bytes)) != -1) {
                baos.write(bytes, 0, c);
            }
            inputStream.close();
        } catch (Exception e) {
            throw new RuntimeException("读取文件二进制内容异常" + e.getMessage(), e);
        }
        return baos.toByteArray();
    }

    /**
     * 向文件写入二进制数据
     *
     * @param path
     * @param datas
     * @throws IOException
     */
    public static void writeByBinary(String path, byte[] datas) throws IOException {
        File file = new File(path);
        if (!file.getParentFile().exists()) {
            boolean mkdir = file.getParentFile().mkdirs();
            if (!mkdir) {
                throw new RuntimeException("创建目标文件所在目录失败！");
            }
        }
        if (!file.exists()) {
            file.createNewFile();
        }
        OutputStream os = new FileOutputStream(path);
        os.write(datas);
        os.close();
    }

    /**
     * 将指定文件写入到输出流中
     *
     * @param out  输出流
     * @param file 待写入输出流文件
     * @throws Exception
     */
    public static void write(OutputStream out, File file) throws Exception {
        InputStream in = null;

        try {
            in = new FileInputStream(file);

            byte[] data = new byte[1024];
            int size = 0;
            while ((size = in.read(data)) != -1) {
                out.write(data, 0, size);
            }
        } finally {
            if (null != in) {
                in.close();
            }
        }
    }


    /**
     * 从输入流对象中读取二进制数据
     *
     * @param in
     * @return
     * @throws Exception
     */
    public static byte[] readToBytes(InputStream in) throws Exception {
        ByteArrayOutputStream outputData = new ByteArrayOutputStream();
        byte[] buf = new byte[1024];
        int num;
        while ((num = in.read(buf)) != -1) {
            outputData.write(buf, 0, num);
        }
        return outputData.toByteArray();
    }


    /**
     * 新建文件夹
     *
     * @param path
     * @return
     */
    public static boolean newPath(String path) {
        File file = new File(path);
        if (!file.exists()) {
            file.mkdir();
        }
        if (!file.getParentFile().exists()) {
            boolean mkdir = file.getParentFile().mkdir();
            if (!mkdir) {
                throw new RuntimeException("创建目标文件所在目录失败！");
            }
        }
        return true;
    }

    public static void main(String[] args) {
        newPath("D:/home/xdja/conf/ra/caCerts/");
    }
}


