package com.xdja.ra.sdk;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.xdja.pki.apache.client.core.ClientErrorEnum;
import com.xdja.pki.apache.client.core.ClientKeyStoreConfig;
import com.xdja.pki.apache.client.result.AdaptClientResult;
import com.xdja.pki.apache.client.result.ClientResult;
import com.xdja.pki.apache.client.result.RAClientResult;
import com.xdja.pki.apache.client.utils.ApacheClientHttpUtils;
import com.xdja.ra.asn1.NISTObjectIdentifiers;
import com.xdja.ra.asn1.RsaObjectIdentifiers;
import com.xdja.ra.asn1.SM2ObjectIdentifiers;
import com.xdja.ra.bean.*;
import com.xdja.ra.constant.SdkCommonVariable;
import com.xdja.ra.constant.SdkConstants;
import com.xdja.ra.error.ErrorEnum;
import com.xdja.ra.helper.PKIMessageHelper;
import com.xdja.ra.utils.SdkJsonUtils;
import com.xdja.ra.utils.SdkP10Utils;
import com.xdja.ra.vo.FreeText;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.cmp.CertConfirmContent;
import org.bouncycastle.asn1.cmp.PKIBody;
import org.bouncycastle.asn1.cmp.PKIMessage;
import org.bouncycastle.asn1.crmf.CertReqMessages;
import org.bouncycastle.asn1.crmf.CertRequest;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;

import java.io.IOException;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;


/**
 * 外部通过CMP访问RA-对外保留的api方法
 *
 * @author wly
 */
public class RaCmpApi {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    /**
     * 发送申请制证消息
     * @param applyType     签发类型：1-签发；2-更新；4-恢复
     * @param transId       申请事务Id
     * @param normalName    第三方系统唯一标识
     * @param normalSignAlgOid 保护pkiMessage 的算法要和第三方系统的证书对应
     * @param applyNo       申请编号
     * @param userCertReq   申请信息 两码和P10
     * @return
     */
    public Result sendCertReqMessage(int applyType, String transId, String normalName, String normalSignAlgOid, String applyNo, UserCertReq userCertReq, Integer keyFormat, ClientKeyStoreConfig clientKeyStoreConfig) {
        logger.debug("签发/更新/恢复申请 ======== 【开始】申请类型为{}，申请事务Id为{}, normalName为{}，normalSignAlgOid为{}, applyNo为{} "
                , applyType, transId, normalName, normalSignAlgOid, applyNo);
        Result sdkResult = new Result();
        logger.debug("签发/更新/恢复申请 ======== 0.参与校验");
        // 对入参进行判空处理
        if (StringUtils.isAnyBlank(transId, normalSignAlgOid, normalName, applyNo)) {
            logger.warn("=============== 参数中transId, normalSignAlgOid,normalName,applyNo存在空值");
            return Result.failure(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
        }
        ASN1ObjectIdentifier protectionAlg;
        try {
             protectionAlg = getProtectionAlg(normalSignAlgOid);
        }catch (Exception e){
            return Result.failure(ErrorEnum.PROTECTION_ALG_IS_NOT_SUPPORT);
        }

        logger.debug("签发/更新/恢复申请 ======== 1.向RA获取随机数");
        String genRandomUtl = "/v1/normal/ra/random";
        byte[] recipNonce;
        Map<String, String> paramMap = new ConcurrentHashMap<>();
        paramMap.put("transId", transId);
        try {
            CloseableHttpResponse httpResponse = ApacheClientHttpUtils.sendApacheClientRequest(null, paramMap, null, SDKService.config.getRaBaseUrl() + genRandomUtl, "application/pkixcmp", SDKService.config.getSignName(), SDKService.config.isHttps(), "get", SDKService.config.isUseHsm(),clientKeyStoreConfig);
            RAClientResult RAClientResultFromClientResponse = null;
            try {
                RAClientResultFromClientResponse = AdaptClientResult.getClientResponse(httpResponse);
                if (!RAClientResultFromClientResponse.isSuccess()){
                    sdkResult.setErrorMsg(new ErrorMsg(RAClientResultFromClientResponse.getErrorMsg().getErrorCode(), RAClientResultFromClientResponse.getErrorMsg().getErrorMsg()));
                    return sdkResult;
                }
            } catch (IOException e) {
                logger.error("请求返回数据解析异常",e);
                sdkResult.setErrorMsg(new ErrorMsg(ClientErrorEnum.RESOLVE_CLIENT_RESULT_EXCEPTION.code,ClientErrorEnum.RESOLVE_CLIENT_RESULT_EXCEPTION.desc));
                return sdkResult;
            }
            recipNonce = ((byte[]) RAClientResultFromClientResponse.getInfo());
        } catch (Exception e) {
            logger.error("===============  获取CMP请求随机数异常", e.getMessage());
            return Result.failure(ErrorEnum.GET_CMP_RANDOM_IS_EXCEPTION);
        }
        if (recipNonce == null) {
            logger.error("============== RA返回数据为空");
            return Result.failure(ErrorEnum.RA_OPEN_API_RETURN_INFO_IS_EMPTY);
        }

        // 请求时间戳随机数
        long certReqId = System.nanoTime() + (int) (Math.random() * 900000) + 10000;;
        logger.debug("签发/更新/恢复申请 ======== 2.1封装CertRequest结构体");
        //公钥信息
        byte[] userP10Str =null;
        PublicKey publicKey = null;
        try {
            // DO  如果传了p10 判断p10 公钥格式的正确性 不对返回错误码
            if (userCertReq.getSignP10Str() != null) {
                boolean verifyP10 = SdkP10Utils.verifyP10Info(userCertReq.getSignP10Str());
                if (!verifyP10) {
                    logger.info("非正确格式的P10");
                    return Result.failure(ErrorEnum.P10_GET_PUBLIC_ERROR);
                } else {
                    logger.info("正确格式的P10");
                    publicKey = SdkP10Utils.p10ToPublicKey(userCertReq.getSignP10Str());
                    userP10Str = publicKey.getEncoded();
                    if("RSA".equals(SDKService.config.getSystemCert().getPublicKey().getAlgorithm())){
                        if(!publicKey.getAlgorithm().equals("RSA")){
                            logger.info("申请书秘钥算法和系统不一致");
                            return Result.failure(ErrorEnum.P10_PUBLIC_KEY_NOT_SAME);
                        }
                    }else {
                        String signAlgName = SDKService.config.getSystemCert().getSigAlgName();
                        SubjectPublicKeyInfo instance = SubjectPublicKeyInfo.getInstance(userP10Str);
                        ASN1Encodable asn1Encodable = instance.getAlgorithmId().getParameters();
                        ASN1ObjectIdentifier oid = new ASN1ObjectIdentifier(asn1Encodable.toString());
                        if ("SM3WithSM2".equalsIgnoreCase(signAlgName) ){
                            if (!SM2ObjectIdentifiers.sm2256.equals(oid)) {
                                logger.info("申请书秘钥算法和系统不一致");
                                return Result.failure(ErrorEnum.P10_PUBLIC_KEY_NOT_SAME);
                            }
                        }else {
                            if (!NISTObjectIdentifiers.nist256.equals(oid)) {
                                logger.info("申请书秘钥算法和系统不一致");
                                return Result.failure(ErrorEnum.P10_PUBLIC_KEY_NOT_SAME);
                            }
                        }
                    }
                }
            }
        } catch (IllegalArgumentException e) {
            return Result.failure(ErrorEnum.P10_PUBLIC_KEY_NOT_SAME);
        }
        CertRequest[] certRequests = new CertRequest[1];
        CertRequest certRequest = null;
        try {
            // 封装证书请求对象 CertRequset  普通用户只有一个CertRequest对象
            certRequest = PKIMessageHelper.genCertRequest(userP10Str, protectionAlg, certReqId, SdkConstants.CERT_TYPE_SIGN_2);
            certRequests[0] = certRequest;
        } catch (Exception e) {
            logger.error(" ===============  封装【签名】certRequest异常:{}", e);
            return Result.failure(ErrorEnum.MAKE_CERT_REQUEST_EXCEPTION);
        }

        // 封装证书请求消息对象 CertReqMessages
        logger.debug("签发申请 ======== 3.封装CertReqMessages结构体");
        CertReqMessages certReqMessages;
        try {
            certReqMessages = PKIMessageHelper.genCertReqMessages(certRequests);
        } catch (Exception e) {
            logger.error(" ============== 封装certRequestMessage异常:{}", e);
            return Result.failure(ErrorEnum.MAKE_CERT_REQUEST_MESSAGE_EXCEPTION);
        }

        // Normal产生十六字节随机数 normalSenderNonce
        logger.debug("签发/更新/恢复申请 ======== 4.normal产生十六字节随机数");
        byte[] normalSenderNonce = new byte[16];
        try {
            SecureRandom secureRandom = new SecureRandom();
            secureRandom.nextBytes(normalSenderNonce);
        } catch (Exception e) {
            logger.error(" ============== 生成normalSenderNonce随机数异常:{}", e);
            return Result.failure(ErrorEnum.GET_RANDOM_BY_SECURERANDOM_EXCEPTION);
        }

        // 缓存BaseCMPdebug数据
        Map<String, Object> headerMap = SdkCommonVariable.getHeaderMap();
        headerMap.put(transId, new BaseCMPInfo(normalSenderNonce, recipNonce, transId, certReqId, 0));
      //  logger.debug("=============== sendCertReqMessage.baseCMPInfo.map>> transId: " + transId + " ========== " + SdkJsonUtils.object2Json(headerMap));
        //缓存申请编号和事务ID对应关系
        Map<String, String> normalMap = SdkCommonVariable.getApplyMap();
        normalMap.put(applyNo, transId);
      //  logger.debug("=============== sendCertReqMessage.normalMap: " + SdkJsonUtils.object2Json(normalMap));

        logger.debug("签发/更新/恢复申请 ======== 5.封装PKIMessage结构体");
        FreeText freeText = new FreeText();
        freeText.setApplyNo(applyNo);
        DoubleCode doubleCode = new DoubleCode();
        doubleCode.setAuthCode(userCertReq.getAuthCode());
        doubleCode.setRefCode(userCertReq.getRefCode());
        freeText.setSignSn(SDKService.config.getUserCertSn().toLowerCase());
        freeText.setDoubleCode(doubleCode);
        freeText.setKeyFormat(keyFormat);
        PKIMessage pkiMessage;
        try {
            int tagNo;
            if (applyType == SdkConstants.CERT_APPLY_TYPE_ISSUE_1) {
                //申请签发
                tagNo = PKIBody.TYPE_INIT_REQ;
            } else if (applyType == SdkConstants.CERT_APPLY_TYPE_UPDATE_2) {
                //更新签发
                tagNo = PKIBody.TYPE_KEY_UPDATE_REQ;
            } else if (applyType == SdkConstants.CERT_APPLY_TYPE_RECOVERY_4) {
                // 恢复申请
                tagNo = PKIBody.TYPE_KEY_RECOVERY_REQ;
            } else {
                logger.error("[RaCmpApi#sendCertReqMessage] 不支持的申请类型{}", applyType);
                throw new RuntimeException("不支持的申请类型");
            }
            pkiMessage = PKIMessageHelper.genPKIMessage(SDKService.config.getPrivateKey(), normalName, tagNo, recipNonce
                    , normalSenderNonce, transId, certReqMessages, protectionAlg, JSON.toJSONString(freeText), SDKService.config.getSystemCert());
        } catch (Exception e) {
            logger.error(" =============== 封装PKIMessage异常", e);
            return Result.failure(ErrorEnum.MAKE_PKI_MESSAGE_EXCEPTION);
        }
        logger.debug("签发/更新/恢复申请 ======== 6.发送证书申请 请求");
        byte[] pkiInfo;
        String genCmpNormalPkiUrl = "/v1/normal/cmp";
        try {
            CloseableHttpResponse httpResponse = ApacheClientHttpUtils.sendApacheClientRequest(pkiMessage.getEncoded(), null, null, SDKService.config.getRaBaseUrl() + genCmpNormalPkiUrl, "application/pkixcmp", SDKService.config.getSignName(), SDKService.config.isHttps(), "post", SDKService.config.isUseHsm(),clientKeyStoreConfig);
            RAClientResult resultFromClientResponse = AdaptClientResult.getClientResponse(httpResponse);
            Result postResult = new Result();
            BeanUtils.copyProperties(resultFromClientResponse, postResult);
            if (!postResult.isSuccess()) {
                sdkResult.setErrorMsg(postResult.getErrorMsg());
                return sdkResult;
            }
            pkiInfo = (byte[]) postResult.getInfo();
        } catch (Exception e) {
            logger.error(" ============= 发送证书申请Http请求异常:{}", e);
            return Result.failure(ErrorEnum.SEND_HTTP_MESSAGE_EXCEPTION);
        }
        if (pkiInfo == null) {
            logger.error("============== 接收RA返回的数据内容为空");
            return Result.failure(ErrorEnum.RA_OPEN_API_RETURN_INFO_IS_EMPTY);
        }

        logger.debug("签发/更新/恢复申请 ======== 7.检查RA返回消息");
        Result checkResult = PKIMessageHelper.checkCmpHeaderAndSign(pkiInfo, transId, normalSenderNonce);
        if (!checkResult.isSuccess()) {
            logger.error("签发/更新/恢复申请 ======== 7.1 解析RA返回的头和签名错误");
            sdkResult.setErrorMsg(checkResult.getErrorMsg());
            return sdkResult;
        }
        // 获取返回body中的证书信息
        logger.debug("签发/更新/恢复申请 ======== 8.获取RA返回的证书信息");
        Result resolveResult = PKIMessageHelper.resolveVarietyRepMessage(pkiInfo, transId,keyFormat);
        if (!resolveResult.isSuccess()) {
            logger.error("签发/更新/恢复申请 ======== 8.1 获取RA返回的证书信息错误");
            sdkResult.setErrorMsg(resolveResult.getErrorMsg());
            return sdkResult;
        }
        logger.debug("签发/更新/恢复申请 ======== 【结束】申请事务Id为：{} ", transId);
        return Result.success(resolveResult.getInfo());
    }

    /**
     * 发送确认消息
     *
     * @param applyNo
     * @param normalSignAlgOid
     * @param normalName
     * @return
     */
    public Result sendConfirmMessage(String normalSignAlgOid, String normalName, String applyNo,ClientKeyStoreConfig clientKeyStoreConfig) {

        logger.debug("发送证书 签发和更新的 确认消息 ======== 【开始】申请编号为：{} ", applyNo);
        // 封装PKIConfirmContent返回给RA
        Result sdkResult = new Result();
        logger.debug("确认消息 ======== 0.参与校验");
        // 对入参进行判空处理
        if (StringUtils.isAnyBlank(normalSignAlgOid, normalName, applyNo)) {
            logger.debug("=============== 参数中normalSignAlgOid,applyNo,normalName存在空值");
            return Result.failure(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
        }
        ASN1ObjectIdentifier protectionAlg;
        try {
            protectionAlg = getProtectionAlg(normalSignAlgOid);
        }catch (Exception e){
            return Result.failure(ErrorEnum.PROTECTION_ALG_IS_NOT_SUPPORT);
        }

        BaseCMPInfo baseCMPInfo;
        String transId;
        //获取缓存数据
        Map<String, String> normalMap = SdkCommonVariable.getApplyMap();
        logger.debug("=============== sendCertReqMessage.normalMap: " + SdkJsonUtils.object2Json(normalMap));

        Map<String, Object> headerMap = SdkCommonVariable.getHeaderMap();
        if (headerMap == null || normalMap == null) {
            logger.error(" ================== 本地缓存CMP数据为空");
            return Result.failure(ErrorEnum.LOCAL_CMP_CACHE_IS_EMPTY);
        }
        try {
            transId = normalMap.get(applyNo);
            logger.debug("发送证书 签发和更新的 确认消息 ======== 事务Id为：{} ", transId);
            baseCMPInfo = (BaseCMPInfo) headerMap.get(transId);
            logger.debug(" =================== sendCertConfirmContent.baseCMPInfo>> transId: " + transId + ">>" + JSONObject.toJSONString(baseCMPInfo));
            if (baseCMPInfo == null) {
                logger.error(" ================== 未找到RA发送的该transId:" + transId);
                return Result.failure(ErrorEnum.CANNOT_GET_TRANS_ID_FORM_LOCAL_CACHE);
            }
        } catch (Exception e) {
            return Result.failure(ErrorEnum.LOCAL_CMP_CACHE_IS_EMPTY);
        }
        byte[] recipNonce = baseCMPInfo.getRecipientNonce();
        byte[] normalSenderNonce = baseCMPInfo.getSenderNonce();
        long certReqId = baseCMPInfo.getRequestId();

        logger.debug("确认消息 ======== 1.封装CertConfirmContent结构体");
        //封装申请编号
        FreeText freeText = new FreeText();
        freeText.setApplyNo(applyNo);
        freeText.setSignSn(SDKService.config.getUserCertSn().toLowerCase());
        CertConfirmContent certConfirmContent = null;
        try {
            certConfirmContent = PKIMessageHelper.genCertConfirmContent(transId, certReqId);
        } catch (Exception e) {
            logger.error("封装CertConfirmContent异常{}", e);
            return Result.failure(ErrorEnum.MAKE_CERT_CONFIRM_CONTENT_EXCEPTION);
        }
        logger.debug("确认消息 ======== 2.封装PkiMessage结构体");
        PKIMessage certConfirmPKIMessage = null;
        try {
            certConfirmPKIMessage = PKIMessageHelper.genPKIMessage(SDKService.config.getPrivateKey(), normalName, PKIBody.TYPE_CERT_CONFIRM, recipNonce, normalSenderNonce, transId, certConfirmContent, protectionAlg, JSON.toJSONString(freeText), SDKService.config.getSystemCert());
        } catch (Exception e) {
            logger.error("封装PKIMessage异常：{}", e);
            return Result.failure(ErrorEnum.MAKE_PKI_MESSAGE_EXCEPTION);
        }

        logger.debug("确认消息 ======== 3.发送证书证书确认消息");
        String url = "/v1/normal/cmp";
        RAClientResult resultFromClientResponse = null;
        try {
            CloseableHttpResponse httpResponse = ApacheClientHttpUtils.sendApacheClientRequest(certConfirmPKIMessage.getEncoded(),null,null,SDKService.config.getRaBaseUrl() + url,"application/pkixcmp", SDKService.config.getSignName(),SDKService.config.isHttps(), "post",SDKService.config.isUseHsm(),clientKeyStoreConfig);
            try {
                resultFromClientResponse = AdaptClientResult.getClientResponse(httpResponse);
                if (!resultFromClientResponse.isSuccess()){
                    sdkResult.setErrorMsg(new ErrorMsg(resultFromClientResponse.getErrorMsg().getErrorCode(),resultFromClientResponse.getErrorMsg().getErrorMsg()));
                    return sdkResult;
                }
            } catch (IOException e) {
                logger.error("请求返回数据解析异常",e);
                sdkResult.setErrorMsg(new ErrorMsg(ClientErrorEnum.RESOLVE_CLIENT_RESULT_EXCEPTION.code,ClientErrorEnum.RESOLVE_CLIENT_RESULT_EXCEPTION.desc));
                return sdkResult;
            }
        } catch (Exception e) {
            logger.error(" ============= 发送确认消息Http请求异常:{}", e.getMessage());
            return Result.failure(ErrorEnum.SEND_HTTP_MESSAGE_EXCEPTION);
        }
        sdkResult.setInfo(resultFromClientResponse.getInfo());
        logger.debug("sendConfirmMessage.sendCmpHttpPost.result>>>>" + SdkJsonUtils.object2Json(sdkResult));
        headerMap.remove(transId);
        normalMap.remove(applyNo);
        logger.debug("发送证书 签发和更新的 确认消息 ========  【结束】申请事务Id为：{}", transId);
        return sdkResult;
    }

    /**
     * 发送错误消息
     *
     * @param applyNo
     * @param normalSignAlgOid
     * @param normalName
     * @param errMsg
     * @param errorCode
     * @return
     */
    public Result sendErrorMessage(String applyNo, String normalSignAlgOid, String normalName, String errMsg, int errorCode,ClientKeyStoreConfig clientKeyStoreConfig) {
        logger.debug("发送错误消息 ======== 【开始】申请编号为：{} ", applyNo);
        Result sdkResult = new Result();
        logger.debug("错误消息 ======== 0.参与校验");
        // 对入参进行判空处理
        if (StringUtils.isAnyBlank(applyNo, normalSignAlgOid, normalName)) {
            logger.debug("=============== applyNo,normalSignAlgOid,normalName存在空值");
            return Result.failure(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
        }

        ASN1ObjectIdentifier protectionAlg;
        try {
            protectionAlg = getProtectionAlg(normalSignAlgOid);
        }catch (Exception e){
            return Result.failure(ErrorEnum.PROTECTION_ALG_IS_NOT_SUPPORT);
        }

        //获取缓存数据
        Map<String, String> normalMap = SdkCommonVariable.getApplyMap();
        Map<String, Object> headerMap = SdkCommonVariable.getHeaderMap();
        if (headerMap == null || normalMap == null) {
            logger.debug(" ================== 本地缓存CMP数据为空");
            return Result.failure(ErrorEnum.LOCAL_CMP_CACHE_IS_EMPTY);
        }
        String transId = normalMap.get(applyNo);
        logger.debug("发送错误消息 ======== 事务Id为：{} ", transId);

        BaseCMPInfo baseCMPInfo = (BaseCMPInfo) headerMap.get(transId);
        logger.debug(" =================== sendCertConfirmContent.baseCMPInfo>> transId: " + transId + ">>" + JSONObject.toJSONString(baseCMPInfo));
        if (baseCMPInfo == null) {
            logger.debug(" ================== 未找到RA发送的该transId:" + transId);
            return Result.failure(ErrorEnum.CANNOT_GET_TRANS_ID_FORM_LOCAL_CACHE);
        }
        byte[] recipNonce = baseCMPInfo.getRecipientNonce();
        byte[] normalSenderNonce = baseCMPInfo.getSenderNonce();
        String url = "/v1/normal/cmp";
        Result errorResult = PKIMessageHelper.genErrorPKIMsg(normalName, SDKService.config.getPrivateKey(), SDKService.config.getTrustCert(), SDKService.config.getSystemCert(), errMsg, errorCode, recipNonce, normalSenderNonce, transId, SDKService.config.getRaBaseUrl() + url, protectionAlg,clientKeyStoreConfig);
        if (!errorResult.isSuccess()) {
            sdkResult.setErrorMsg(errorResult.getErrorMsg());
            return sdkResult;
        }

        headerMap.remove(transId);
        normalMap.remove(applyNo);
        logger.debug("发送证书 错误消息 ========  【结束】申请事务Id为：{}", transId);
        return sdkResult;
    }


    public ASN1ObjectIdentifier getProtectionAlg(String normalSignAlg) {
        ASN1ObjectIdentifier protectionAlg;
        if (SdkConstants.SIGN_ALG_NAME_SM3_WHIT_SM2.equals(normalSignAlg)) {
            protectionAlg = SM2ObjectIdentifiers.sm2SignWithSm3;
        } else if (SdkConstants.SIGN_ALG_NAME_SHA1_WHIT_RSA.equals(normalSignAlg)) {
            protectionAlg = RsaObjectIdentifiers.sha1WithRSA;
        } else if (SdkConstants.SIGN_ALG_NAME_SHA256_WHIT_RSA.equals(normalSignAlg)) {
            protectionAlg = RsaObjectIdentifiers.sha256WithRSA;
        } else if(SdkConstants.SIGN_ALG_NAME_NISTP256.equals(normalSignAlg)){
            protectionAlg = NISTObjectIdentifiers.nistSignAlgorithm;
        }else {
            logger.debug("===============  pkiMessage保护算法不支持：" + normalSignAlg);
            throw new RuntimeException("不支持的请求方法");
        }
        return protectionAlg;
    }


}
