package com.xdja.pki.ra.service.manager.auditlog;

import com.xdja.pki.auditlog.dao.ArchiveLogDao;
import com.xdja.pki.auditlog.dao.model.ArchiveLogDO;
import com.xdja.pki.auditlog.service.ArchiveLogService;
import com.xdja.pki.auditlog.service.bean.*;
import com.xdja.pki.auditlog.service.bean.ra.AuditLogOperatorTypeEnum;
import com.xdja.pki.auth.service.AuditLogService;
import com.xdja.pki.core.bean.CoreResult;
import com.xdja.pki.core.bean.ErrorBean;
import com.xdja.pki.core.bean.PageInfo;
import com.xdja.pki.ra.core.common.Result;
import com.xdja.pki.ra.core.common.CommonVariable;
import com.xdja.pki.ra.core.commonenum.ErrorEnum;
import com.xdja.pki.ra.core.constant.Constants;
import com.xdja.pki.ra.core.pkcs7.SignedDataUtils;
import com.xdja.pki.ra.core.util.cert.CertUtils;
import com.xdja.pki.ra.core.util.cert.HsmUtils;
import com.xdja.pki.ra.manager.dao.AdminCertDao;
import com.xdja.pki.ra.manager.dao.RaCertDao;
import com.xdja.pki.ra.manager.dao.model.AdminCertDO;
import com.xdja.pki.ra.manager.dao.model.RaCertDO;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.security.cert.X509Certificate;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;

/**
 * @ClassName: ArchiveLogServiceImpl class
 * @Description: 归档日志业务实现类型
 * @Author: songxuetao
 * @Modifiedby:
 **/
@Service
public class ArchiveLogServiceImpl implements ArchiveLogService {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private ArchiveLogDao archiveLogDao;

    @Autowired
    public AdminCertDao adminCertDao;

    @Autowired
    public RaCertDao raCertDao;

    @Autowired
    private AuditLogService auditLogService;

    private SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

    @Override
    public Object listArchiveLogs(Integer pageNo, Integer pageSize, Integer operatorType, String startTime, String endTime, boolean isExport) {
        if(isExport){
            List<ArchiveLogDO> archiveLogs = (List<ArchiveLogDO>)archiveLogDao.listArchiveLog(pageNo, pageSize, operatorType, startTime, endTime, isExport);
            return generateListVO(archiveLogs);
        } else {
            PageInfo pageInfo = (PageInfo)archiveLogDao.listArchiveLog(pageNo, pageSize, operatorType, startTime, endTime, isExport);
            pageInfo.setDatas(generateListVO((List<ArchiveLogDO>)pageInfo.getDatas()));
            return pageInfo;
        }
    }

    /**
     * @MethodName: generateListVO
     * @Description: 组装Excel列表数据
     * @Param: archiveLogs
     * @Return: java.util.List<com.xdja.pki.ca.securityaudit.service.bean.ArchiveLogListVO>
     * @Author: songxuetao
    **/
    private List<ArchiveLogListVO> generateListVO(List<ArchiveLogDO> archiveLogs) {
        List<ArchiveLogListVO> archiveLogList = new ArrayList<>();
        for(ArchiveLogDO archiveLogDO : archiveLogs){
            ArchiveLogListVO archiveLogListVO = new ArchiveLogListVO();
            archiveLogListVO.setId(archiveLogDO.getId());
            archiveLogListVO.setOperatorSubject(archiveLogDO.getOperatorSubject());
            archiveLogListVO.setOperatorType(archiveLogDO.getOperatorType());
            archiveLogListVO.setOperatorTypeString(AuditLogOperatorTypeEnum.getDescFromType(archiveLogDO.getOperatorType()));
            archiveLogListVO.setOperateClientIp(archiveLogDO.getOperateClientIp());
            archiveLogListVO.setOperateTime(sdf.format(archiveLogDO.getOperateTime()));
            archiveLogListVO.setOperateResult(archiveLogDO.getOperateResult());
            archiveLogListVO.setOperateResultString(AuditLogResultEnum.getValueFromId(archiveLogDO.getOperateResult().intValue()));
            archiveLogListVO.setIsAudit(archiveLogDO.getIsAudit());
            archiveLogListVO.setIsAuditString(AuditLogIsAuditEnum.getValueFromId(archiveLogDO.getIsAudit().intValue()));
            archiveLogListVO.setArchiveTime(sdf.format(archiveLogDO.getArchiveTime()));
            archiveLogList.add(archiveLogListVO);
        }
        return archiveLogList;
    }

    @Override
    public Object getArchiveLogbyId(int id, Integer verify) {
        ArchiveLogDO archiveLogDO;
        try {
            archiveLogDO = archiveLogDao.get(id);
            if (null == archiveLogDO){
                return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.code,ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.desc));
            }
        } catch (Exception e){
            logger.error("获取操作日志实例异常", e);
            return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.code,ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.desc));
        }
        ArchiveLogVO archiveLogVO = new ArchiveLogVO();
        archiveLogVO.setId(archiveLogDO.getId());
        archiveLogVO.setOperatorSubject(archiveLogDO.getOperatorSubject());
        archiveLogVO.setOperatorSn(archiveLogDO.getOperatorSn());
        archiveLogVO.setOperatorType(archiveLogDO.getOperatorType());
        archiveLogVO.setOperatorTypeString(AuditLogOperatorTypeEnum.getDescFromType(archiveLogDO.getOperatorType()));
        archiveLogVO.setOperateClientIp(archiveLogDO.getOperateClientIp());
        archiveLogVO.setOperateContent(archiveLogDO.getOperateContent());
        archiveLogVO.setOperateResult(archiveLogDO.getOperateResult());
        archiveLogVO.setOperateTime(sdf.format(archiveLogDO.getOperateTime()));
        archiveLogVO.setOperateResultString(AuditLogResultEnum.getValueFromId(archiveLogDO.getOperateResult().intValue()));
        archiveLogVO.setOperateModifyDetail(archiveLogDO.getOperateModifyDetail());
        archiveLogVO.setOperateSign(archiveLogDO.getOperateSign());
        archiveLogVO.setIsAudit(archiveLogDO.getIsAudit());
        archiveLogVO.setIsAuditString(AuditLogIsAuditEnum.getValueFromId(archiveLogDO.getIsAudit().intValue()));
        if (archiveLogDO.getIsVerify() != null) {
            archiveLogVO.setIsVerify(archiveLogDO.getIsVerify());
            archiveLogVO.setIsVerifyString(AuditLogIsVerifyEnum.getValueFromId(archiveLogDO.getIsVerify().intValue()));
        }
        //已审核日志具有以下信息
        if (archiveLogDO.getIsAudit().intValue() == 2) {
            archiveLogVO.setAuditSubject(archiveLogDO.getAuditSubject());
            archiveLogVO.setAuditSn(archiveLogDO.getAuditSn());
            archiveLogVO.setAuditNote(archiveLogDO.getAuditNote());
            archiveLogVO.setAuditTime(sdf.format(archiveLogDO.getAuditTime()));
            archiveLogVO.setAuditClientIp(archiveLogDO.getAuditClientIp());
        }
        //1 验签
        if (null != verify && verify == 1) {
            logger.info("get archive log info with verify!");
            Result v;
            if (archiveLogDO.getIsAudit().intValue() == 1){
                v = verifyOperateSign(archiveLogDO);
            } else {
                v = verifyAuditOperateSign(archiveLogDO);
            }
            AuditLogIsVerifyEnum isVerifyEnum = AuditLogIsVerifyEnum.getInstance(v.isSuccess());
            archiveLogVO.setIsVerifyString(isVerifyEnum.value);
        }
        //归档时间
        archiveLogVO.setArchiveTime(sdf.format(archiveLogDO.getArchiveTime()));
        return CoreResult.success(archiveLogVO);
    }

    @Override
    public CoreResult verifyArchiveLog(int id) {
        ArchiveLogDO archiveLogDO;
        try {
            archiveLogDO = archiveLogDao.get(id);
            if (null == archiveLogDO){
                return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.code,ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.desc));
            }
        } catch (Exception e){
            logger.error("获取操作日志实例异常", e);
            return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.code,ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.desc));
        }
        Result result;
        if (archiveLogDO.getIsAudit().intValue() == 1){
            result = verifyOperateSign(archiveLogDO);
        } else {
            result = verifyAuditOperateSign(archiveLogDO);
        }
        ErrorEnum errorEnum = result.getError();
        ErrorBean errorBean = new ErrorBean();
        if(null != errorEnum){
            errorBean = new ErrorBean(errorEnum.code, errorEnum.desc);
        }
        return new CoreResult(result.getCode(), result.getInfo(), errorBean);
    }

    /**
     * @MethodName: verifyOperateSign
     * @Description: 验签操作员签名，第一次服务器证书签名（验签操作记录）
     * @Param: archiveLogDO
     * @Return: com.xdja.pki.ca.core.common.CoreResult
     * @Author: songxuetao
     * @ModifiedBy:
     * @Date: 2020/7/1 20:02
     **/
    private Result verifyOperateSign(ArchiveLogDO archiveLogDO) {
        logger.info("verify:{}",archiveLogDO);
        //获取操作员证书
        String operatorSn = archiveLogDO.getOperatorSn();
        AdminCertDO adminCertDO = adminCertDao.getAdminCertInfo(operatorSn, 2);
        //ManageCertDataDO operateCertDataDO = managerCertDataDao.queryManagerCertDataById(archiveLogDO.getOperatorCertId());
        logger.info("证书：{}",adminCertDO.getCertInfo());
        if (adminCertDO == null) {

            return Result.failure(ErrorEnum.GET_OPERATOR_CERT_EMPTY);
        }
        X509Certificate operateCert = CertUtils.getCertFromStr((adminCertDO.getCertInfo()));
        if (operateCert == null) {
            return Result.failure(ErrorEnum.GET_OPERATOR_CERT_EMPTY);
        }
        boolean verifySignedData = false;
        try {
            verifySignedData = SignedDataUtils.verifySignedData(archiveLogDO.getOperateSign(), operateCert.getPublicKey());
        } catch (Exception e) {
            logger.error("验证管理员操作签名失败", e);
            return Result.failure(ErrorEnum.VERIFY_ADMIN_OPERATOR_SIGN_FAIL);
        }
        if (!verifySignedData) {
            logger.error("验证管理员操作签名失败");
            return Result.failure(ErrorEnum.VERIFY_ADMIN_OPERATOR_SIGN_FAIL);
        }

        boolean verify = false;

        try {
            //获取RA服务器证书
            RaCertDO raCertDO = raCertDao.queryRaCertDataById(archiveLogDO.getServerCertId());
            X509Certificate raServerCert = CertUtils.getCertFromStr(raCertDO.getCertInfo());
            if(Constants.HSM_SERVER_0 != CommonVariable.getIsHsm()){
                verify = HsmUtils.verifyCertByYunHsm(raServerCert.getSigAlgName(), raServerCert.getPublicKey(), archiveLogDO.operatorBase64Encode(), archiveLogDO.getServerSign());
            } else {
                verify = HsmUtils.verifyCertByBC(raServerCert.getSigAlgName(), raServerCert.getPublicKey(), Base64.decode(archiveLogDO.getServerSign()), Base64.decode(archiveLogDO.operatorBase64Encode()));
            }
        } catch (Exception e) {
            logger.error("verify audit error", e);
            return Result.failure(ErrorEnum.VERIFY_SERVER_CERT_SIGN_FAIL);
        }
        if (verify) {
            return Result.success();
        } else {
            return Result.failure(ErrorEnum.VERIFY_SERVER_CERT_SIGN_FAIL);
        }
    }

    /**
     * @MethodName: verifyAuditOperateSign
     * @Description: 验签审计员签名，第二次服务器证书签名（验签操作记录+审计记录）
     * @Param: archiveLogDO
     * @Return: CoreResult
     * @Author: songxuetao
     **/
    private Result verifyAuditOperateSign(ArchiveLogDO archiveLogDO) {
        logger.info("verify:{}",archiveLogDO);
        //获取审计员证书
        String auditSn = archiveLogDO.getAuditSn();
        AdminCertDO adminCertDO = adminCertDao.getAdminCertInfo(auditSn, 2);
        //ManageCertDataDO operateCertDataDO = managerCertDataDao.queryManagerCertDataById(archiveLogDO.getAuditCertId());
        logger.info("证书：{}",adminCertDO.getCertInfo());
        if (adminCertDO == null) {
            return Result.failure(ErrorEnum.GET_AUDITOR_CERT_EMPTY);
        }
        X509Certificate operateCert = CertUtils.getCertFromStr((adminCertDO.getCertInfo()));
        if (operateCert == null) {
            return Result.failure(ErrorEnum.GET_AUDITOR_CERT_EMPTY);
        }
        boolean verifySignedData = false;
        try {
            verifySignedData = SignedDataUtils.verifySignedData(archiveLogDO.getOperateSign(), operateCert.getPublicKey());
        } catch (Exception e) {
            logger.error("验证审计员操作签名失败", e);
            return Result.failure(ErrorEnum.VERIFY_ADMIN_OPERATOR_SIGN_FAIL);
        }
        if (!verifySignedData) {
            logger.error("验证审计员操作签名失败");
            return Result.failure(ErrorEnum.VERIFY_ADMIN_OPERATOR_SIGN_FAIL);
        }
        boolean verify = false;

        try {
            //获取RA服务器证书
            RaCertDO raCertDO = raCertDao.queryRaCertDataById(archiveLogDO.getServerCertId());
            X509Certificate raServerCert = CertUtils.getCertFromStr(raCertDO.getCertInfo());
            if(Constants.HSM_SERVER_0 != CommonVariable.getIsHsm()){
                verify = HsmUtils.verifyCertByYunHsm(raServerCert.getSigAlgName(), raServerCert.getPublicKey(), archiveLogDO.operatorWithAuditInfoBase64Encode(), archiveLogDO.getServerSign());
            } else {
                verify = HsmUtils.verifyCertByBC(raServerCert.getSigAlgName(), raServerCert.getPublicKey(), Base64.decode(archiveLogDO.getServerSign()), Base64.decode(archiveLogDO.operatorWithAuditInfoBase64Encode()));
            }
        } catch (Exception e) {
            logger.error("verify audit error", e);
            return Result.failure(ErrorEnum.VERIFY_SERVER_CERT_SIGN_FAIL);
        }
        if (verify) {
            return Result.success();
        } else {
            return Result.failure(ErrorEnum.VERIFY_SERVER_CERT_SIGN_FAIL);
        }
    }
}
