package com.xdja.pki.ra.service.manager.auditlog;

import com.xdja.pki.auditlog.dao.ArchiveLogDao;
import com.xdja.pki.auditlog.dao.AuditLogDao;
import com.xdja.pki.auditlog.dao.model.ArchiveLogDO;
import com.xdja.pki.auditlog.dao.model.AuditLogDO;
import com.xdja.pki.auditlog.dao.model.AuditLogListDO;
import com.xdja.pki.auditlog.service.bean.*;
import com.xdja.pki.auditlog.service.bean.ra.AuditLogOperatorTypeEnum;
import com.xdja.pki.auth.service.AuditLogService;
import com.xdja.pki.auth.service.bean.AuditLogAuditReq;
import com.xdja.pki.auth.service.bean.AuditSignBean;
import com.xdja.pki.core.bean.CoreResult;
import com.xdja.pki.core.bean.ErrorBean;
import com.xdja.pki.core.bean.PageInfo;
import com.xdja.pki.ra.core.common.Result;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.ra.core.common.CommonVariable;
import com.xdja.pki.ra.core.commonenum.ErrorEnum;
import com.xdja.pki.ra.core.config.ArchiveConfigBean;
import com.xdja.pki.ra.core.constant.Constants;
import com.xdja.pki.ra.core.pkcs7.SignedDataUtils;
import com.xdja.pki.ra.core.util.cert.CertUtils;
import com.xdja.pki.ra.core.util.cert.HsmUtils;
import com.xdja.pki.ra.core.util.time.DateUtils;
import com.xdja.pki.ra.manager.dao.*;
import com.xdja.pki.ra.manager.dao.model.*;
import com.xdja.pki.ra.security.bean.Operator;
import com.xdja.pki.ra.security.util.OperatorUtil;
import com.xdja.pki.ra.service.manager.login.bean.CurrentAdminInfo;
import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.crypto.CryptoException;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.security.cert.X509Certificate;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;

@Service
public class AuditLogServiceImpl implements AuditLogService {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    public AuditLogDao auditLogDao;

    @Autowired
    public AdminCertDao adminCertDao;

    @Autowired
    public RaCertDao raCertDao;

    @Autowired
    private ArchiveLogDao archiveLogDao;

    @Autowired
    private AdminRoleDao adminRoleDao;

    //归档提醒消息参数
    private static double ARCHIVE_PERCENT = 0.8;
    //每天毫秒数
    private static long MS_PER_DAY = 24 * 60 * 60 * 1000;

    @Override
    public CoreResult save(int operatorType, String operatorContext, int operatorResult, String operatorSign) {
        AuditLogDO auditLogDO = new AuditLogDO();
        // 获取当前登录管理员的信息
        Operator operator = OperatorUtil.getOperator();
        if (operator == null || operator.getCurrUser() == null) {
            return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.CANNOT_FIND_CURRENT_LOGIN_ADMIN.code,ErrorEnum.CANNOT_FIND_CURRENT_LOGIN_ADMIN.desc));
        }
        CurrentAdminInfo currentAdminInfo = operator.getCurrUser();

        //操作员证书主体DN
        auditLogDO.setOperatorSubject(currentAdminInfo.getCertDn());
        //操作员证书序列号SN
        auditLogDO.setOperatorSn(currentAdminInfo.getCertSn());
        //操作员证书ID
        auditLogDO.setOperatorCertId(currentAdminInfo.getId());
        //操作时间
        SimpleDateFormat sdf = new SimpleDateFormat("yy-MM-dd HH:mm:ss");
        String format = sdf.format(new Date());
        try {
            Date date = sdf.parse(format);
            auditLogDO.setOperateTime(date);
        } catch (ParseException e) {
            logger.error("时间转换异常", e);
            return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.TIME_CONVERSION_EXCEPTION.code,ErrorEnum.TIME_CONVERSION_EXCEPTION.desc));
        }
        //操作类型
        auditLogDO.setOperatorType(String.valueOf(operatorType));
        //操作者ip
        auditLogDO.setOperateClientIp(currentAdminInfo.getClientIpAddress());
        //操作内容原文
        auditLogDO.setOperateContent(operatorContext);
        //操作结果
        auditLogDO.setOperateResult(operatorResult);
        //操作员签名
        auditLogDO.setOperateSign(operatorSign);

        auditLogDO.setIsAudit(AuditLogIsAuditEnum.FALSE.id);

        try {
            RaCertDO raServerCert = raCertDao.getNewRaCertInfo();
            if (raServerCert == null) {
                logger.error("获取RA服务器证书为空!");
                return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.GET_RA_SERVER_CERT_EMPTY.code,ErrorEnum.GET_RA_SERVER_CERT_EMPTY.desc));
            }
            //ra 服务器证书id DO BUG-80738 20200507-WLY
            auditLogDO.setServerCertId(raServerCert.getId());
            String content = auditLogDO.operatorBase64Encode();

            String sign = null;
           // if (Constants.KEY_ALG_NAME_SM2.equalsIgnoreCase(CommonVariable.getKeyAlgName())) {
           // if(CommonVariable.isUseHsm() && Constants.KEY_ALG_NAME_SM2.equalsIgnoreCase(CommonVariable.getKeyAlgName())){
            if(Constants.HSM_SERVER_XDJA_1 == CommonVariable.getIsHsm() && Constants.KEY_ALG_NAME_SM2.equalsIgnoreCase(CommonVariable.getKeyAlgName())){
                sign = HsmUtils.signByYunHsm(raServerCert.getSignAlg(), CommonVariable.getKeyIndex(), CommonVariable.getKeyPwd(), content);
            } else {
                sign = HsmUtils.signByBC(raServerCert.getSignAlg(), CommonVariable.getRaSignPriKey(), content);
            }
            if (StringUtils.isBlank(sign)) {
                return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.SIGN_BY_HSM_IS_ERROR.code,ErrorEnum.SIGN_BY_HSM_IS_ERROR.desc));
            }
            auditLogDO.setServerSign(sign);
            auditLogDao.saveLog(auditLogDO);
        } catch (CryptoException | SdfSDKException e) {
            logger.error("RA服务器证书签名异常!", e);
            return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.RA_SERVER_CERT_SIGN_EXCEPTION.code,ErrorEnum.RA_SERVER_CERT_SIGN_EXCEPTION.desc));
        } catch (Exception e) {
            return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.RA_SERVER_CERT_SIGN_EXCEPTION.code,ErrorEnum.RA_SERVER_CERT_SIGN_EXCEPTION.desc));
        }
        return new CoreResult(Result.SUCCESS, null);
    }

    @Override
    public boolean saveAuditLog(int operatorType, String operatorContext, int operatorResult, String operatorSign) {
        AuditLogDO auditLogDO = new AuditLogDO();
        // 获取当前登录管理员的信息
        Operator operator = OperatorUtil.getOperator();
        if (operator == null || operator.getCurrUser() == null) {
            return false;
        }
        CurrentAdminInfo currentAdminInfo = operator.getCurrUser();

        //操作员证书主体DN
        auditLogDO.setOperatorSubject(currentAdminInfo.getCertDn());
        //操作员证书序列号SN
        auditLogDO.setOperatorSn(currentAdminInfo.getCertSn());
        //操作员证书ID
        auditLogDO.setOperatorCertId(currentAdminInfo.getId());
        //操作时间
        SimpleDateFormat sdf = new SimpleDateFormat("yy-MM-dd HH:mm:ss");
        String format = sdf.format(new Date());
        try {
            Date date = sdf.parse(format);
            auditLogDO.setOperateTime(date);
        } catch (ParseException e) {
            logger.error("时间转换异常", e);
            return false;
        }
        //操作类型
        auditLogDO.setOperatorType(String.valueOf(operatorType));
        //操作者ip
        auditLogDO.setOperateClientIp(currentAdminInfo.getClientIpAddress());
        //操作内容原文
        auditLogDO.setOperateContent(operatorContext);
        //操作结果
        auditLogDO.setOperateResult(operatorResult);
        //操作员签名
        auditLogDO.setOperateSign(operatorSign);

        auditLogDO.setIsAudit(AuditLogIsAuditEnum.FALSE.id);

        try {
            RaCertDO raServerCert = raCertDao.getNewRaCertInfo();
            if (raServerCert == null) {
                logger.error("获取RA服务器证书为空!");
                return false;
            }
            //ra 服务器证书id DO BUG-80738 20200507-WLY
            auditLogDO.setServerCertId(raServerCert.getId());
            String content = auditLogDO.operatorBase64Encode();

            String sign = null;
            // if (Constants.KEY_ALG_NAME_SM2.equalsIgnoreCase(CommonVariable.getKeyAlgName())) {
            // if(CommonVariable.isUseHsm() && Constants.KEY_ALG_NAME_SM2.equalsIgnoreCase(CommonVariable.getKeyAlgName())){
            if(Constants.HSM_SERVER_XDJA_1 == CommonVariable.getIsHsm() && Constants.KEY_ALG_NAME_SM2.equalsIgnoreCase(CommonVariable.getKeyAlgName())){
                sign = HsmUtils.signByYunHsm(raServerCert.getSignAlg(), CommonVariable.getKeyIndex(), CommonVariable.getKeyPwd(), content);
            } else {
                sign = HsmUtils.signByBC(raServerCert.getSignAlg(), CommonVariable.getRaSignPriKey(), content);
            }
            if (StringUtils.isBlank(sign)) {
                return false;
            }
            auditLogDO.setServerSign(sign);
            auditLogDao.saveLog(auditLogDO);
        } catch (CryptoException | SdfSDKException e) {
            logger.error("RA服务器证书签名异常!", e);
            return false;
        } catch (Exception e) {
            return false;
        }
        return true;
    }

    @Override
    public boolean saveAuditLog(int operatorType, String operatorContext, int operatorResult, String operatorSign, String operateModifyDetail) {
        AuditLogDO auditLogDO = new AuditLogDO();
        // 获取当前登录管理员的信息
        Operator operator = OperatorUtil.getOperator();
        if (operator == null || operator.getCurrUser() == null) {
            return false;
        }
        CurrentAdminInfo currentAdminInfo = operator.getCurrUser();

        //操作员证书主体DN
        auditLogDO.setOperatorSubject(currentAdminInfo.getCertDn());
        //操作员证书序列号SN
        auditLogDO.setOperatorSn(currentAdminInfo.getCertSn());
        //操作员证书ID
        auditLogDO.setOperatorCertId(currentAdminInfo.getId());
        //操作时间
        SimpleDateFormat sdf = new SimpleDateFormat("yy-MM-dd HH:mm:ss");
        String format = sdf.format(new Date());
        try {
            Date date = sdf.parse(format);
            auditLogDO.setOperateTime(date);
        } catch (ParseException e) {
            logger.error("时间转换异常", e);
            return false;
        }
        //操作类型
        auditLogDO.setOperatorType(String.valueOf(operatorType));
        //操作者ip
        auditLogDO.setOperateClientIp(currentAdminInfo.getClientIpAddress());
        //操作内容原文
        auditLogDO.setOperateContent(operatorContext);
        //操作结果
        auditLogDO.setOperateResult(operatorResult);
        //操作员签名
        auditLogDO.setOperateSign(operatorSign);
        //操作更改详情
        auditLogDO.setOperateModifyDetail(operateModifyDetail);

        auditLogDO.setIsAudit(AuditLogIsAuditEnum.FALSE.id);

        try {
            RaCertDO raServerCert = raCertDao.getNewRaCertInfo();
            if (raServerCert == null) {
                logger.error("获取RA服务器证书为空!");
                return false;
            }
            //ra 服务器证书id DO BUG-80738 20200507-WLY
            auditLogDO.setServerCertId(raServerCert.getId());
            String content = auditLogDO.operatorBase64Encode();

            String sign = null;
            // if (Constants.KEY_ALG_NAME_SM2.equalsIgnoreCase(CommonVariable.getKeyAlgName())) {
            // if(CommonVariable.isUseHsm() && Constants.KEY_ALG_NAME_SM2.equalsIgnoreCase(CommonVariable.getKeyAlgName())){
            if(Constants.HSM_SERVER_XDJA_1 == CommonVariable.getIsHsm() && Constants.KEY_ALG_NAME_SM2.equalsIgnoreCase(CommonVariable.getKeyAlgName())){
                sign = HsmUtils.signByYunHsm(raServerCert.getSignAlg(), CommonVariable.getKeyIndex(), CommonVariable.getKeyPwd(), content);
            } else {
                sign = HsmUtils.signByBC(raServerCert.getSignAlg(), CommonVariable.getRaSignPriKey(), content);
            }
            if (StringUtils.isBlank(sign)) {
                return false;
            }
            auditLogDO.setServerSign(sign);
            auditLogDao.saveLog(auditLogDO);
        } catch (CryptoException | SdfSDKException e) {
            logger.error("RA服务器证书签名异常!", e);
            return false;
        } catch (Exception e) {
            return false;
        }
        return true;
    }

    @Override
    public CoreResult getAuditLogList(Integer pageNo, Integer pageSize, String operatorSubject, Integer operatorType, Integer isAudit, String startTime, String endTime) {
        if (pageNo == 0 && pageSize == 0) {
            List<AuditLogListDO> doList = (List<AuditLogListDO>) auditLogDao.getLogList(pageNo, pageSize, operatorSubject, operatorType, isAudit, startTime, endTime);
            return new CoreResult(Result.SUCCESS, setDataList(doList));
        } else {
            PageInfo list = (PageInfo) auditLogDao.getLogList(pageNo, pageSize, operatorSubject, operatorType, isAudit, startTime, endTime);
            list.setDatas(setDataList((List<AuditLogListDO>)list.getDatas()));
            return new CoreResult(Result.SUCCESS, list);
        }
    }

    private List<AuditLogListVO> setDataList(List<AuditLogListDO> list) {
        List<AuditLogListVO> infos = new ArrayList<>();
        for (Object obj : list) {
            AuditLogListDO logDo = (AuditLogListDO) obj;
            AuditLogListVO logVo = new AuditLogListVO();
            logVo.setId(logDo.getId());
            logVo.setOperatorSubject(logDo.getOperatorSubject());
            logVo.setOperatorType(logDo.getOperatorType());
            logVo.setOperatorTypeString(AuditLogOperatorTypeEnum.getDescFromType(logDo.getOperatorType()));
            logVo.setOperateClientIp(logDo.getOperateClientIp());
            logVo.setOperateResult(logDo.getOperateResult());
            logVo.setOperateResultString(AuditLogResultEnum.getValueFromId(logDo.getOperateResult()));
            logVo.setIsAudit(logDo.getIsAudit());
            logVo.setIsAuditString(AuditLogIsAuditEnum.getValueFromId(logDo.getIsAudit()));
            logVo.setOperateTime(logDo.getOperateTime());
            infos.add(logVo);
        }
        return infos;
    }

    @Override
    public CoreResult doAuditLog(AuditLogAuditReq auditReq, AuditSignBean bean) {
        AuditLogDO auditLogDO;
        try {
            auditLogDO = auditLogDao.getLogById(auditReq.getId());
            if (null == auditLogDO){
                return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.code,ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.desc));
            }
        } catch (Exception e){
            logger.error("获取操作日志实例异常", e);
            return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.code,ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.desc));
        }
        if (auditLogDO.getIsAudit() == AuditLogIsAuditEnum.TRUE.id) {
            logger.error("该日志已经被审核过!");
            return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.LOG_IS_ALREADY_AUDIT.code,ErrorEnum.LOG_IS_ALREADY_AUDIT.desc));
        }
        //审计员校验操作员签名，第一次服务器签名
        Result auditVerifyResult = verifyOperateSign(auditLogDO);
        if(!auditVerifyResult.isSuccess()){
            return new CoreResult(auditVerifyResult.getCode(),auditVerifyResult.getInfo());
        }
        // 获取当前登录管理员的信息
        Operator operator = OperatorUtil.getOperator();
        if (operator == null || operator.getCurrUser() == null) {
            return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.CANNOT_FIND_CURRENT_LOGIN_ADMIN.code,ErrorEnum.CANNOT_FIND_CURRENT_LOGIN_ADMIN.desc));
        }
        CurrentAdminInfo currentAdminInfo = operator.getCurrUser();
        auditLogDO.setAuditClientIp(currentAdminInfo.getClientIpAddress());
        auditLogDO.setAuditCertId(currentAdminInfo.getId());
        auditLogDO.setAuditSn(currentAdminInfo.getCertSn());
        auditLogDO.setAuditSubject(currentAdminInfo.getCertDn());

        auditLogDO.setAuditSign(bean.getSign());
        auditLogDO.setAuditTime(new Date());
        auditLogDO.setIsAudit(AuditLogIsAuditEnum.TRUE.id);
        auditLogDO.setAuditNote(auditReq.getAuditNote());
        auditLogDO.setIsVerify(auditReq.getIsVerify());
        //auditLog整条记录内容
        String content = auditLogDO.operatorWithAuditInfoBase64Encode();
        String sign;
        try{
            RaCertDO raServerCert = raCertDao.getNewRaCertInfo();
            if (raServerCert == null) {
                logger.error("获取RA服务器证书为空!");
                return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.GET_RA_SERVER_CERT_EMPTY.code,ErrorEnum.GET_RA_SERVER_CERT_EMPTY.desc));
            }
            if(Constants.HSM_SERVER_XDJA_1 == CommonVariable.getIsHsm() && Constants.KEY_ALG_NAME_SM2.equalsIgnoreCase(CommonVariable.getKeyAlgName())){
                sign = HsmUtils.signByYunHsm(raServerCert.getSignAlg(), CommonVariable.getKeyIndex(), CommonVariable.getKeyPwd(), content);
            } else {
                sign = HsmUtils.signByBC(raServerCert.getSignAlg(), CommonVariable.getRaSignPriKey(), content);
            }
            if (StringUtils.isBlank(sign)) {
                return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.SIGN_BY_HSM_IS_ERROR.code,ErrorEnum.SIGN_BY_HSM_IS_ERROR.desc));
            }
            //更新服务器对auditLog整条记录的签名（第二次服务器签名）
            auditLogDO.setServerSign(sign);
        } catch (Exception e) {
            logger.error("服务器签名值验签失败", e);
            return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.VERIFY_SERVER_CERT_SIGN_FAIL.code,ErrorEnum.VERIFY_SERVER_CERT_SIGN_FAIL.desc));
        }
        int success = auditLogDao.updateLog(auditLogDO);
        if (success == 1) {
            return new CoreResult(Result.SUCCESS,null);
        } else {
            logger.error("更新审核日志失败!");
            return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.UPDATE_AUDIT_IS_FAILURE.code,ErrorEnum.UPDATE_AUDIT_IS_FAILURE.desc));
        }
    }

    @Override
    public Object getOperateLogById(int id) {
        AuditLogDO auditLogDO;
        try {
            auditLogDO = auditLogDao.getLogById(id);
            if (null == auditLogDO){
                return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.code,ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.desc));
            }
        } catch (Exception e){
            logger.error("获取操作日志实例异常", e);
            return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.code,ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.desc));
        }
        return CoreResult.success(auditLogDO);
    }

    @Override
    public CoreResult get(int id, Integer verify) {
        AuditLogDO logDO;
        try {
            logDO = auditLogDao.getLogById(id);
            if (null == logDO){
                return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.code,ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.desc));
            }
        } catch (Exception e){
            logger.error("获取操作日志实例异常", e);
            return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.code,ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.desc));
        }
        AuditLogVO logVO = new AuditLogVO();
        logVO.setId(logDO.getId());
        logVO.setOperatorSubject(logDO.getOperatorSubject());
        logVO.setOperatorSn(logDO.getOperatorSn());
        logVO.setOperatorType(logDO.getOperatorType());
        logVO.setOperatorTypeString(AuditLogOperatorTypeEnum.getDescFromType(logDO.getOperatorType()));
        logVO.setOperateClientIp(logDO.getOperateClientIp());
        logVO.setOperateContent(logDO.getOperateContent());
        logVO.setOperateResult(logDO.getOperateResult());
        logVO.setOperateResultString(AuditLogResultEnum.getValueFromId(logDO.getOperateResult()));
        logVO.setOperateModifyDetail(logDO.getOperateModifyDetail());
        logVO.setOperateSign(logDO.getOperateSign());
        logVO.setIsAudit(logDO.getIsAudit());
        logVO.setIsAuditString(AuditLogIsAuditEnum.getValueFromId(logDO.getIsAudit()));
        logVO.setOperateTime(logDO.getOperateTime());
        if (logDO.getIsVerify() != null) {
            logVO.setIsVerify(logDO.getIsVerify());
            logVO.setIsVerifyString(AuditLogIsVerifyEnum.getValueFromId(logDO.getIsVerify()));
        }
        //已审核日志具有以下信息
        if (logDO.getIsAudit().intValue() == 2) {
            logVO.setAuditSubject(logDO.getAuditSubject());
            logVO.setAuditSn(logDO.getAuditSn());
            logVO.setAuditNote(logDO.getAuditNote());
            logVO.setAuditTime(logDO.getAuditTime());
            logVO.setAuditClientIp(logDO.getAuditClientIp());
        }
        if (verify != null && verify == 1) {
            logger.info("get audit log info with verify!");
            Result v;
            //判断是否审核
            if (logDO.getIsAudit().intValue() == 1){
                v = verifyOperateSign(logDO);
            } else {
                v = verifyAuditOperateSign(logDO);
            }
            AuditLogIsVerifyEnum isVerifyEnum = AuditLogIsVerifyEnum.getInstance(v.isSuccess());
            logVO.setIsVerify(isVerifyEnum.id);
            logVO.setIsVerifyString(isVerifyEnum.value);
        }
        return new CoreResult(Result.SUCCESS, logVO);
    }

    @Override
    public CoreResult verify(int id) {
        AuditLogDO auditLogDO;
        try {
            auditLogDO = auditLogDao.getLogById(id);
            if (null == auditLogDO){
                return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.code,ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.desc));
            }
        } catch (Exception e){
            logger.error("获取操作日志实例异常", e);
            return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.code,ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.desc));
        }
        Result result = verifyAuditOperateSign(auditLogDO);
        ErrorEnum errorEnum = result.getError();
        ErrorBean errorBean = new ErrorBean();
        if(null != errorEnum){
            errorBean = new ErrorBean(errorEnum.code, errorEnum.desc);
        }
        return new CoreResult(result.getCode(), result.getInfo(), errorBean);
    }

    private Result verifyOperateSign(AuditLogDO auditLogDO) {
        //操作员签名验签,获取操作员证书
        String operatorSn = auditLogDO.getOperatorSn();
        AdminCertDO adminCertDO = adminCertDao.getAdminCertInfo(operatorSn, 2);
        if (adminCertDO == null) {
            logger.error("获取操作员证书为空");
            return Result.failure(ErrorEnum.GET_OPERATOR_CERT_EMPTY);
        }
        String data = adminCertDO.getCertInfo();
        X509Certificate operatorCert;
        try {
            //获取操作员证书。
            operatorCert = CertUtils.getCertFromStr(data);
            boolean verifySignedData;
            try {
                verifySignedData = SignedDataUtils.verifySignedData(auditLogDO.getOperateSign(), operatorCert.getPublicKey());
            } catch (Exception e) {
                logger.error("验证管理员操作签名失败", e);
                return Result.failure(ErrorEnum.VERIFY_ADMIN_OPERATOR_SIGN_FAIL);
            }
            if (!verifySignedData) {
                logger.error("验证管理员操作签名失败");
                return Result.failure(ErrorEnum.VERIFY_ADMIN_OPERATOR_SIGN_FAIL);
            }
            //RA 服务器公钥对签名数据验签
            boolean verify;
            try {
                //获取服务器证书
                RaCertDO raCertDO = raCertDao.queryRaCertDataById(auditLogDO.getServerCertId());
                X509Certificate raServerCert = CertUtils.getCertFromStr(raCertDO.getCertInfo());
                if(Constants.HSM_SERVER_0 != CommonVariable.getIsHsm()){
                    verify = HsmUtils.verifyCertByYunHsm(raServerCert.getSigAlgName(), raServerCert.getPublicKey(), auditLogDO.operatorBase64Encode(), auditLogDO.getServerSign());
                } else {
                    verify = HsmUtils.verifyCertByBC(raServerCert.getSigAlgName(), raServerCert.getPublicKey(), Base64.decode(auditLogDO.getServerSign()), Base64.decode(auditLogDO.operatorBase64Encode()));
                }
            } catch (Exception e) {
                logger.error("服务器签名值验签失败", e);
                return Result.failure(ErrorEnum.VERIFY_SERVER_CERT_SIGN_FAIL);
            }
            if (verify) {
                return Result.success();
            } else {
                logger.error("服务器签名值验签失败");
                return Result.failure(ErrorEnum.VERIFY_SERVER_CERT_SIGN_FAIL);
            }
        } catch (Exception e) {
            logger.error("证书转换错误", e);
            return Result.failure(ErrorEnum.GENERATE_CERT_IS_ERROR);
        }
    }

    private Result verifyAuditOperateSign(AuditLogDO auditLogDO) {
        //审计员签名验签,获取审计员证书
        String auditSn = auditLogDO.getAuditSn();
        AdminCertDO adminCertDO = adminCertDao.getAdminCertInfo(auditSn, 2);
        if (adminCertDO == null) {
            logger.error("获取审计员证书为空");
            return Result.failure(ErrorEnum.GET_AUDITOR_CERT_EMPTY);
        }
        String data = adminCertDO.getCertInfo();
        X509Certificate operatorCert;
        try {
            //获取操作员证书。
            operatorCert = CertUtils.getCertFromStr(data);
            boolean verifySignedData;
            try {
                verifySignedData = SignedDataUtils.verifySignedData(auditLogDO.getAuditSign(), operatorCert.getPublicKey());
            } catch (Exception e) {
                logger.error("验证审计员操作签名失败", e);
                return Result.failure(ErrorEnum.VERIFY_AUDITOR_SIGN_FAIL);
            }
            if (!verifySignedData) {
                logger.error("验证审计员操作签名失败");
                return Result.failure(ErrorEnum.VERIFY_AUDITOR_SIGN_FAIL);
            }
            //RA 服务器公钥对签名数据验签
            boolean verify;
            try {
                //获取服务器证书
                RaCertDO raCertDO = raCertDao.queryRaCertDataById(auditLogDO.getServerCertId());
                X509Certificate raServerCert = CertUtils.getCertFromStr(raCertDO.getCertInfo());
                if(Constants.HSM_SERVER_0 != CommonVariable.getIsHsm()){
                    verify = HsmUtils.verifyCertByYunHsm(raServerCert.getSigAlgName(), raServerCert.getPublicKey(), auditLogDO.operatorWithAuditInfoBase64Encode(), auditLogDO.getServerSign());
                } else {
                    verify = HsmUtils.verifyCertByBC(raServerCert.getSigAlgName(), raServerCert.getPublicKey(), Base64.decode(auditLogDO.getServerSign()), Base64.decode(auditLogDO.operatorWithAuditInfoBase64Encode()));
                }
            } catch (Exception e) {
                logger.error("服务器签名值验签失败", e);
                return Result.failure(ErrorEnum.VERIFY_SERVER_CERT_SIGN_FAIL);
            }
            if (verify) {
                return Result.success();
            } else {
                logger.error("服务器签名值验签失败");
                return Result.failure(ErrorEnum.VERIFY_SERVER_CERT_SIGN_FAIL);
            }
        } catch (Exception e) {
            logger.error("证书转换错误", e);
            return Result.failure(ErrorEnum.GENERATE_CERT_IS_ERROR);
        }
    }

    @Override
    public CoreResult doArchiveOperateLog(String operatorSubject, Integer operatorType, Integer isAudit, String startTime, String endTime) {
        HashMap<String, Object> archiveMap = new HashMap<>();
        try {
            List<AuditLogDO> auditLogDOS = (List<AuditLogDO>) auditLogDao.archiveList(operatorSubject, operatorType, isAudit, startTime, endTime);
            if (null == auditLogDOS || auditLogDOS.isEmpty()){
                logger.info("手动归档操作日志 没有可归档的日志");
                archiveMap.put("archiveResult", true);
                archiveMap.put("archiveNum", 0);
                return new CoreResult(Result.SUCCESS,archiveMap);
            }
            List<ArchiveLogDO> archiveLogDOList = generateArchiveLogFromAuditLog(auditLogDOS);
            int archiveResult = archiveLogDao.saveArchiveLogList(archiveLogDOList);
            int deleteResult = auditLogDao.deleteAuditLogList(auditLogDOS);
            logger.info("手动归档 删除从AUDIT_LOG表 归档到ARCHIVE_LOG表 "+ deleteResult + "数据");
            if (archiveResult != 0) {
                archiveMap.put("archiveResult", true);
                archiveMap.put("archiveNum", archiveResult);
                return new CoreResult(Result.SUCCESS,archiveMap);
            }
            archiveMap.put("archiveResult", false);
            archiveMap.put("archiveNum", auditLogDOS.size());
            return new CoreResult(Result.FAILURE,archiveMap);
        } catch (Exception e){
            logger.error("手动归档操作日志出现异常", e);
            return new CoreResult(Result.FAILURE, "手动归档操作日志出现异常");
        }
    }

    /**
     * @MethodName: generateArchiveLogFromAuditLog
     * @Description: auditLog转换archiveLog
     * @Param: auditLogDOS
     * @Return: java.util.List<com.xdja.pki.ca.securityaudit.dao.models.ArchiveLogDO>
     * @Author: songxuetao
     * @ModifiedBy:
     * @Date: 2020/7/4 15:14
     **/
    private List<ArchiveLogDO> generateArchiveLogFromAuditLog(List<AuditLogDO> auditLogDOS){
        List<ArchiveLogDO> archiveLogDOList = new ArrayList<>();
        for(AuditLogDO auditLogDO : auditLogDOS){
            ArchiveLogDO archiveLogDO = new ArchiveLogDO();
            /*archiveLogDO.setOperatorSubject(auditLogDO.getOperatorSubject());
            archiveLogDO.setOperatorSn(auditLogDO.getOperatorSn());
            archiveLogDO.setOperatorCertId(auditLogDO.getOperatorCertId());
            archiveLogDO.setOperatorType(auditLogDO.getOperatorType());
            archiveLogDO.setOperateClientIp(auditLogDO.getOperateClientIp());
            archiveLogDO.setOperateContent(auditLogDO.getOperateContent());
            archiveLogDO.setOperateResult(auditLogDO.getOperateResult());
            archiveLogDO.setOperateSign(auditLogDO.getOperateSign());
            archiveLogDO.setIsAudit(auditLogDO.getIsAudit());
            archiveLogDO.setIsVerify(auditLogDO.getIsVerify());
            archiveLogDO.setAuditSubject(auditLogDO.getAuditSubject());
            archiveLogDO.setAuditSn(auditLogDO.getAuditSn());
            archiveLogDO.setAuditCertId(auditLogDO.getAuditCertId());
            archiveLogDO.setAuditNote(auditLogDO.getAuditNote());
            archiveLogDO.setAuditSign(auditLogDO.getAuditSign());
            archiveLogDO.setServerSign(auditLogDO.getServerSign());
            archiveLogDO.setServerCertId(auditLogDO.getServerCertId());
            archiveLogDO.setOperateTime(auditLogDO.getOperateTime());
            archiveLogDO.setAuditTime(auditLogDO.getAuditTime());
            archiveLogDO.setAuditClientIp(auditLogDO.getAuditClientIp());*/
            BeanUtils.copyProperties(auditLogDO, archiveLogDO, "id");
            archiveLogDO.setArchiveTime(new Date());
            archiveLogDOList.add(archiveLogDO);
        }
        return archiveLogDOList;
    }

    @Override
    public Boolean isAllowLogin() {
        try {
            ArchiveConfigBean archiveConfigBean = CommonVariable.getArchiveConfig();
            if(null == archiveConfigBean || archiveConfigBean.getMaxAuditLogLimit() == null){
                return true;
            }
            int maxLogLimit = archiveConfigBean.getMaxAuditLogLimit().intValue();
            int currentRecordNum = auditLogDao.getTotalRecords();
            if (maxLogLimit > currentRecordNum) {
                return true;
            }
            return false;
        } catch (Exception e){
            logger.error("获取登录许可或日志写入出现异常", e);
            return false;
        }
    }

    @Override
    public CoreResult getAuditMsgInfo() {
        try {
            ArchiveConfigBean archiveConfigBean = CommonVariable.getArchiveConfig();
            if(null == archiveConfigBean || archiveConfigBean.getAuditLogPeriod() == null){
                return new CoreResult(Result.SUCCESS, null);
            }
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
            long currentTime = DateUtils.getStartTime(sdf.format(new Date()));
            long auditLogPeriod = archiveConfigBean.getAuditLogPeriod().intValue();
            long logPeriod = auditLogPeriod * MS_PER_DAY;
            String compareTime = sdf.format(currentTime - logPeriod);
            long overAudit = auditLogDao.getOverAuditRecordNum(compareTime);
            if (overAudit != 0) {
                return new CoreResult(Result.SUCCESS,"有" + overAudit + "条日志已超审核周期，请及时进行审核");
            }
            return new CoreResult(Result.SUCCESS, null);
        } catch (Exception e){
            return new CoreResult(Result.FAILURE, "获取审核提醒消息出现异常");
        }
    }

    @Override
    public CoreResult getArchiveMsgInfo() {
        try {
            ArchiveConfigBean archiveConfigBean = CommonVariable.getArchiveConfig();
            if(null == archiveConfigBean || archiveConfigBean.getMaxAuditLogLimit() == null){
                return new CoreResult(Result.SUCCESS, null);
            }
            long maxLogLimit = archiveConfigBean.getMaxAuditLogLimit().intValue();
            long currentRecordNum = auditLogDao.getTotalRecords();
            if (currentRecordNum >= maxLogLimit * ARCHIVE_PERCENT) {
                return new CoreResult(Result.SUCCESS,"日志即将达到存储操作日志的最大规模，请及时进行归档");
            }
            return new CoreResult(Result.SUCCESS, null);
        } catch (Exception e){
            return new CoreResult(Result.FAILURE, "获取归档提醒消息出现异常");
        }
    }

    @Override
    public CoreResult doAutoArchiveOperateLog() {
        try {
            logger.debug("自动归档开始》》》》》》");
            HashMap<String, Object> archiveMap = new HashMap<>();
            ArchiveConfigBean archiveConfigBean = CommonVariable.getArchiveConfig();
            if (null == archiveConfigBean || archiveConfigBean.getAutoArchiveFlag().intValue() != 1) {
                return new CoreResult(Result.FAILURE, "自动归档未开启");
            }
            SimpleDateFormat sdf = new SimpleDateFormat("yy-MM-dd HH:mm:ss");
            long autoArchiveAgo = archiveConfigBean.getAutoArchiveAgo().intValue();
            long logArchiveAgo = autoArchiveAgo * MS_PER_DAY;
            long currentTime = DateUtils.getCurrentTimeMillis();
            String compareTime = sdf.format(currentTime - logArchiveAgo);
            List<AuditLogDO> auditLogDOS = (List<AuditLogDO>) auditLogDao.autoArchiveList(compareTime);
            if (null == auditLogDOS || auditLogDOS.isEmpty()){
                logger.info("自动归档操作日志 没有可归档的日志");
                archiveMap.put("archiveResult", true);
                archiveMap.put("archiveNum", 0);
                return new CoreResult(Result.SUCCESS, archiveMap);
            }
            List<ArchiveLogDO> archiveLogDOList = generateArchiveLogFromAuditLog(auditLogDOS);
            int archiveResult = archiveLogDao.saveArchiveLogList(archiveLogDOList);
            if (archiveResult != 0) {
                int deleteResult = auditLogDao.deleteAuditLogList(auditLogDOS);
                if(deleteResult != archiveResult){
                    return new CoreResult(Result.FAILURE, "删除操作日志表数据异常");
                }
                logger.info("自动归档 删除从AUDIT_LOG表 归档到ARCHIVE_LOG表 "+ deleteResult + "数据");
                archiveMap.put("archiveResult", true);
                archiveMap.put("archiveNum", archiveResult);
                logger.debug("自动归档结束》》》》》》");
                return new CoreResult(Result.SUCCESS, archiveMap);
            }
            archiveMap.put("archiveResult", false);
            archiveMap.put("archiveNum", auditLogDOS.size());
            logger.debug("自动归档结束》》》》》》");
            return new CoreResult(Result.FAILURE, archiveMap);
        } catch (Exception e){
            logger.error("自动归档操作日志出现异常", e);
            return new CoreResult(Result.FAILURE, "自动归档操作日志出现异常");
        }
    }

    @Override
    public CoreResult getRemindMsg() {
        try {
            Operator operator = OperatorUtil.getOperator();
            if (operator == null || operator.getCurrUser() == null) {
                return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.CANNOT_FIND_CURRENT_LOGIN_ADMIN.code,ErrorEnum.CANNOT_FIND_CURRENT_LOGIN_ADMIN.desc));
            }
            CurrentAdminInfo currUser = operator.getCurrUser();
            List<RoleDO> roleDOS = adminRoleDao.getRoleListByAdminCertId(currUser.getId());
            Map<String, Object> resultInfo = new HashMap<>();

            //判断是否是审计员
            boolean isAuditor = false;
            for (RoleDO roleDO : roleDOS){
                if(Constants.ADMIN_ROLE_AUDIT_2 == roleDO.getType().intValue()){
                    isAuditor = true;
                }
            }

            //首页刷新提醒消息组装（审计员）
            Map<String, Object> msgInfo = new HashMap<>();
            if (isAuditor) {
                CoreResult auditMsgResult = this.getAuditMsgInfo();
                if (auditMsgResult.isSuccess()) {
                    msgInfo.put("auditMsg", auditMsgResult.getInfo());
                }
                CoreResult archiveMsgResult = this.getArchiveMsgInfo();
                if (archiveMsgResult.isSuccess()) {
                    msgInfo.put("archiveMsg", archiveMsgResult.getInfo());
                }
            }
            if (!msgInfo.isEmpty()) {
                resultInfo.put("msg", msgInfo);
            }
            return new CoreResult(Result.SUCCESS, resultInfo);
        } catch (Exception e){
            logger.error("获取提醒消息异常", e);
            return new CoreResult(Result.FAILURE, "获取提醒消息异常");
        }
    }

    @Override
    public CoreResult existOrNot(int id) {
        AuditLogDO auditLogDO;
        try {
            auditLogDO = auditLogDao.get(id);
            if (null == auditLogDO){
                return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.code,ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.desc));
            }
        } catch (Exception e){
            logger.error("判断操作日志实例存在异常", e);
            return new CoreResult(Result.FAILURE,null ,new ErrorBean(ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.code,ErrorEnum.LOG_NOT_EXIST_OR_ARCHIVED.desc));
        }
        return CoreResult.success();
    }
}
