package com.xdja.pki.ra.service.manager.auth;

import com.xdja.pki.auth.service.AuditSignService;
import com.xdja.pki.auth.service.bean.CertInfoDTO;
import com.xdja.pki.ra.core.asn1.DigestObjectIdentifiers;
import com.xdja.pki.ra.core.common.CommonVariable;
import com.xdja.pki.ra.core.commonenum.ErrorEnum;
import com.xdja.pki.ra.core.commonenum.KeyAlgEnum;
import com.xdja.pki.ra.core.common.Result;
import com.xdja.pki.ra.core.constant.Constants;
import com.xdja.pki.ra.core.exception.ServiceException;
import com.xdja.pki.ra.core.util.cert.CertUtils;
import com.xdja.pki.ra.core.util.cert.HsmUtils;
import com.xdja.pki.ra.manager.dao.AdminCertDao;
import com.xdja.pki.ra.manager.dao.model.AdminCertDO;
import com.xdja.pki.ra.manager.sdk.business.CaBusinessManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.servlet.http.HttpServletResponse;
import java.security.PublicKey;
import java.security.cert.X509Certificate;
import java.util.Map;



/**
 * 通用审计验签业务接口实现
 *
 * @author syg
 */
@Service
public class AuditSignServiceImpl implements AuditSignService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());
    /**
     * 客户端与服务器时间偏移量
     */
    @Value("${client.offset.time}")
    private int offsetTime;

    @Autowired
    AdminCertDao adminCertDao;

    @Autowired
    CaBusinessManager caBusinessManager;

    @Override
    public int getOffsetTime() {
        return offsetTime;
    }

    /**
     * 验证签名
     *
     * @param keyAlg 密钥算法，1-RSA；2-SM2
     * @param digestAlg 摘要算法，1-SHA1；2-SM3；3-SHA256
     * @param publicKey 公钥
     * @param sourceData 原文
     * @param signData 签名数据
     * @return true-通过；false-未通过
     */
    @Override
    public boolean verifySign(int keyAlg, int digestAlg, PublicKey publicKey, byte[] sourceData, byte[] signData) {
        boolean verifyResult = false;
        logger.info("keyAlg:"+keyAlg +" digestAlg:"+digestAlg);
        try {
        //    if (CommonVariable.isUseHsm()){
            if(Constants.HSM_SERVER_0 != CommonVariable.getIsHsm()){
                verifyResult = HsmUtils.verifyByYunHsmWithAlgId(digestAlg,keyAlg,publicKey,sourceData,signData);
            }else {
                verifyResult = HsmUtils.verifyByBCWithAlgId(digestAlg,keyAlg,publicKey,sourceData, signData);
            }
        } catch (Exception e) {
            logger.error(" =================== 加密机验签异常{}", e);
        }
        if (!verifyResult) {
            logger.info(" =================== 加密机验签失败");
        }
        logger.info("=========================== "+verifyResult);
        return verifyResult;
    }

    @Override
    public int getSystemKeyAlg() {
        String keyAlgName = CommonVariable.getKeyAlgName();
        if (keyAlgName.equalsIgnoreCase(KeyAlgEnum.RSA.desc)){
            return KeyAlgEnum.RSA.value;
        } else if (keyAlgName.equalsIgnoreCase(KeyAlgEnum.SM2.desc)) {
            return KeyAlgEnum.SM2.value;
        } else if (keyAlgName.equalsIgnoreCase(KeyAlgEnum.NISTP256.desc)) {
            return KeyAlgEnum.NISTP256.value;
        }
        return 0;
    }

    @Override
    public CertInfoDTO getCertBySn(String s, int i) {
        logger.info("当前操作的管理员sn为:"+s);
        AdminCertDO adminCertInfo = adminCertDao.getAdminCertInfo(s, Constants.CERT_TYPE_SIGN_2);
        if (adminCertInfo == null){
            logger.info("本地无该sn：{}对应的证书信息，先请求CA进行验证！",s);
            Result result;
            try {
                result = caBusinessManager.raAdminLoginAuthen(s);
            } catch (Exception e) {
                logger.error("待CA认证管理员接口异常{}",e);
                throw new ServiceException(e);
            }

            Map<String,Object> caAuthenMap = (Map<String, Object>) result.getInfo();
            if (CollectionUtils.isEmpty(caAuthenMap)){
                return null;
            }

            String signCertData = (String) caAuthenMap.get("signCertData");
            X509Certificate signCert = CertUtils.getCertFromStr(signCertData);


            // 判断管理员状态
            int signCertStatus = (int) caAuthenMap.get("signCertStatus");
            CertInfoDTO certInfo = new CertInfoDTO();
            certInfo.setSn(signCert.getSerialNumber().toString(16));
            certInfo.setKeyAlg(signCert.getPublicKey().getAlgorithm().equalsIgnoreCase("RSA") ? 1 : 2);
            certInfo.setSubject(CertUtils.getSubjectByX509Cert(signCert));
            certInfo.setStatus(signCertStatus);
            certInfo.setData(signCertData);
            certInfo.setPublicKey(signCert.getPublicKey());
            return certInfo;
        }

        CertInfoDTO certInfo = new CertInfoDTO();
        certInfo.setSn(adminCertInfo.getCertSn());
        certInfo.setKeyAlg(adminCertInfo.getPublicKeyAlg().equalsIgnoreCase("RSA") ? 1 : 2);
        certInfo.setSubject(adminCertInfo.getCertDn());
        certInfo.setStatus(adminCertInfo.getCertStatus());
        certInfo.setData(adminCertInfo.getCertInfo());
        X509Certificate signCert = CertUtils.getCertFromStr(adminCertInfo.getCertInfo());
        certInfo.setPublicKey(signCert.getPublicKey());

        return certInfo;
    }


    @Override
    public byte[] getDigest(byte[] bytes) {
        byte[] digest = null;
        try {
            //    if (Constants.KEY_ALG_NAME_SM2.equalsIgnoreCase(CommonVariable.getKeyAlgName())){
          //  if(CommonVariable.isUseHsm()){
            if (Constants.HSM_SERVER_0 != CommonVariable.getIsHsm()) {
                if (Constants.SIGN_ALG_NAME_SHA256_WHIT_ECDSA.equalsIgnoreCase(CommonVariable.getSigAlgName()) || Constants.SIGN_ALG_NAME_SHA256_WHIT_RSA.equalsIgnoreCase(CommonVariable.getSigAlgName())) {
                    digest = HsmUtils.digestByYunHsmWithOid(DigestObjectIdentifiers.id_sha256.getId(), bytes);
                } else if(Constants.SIGN_ALG_NAME_SM3_WHIT_SM2.equalsIgnoreCase(CommonVariable.getSigAlgName())){
                    digest = HsmUtils.digestByYunHsmWithOid(DigestObjectIdentifiers.sm3.getId(),bytes);
                }else {
                    digest = HsmUtils.digestByYunHsmWithOid(DigestObjectIdentifiers.id_sha1.getId(),bytes);
                }

            }else {
                if (Constants.SIGN_ALG_NAME_SHA256_WHIT_ECDSA.equalsIgnoreCase(CommonVariable.getSigAlgName()) || Constants.SIGN_ALG_NAME_SHA256_WHIT_RSA.equalsIgnoreCase(CommonVariable.getSigAlgName())) {
                    digest = HsmUtils.digestByBCWithOid(DigestObjectIdentifiers.id_sha256.getId(), bytes);
                }else if(Constants.SIGN_ALG_NAME_SM3_WHIT_SM2.equalsIgnoreCase(CommonVariable.getSigAlgName())){
                    digest = HsmUtils.digestByBCWithOid(DigestObjectIdentifiers.sm3.getId(), bytes);
                }else {
                    digest = HsmUtils.digestByBCWithOid(DigestObjectIdentifiers.id_sha1.getId(),bytes);
                }

            }
        } catch (Exception e) {
            logger.error(" =================== 加密机获取摘要异常{}", e);
        }

        return digest;
    }

    @Override
    public Object getIllegalParamError(HttpServletResponse httpServletResponse) {
        return ErrorEnum.ILLEGAL_REQUEST_PARAMETER.resp(httpServletResponse);
    }

    @Override
    public Object getVerifyTimeError(HttpServletResponse httpServletResponse) {
        return ErrorEnum.CLIENT_TIME_AND_SERVER_DISACCORD.resp(httpServletResponse);
    }

    @Override
    public Object getContentDisaccordError(HttpServletResponse httpServletResponse) {
        return ErrorEnum.REQUEST_BODY_AND_SIGN_BODY_DISACCORD.resp(httpServletResponse);
    }

    @Override
    public Object getIllegalSignAlgError(HttpServletResponse httpServletResponse) {
        return ErrorEnum.NOT_SUPPORTED_SIGN_ALG.resp(httpServletResponse);
    }

    @Override
    public Object getCertNotExistError(HttpServletResponse httpServletResponse) {
        return ErrorEnum.ADMIN_CERT_NOT_EXIST.resp(httpServletResponse);
    }

    @Override
    public Object getCertStatusError(int i, HttpServletResponse httpServletResponse) {
        return ErrorEnum.ADMIN_CERT_STATUS_IS_NOT_NORMAL.resp(httpServletResponse);
    }

    @Override
    public Object getVerifySignFailError(HttpServletResponse httpServletResponse) {
        return ErrorEnum.VERIFY_ADMIN_OPERATOR_SIGN_FAIL.resp(httpServletResponse);
    }

    @Override
    public Object getServerInternalError(HttpServletResponse httpServletResponse) {
        return ErrorEnum.SERVER_INTERNAL_EXCEPTION.resp(httpServletResponse);
    }

}
