package com.xdja.pki.ra.service.manager.baseuser;

import com.xdja.pki.ra.core.commonenum.ErrorEnum;
import com.xdja.pki.ra.core.common.Result;
import com.xdja.pki.ra.core.commonenum.UserTypeEnum;
import com.xdja.pki.ra.core.constant.Constants;
import com.xdja.pki.ra.core.exception.ServiceException;
import com.xdja.pki.ra.manager.dao.*;
import com.xdja.pki.ra.manager.dao.model.BaseUserDO;
import com.xdja.pki.ra.manager.dao.model.CertApplyDO;
import com.xdja.pki.ra.manager.dao.model.ContactUserDO;
import com.xdja.pki.ra.manager.dto.*;
import com.xdja.pki.ra.service.manager.baseuser.bean.*;
import com.xdja.pki.ra.service.manager.certapply.CertApplyService;
import com.xdja.pki.ra.service.manager.deviceuser.DeviceUserService;
import com.xdja.pki.ra.service.manager.deviceuser.bean.DeviceUserInfo;
import com.xdja.pki.ra.service.manager.organuser.OrganUserService;
import com.xdja.pki.ra.service.manager.baseuser.bean.DecryptUserInfo;
import com.xdja.pki.ra.service.manager.baseuser.bean.EncryptUserInfo;
import com.xdja.pki.ra.service.manager.personuser.PersonUserService;
import com.xdja.pki.ra.service.manager.personuser.bean.PersonUserInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * @author wly
 */
@Service
public class BaseUserServiceImpl implements BaseUserService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    BaseUserDao baseUserDao;

    @Autowired
    CertApplyDao certApplyDao;

    @Autowired
    CertApplyService certApplyService;

    @Autowired
    PersonUserService personUserService;

    @Autowired
    OrganUserService organUserService;

    @Autowired
    DeviceUserService deviceUserService;

    @Autowired
    DeviceUserDao deviceUserDao;

    @Autowired
    OrganUserDao organUserDao;

    @Autowired
    PersonUserDao personUserDao;

    @Autowired
    UserCertDao userCertDao;

    @Override
    public Result getAllSystemName() {
        Result result = new Result();
        List<SystemInfo> systemNames ;
        try {
            systemNames = baseUserDao.getAllSystemName();
        } catch (DataAccessException e) {
            systemNames = new ArrayList<>();
        }
        /**
         * 终端设备内置系统名称
         */
        SystemInfo systemInfo = new SystemInfo(Constants.SYSTEM_FLAG_V2X,Constants.SYSTEM_FLAG_V2X);
        systemNames.add(systemInfo);
        result.setInfo(systemNames);
        return result;
    }

    /**
     * 查询base_user表
     *
     * @param baseId
     * @return
     */
    @Override
    public Result getBaseUser(Long baseId) {

        Result result = new Result();

        BaseUserDO baseUserDO;
        try {
            baseUserDO = baseUserDao.getBaseUserInfo(baseId);
        } catch (Exception e) {
            logger.error("获取用户表信息异常");
            result.setError(ErrorEnum.GET_BASE_USER_INFO_IS_EXCEPTION);
            return result;
        }
        if (null == baseUserDO) {
            logger.error("获取base_user用户表信息为空");
            result.setError(ErrorEnum.GET_BASE_USER_INFO_IS_EMPTY);
            return result;
        }
        result.setInfo(baseUserDO);
        return result;
    }

    /**
     * 更新--base_user表 状态
     *
     * @param userId
     * @param status
     * @return
     */
    @Transactional
    @Override
    public Result updateUserStatus(Integer userId, int status, boolean isOnline, String systemFlag) {
        try {
            Result result = new Result();
            baseUserDao.updateUserStatus(userId, status);
            // 针对每一条需要停止流转的申请  记录其对应的操作记录
            List<CertApplyDO> unEndCertApplyList = certApplyDao.getUnEndCertApplyByUserId(userId);
            if (!CollectionUtils.isEmpty(unEndCertApplyList)) {
                for (CertApplyDO certApplyDO : unEndCertApplyList) {
                    certApplyService.insertCertApplyRecord(certApplyDO.getApplyType(), Constants.OPERATE_TYPE_ISSUE_5, certApplyDO.getApplyNo(), systemFlag, Constants.CERT_APPLY_STATUS_ISSUE_FAIL_4, "用户被停用", Constants.CERT_APPLY_OPERATE_TYPE_ISSUE_FAIL_7, false, isOnline);
                }
            }
            // 更新完用户状态后，需要关联修改在流转申请状态
            certApplyDao.updateCertStatusByUserId(userId, Constants.CERT_APPLY_STATUS_ISSUE_FAIL_4);
            return result;
        } catch (Exception e) {
            throw new ServiceException("更新用户状态失败", e);
        }
    }

    /**
     * 更新用户状态信息 启用/停用
     *
     * @param userType
     * @param identType
     * @param licenseNumber
     * @param status
     * @return
     */
    @Override
    public Result updateUserStatus(Integer userType, Integer identType, String licenseNumber, Integer status, String systemFlag) {
        int count = queryUserExist(identType, licenseNumber, userType, systemFlag);
        if (count == 0) {
            logger.error("该用户不存在：userType:[{}],identType:[{}],licenseNumber:[{}],systemFlag:[{}]", userType, identType, licenseNumber, systemFlag);
            return Result.failure(ErrorEnum.GET_PERSON_USER_INFO_IS_EMPTY);
        }

        long userId = queryUserId(userType,systemFlag, identType, licenseNumber);
        Result result = this.updateUserStatus((int) userId, status, true, systemFlag);

        return result;
    }

    /**
     * 接口 用户已存在校验
     * @param licenseType
     * @param licenseNumber
     * @param userType
     * @param systemFlag
     * @return
     */
    @Override
    public int queryUserExist(Integer licenseType, String licenseNumber, Integer userType, String systemFlag) {
        try {
            licenseNumber = EncryptUserInfo.getEncryptString(licenseNumber);
            return baseUserDao.queryUser(licenseType, licenseNumber, userType, systemFlag);
        } catch (Exception e) {
            throw new ServiceException("用户已存在校验异常", e);
        }
    }

    /**
     * open-api 用户注册
     *
     * @param userType
     * @param userInfo
     * @return
     */
    @Override
    public Result userRegister(Integer userType, UserInfo userInfo, String systemFlag) {
        try {
            Result result;
            if (userType == UserTypeEnum.PERSON_USER.id) {
                PersonUserDTO personUserDTO = new PersonUserDTO();
                /**
                 * 数据转换
                 */
                BeanUtils.copyProperties(userInfo.getPersonUser(), personUserDTO);
                personUserDTO.setSystemFlag(systemFlag);
                result = personUserService.savePersonUser(personUserDTO);
            } else if (userType == UserTypeEnum.ORGAN_USER.id) {
                OrganUserInfoDTO organUserDTO = new OrganUserInfoDTO();
                OrganUser organUser = userInfo.getOrganUser();
                PersonUser personInfo = organUser.getUserInfo();
                UserInfoDTO userInfoDTO = new UserInfoDTO();
                /**
                 * 数据转换
                 */
                BeanUtils.copyProperties(organUser, organUserDTO);
                if (null != personInfo) {
                    BeanUtils.copyProperties(personInfo, userInfoDTO);
                }
                organUserDTO.setSystemFlag(systemFlag);
                result = organUserService.saveOrganUser(userInfoDTO, organUserDTO);
            } else {
                com.xdja.pki.ra.service.manager.deviceuser.bean.UserInfo userInfoDTO = new com.xdja.pki.ra.service.manager.deviceuser.bean.UserInfo();
                DeviceUserInfo deviceUserInfo = new DeviceUserInfo();
                DeviceUser deviceUser = userInfo.getDeviceUser();
                PersonUser personInfo = deviceUser.getUserInfo();
                /**
                 * 数据转换
                 */
                BeanUtils.copyProperties(deviceUser, deviceUserInfo);
                if (null != personInfo) {
                    BeanUtils.copyProperties(personInfo, userInfoDTO);
                }
                deviceUserInfo.setSystemFlag(systemFlag);
                result = deviceUserService.registerDeviceUser(deviceUserInfo, userInfoDTO,false);
            }
            return result;
        } catch (BeansException e) {
            throw new ServiceException("open-api 用户注册异常", e);
        }

    }

    /**
     * open-api 用户查询
     *
     * @param userType
     * @param identType
     * @param licenseNumber
     * @return
     */
    @Override
    public Result userQuery(Integer userType, Integer identType, String licenseNumber, String systemFlag) {
        try {
            UserInfo userInfo = new UserInfo();
            Result result;
            if (userType == UserTypeEnum.PERSON_USER.id) {
                result = personUserService.getPersonUserDetailsByLicenseTypeAndLicenseNumber(identType, licenseNumber, systemFlag);
                if (!result.isSuccess()) {
                    return result;
                }
                PersonUserDTO personUserDTO = (PersonUserDTO) result.getInfo();
                PersonUser personUser = new PersonUser();
                BeanUtils.copyProperties(personUserDTO, personUser);
                personUser = DecryptUserInfo.getDecryptPersonUser(personUser);
                userInfo.setPersonUser(personUser);
            } else if (userType == UserTypeEnum.ORGAN_USER.id) {
                result = organUserService.getOrganUser(identType, licenseNumber, systemFlag);
                if (!result.isSuccess()) {
                    return result;
                }
                Map<String, Object> map = (Map<String, Object>) result.getInfo();
                PersonUser personInfo = new PersonUser();
                OrganUser organUser = new OrganUser();
                BeanUtils.copyProperties(map.get("userInfo"), personInfo);
                BeanUtils.copyProperties(map.get("organUserInfo"), organUser);
                /**
                 * 0是int默认值
                 */
                if (null == personInfo.getLicenseType()) {
                    personInfo = null;
                }
                personInfo = DecryptUserInfo.getDecryptPersonUser(personInfo);
                organUser = DecryptUserInfo.getDecryptOrganUser(organUser);
                organUser.setUserInfo(personInfo);
                userInfo.setOrganUser(organUser);
            } else {
                result = deviceUserService.getDevice(identType, licenseNumber, systemFlag);
                if (!result.isSuccess()) {
                    return result;
                }
                PersonUser personInfo = new PersonUser();
                Map<String, Object> map = (Map<String, Object>) result.getInfo();
                DeviceUser deviceUser = new DeviceUser();
                DeviceUserDetailsDTO deviceUserDetailsDTO = (DeviceUserDetailsDTO) map.get("deviceUser");
                BeanUtils.copyProperties(deviceUserDetailsDTO, deviceUser);
                deviceUser.setUserName(deviceUserDetailsDTO.getDeviceName());
                deviceUser.setLicenseNumber(deviceUserDetailsDTO.getLicenseNumber());
                BeanUtils.copyProperties(map.get("userInfo") , personInfo);

                /**
                 * 0是int默认值
                 */
                if (personInfo.getLicenseType() == 0) {
                    personInfo = null;
                }
                personInfo = DecryptUserInfo.getDecryptPersonUser(personInfo);
                deviceUser = DecryptUserInfo.getDecryptDeviceUser(deviceUser);
                deviceUser.setUserInfo(personInfo);
                userInfo.setDeviceUser(deviceUser);
            }
            result.setInfo(userInfo);
            return result;
        } catch (Exception e) {
            throw new ServiceException("open-api 用户查询", e);
        }
    }

    /**
     * 根据签名证书sn查询用户信息
     *
     * @param sn
     * @return
     */
    @Override
    public Result userQuery(String sn, String systemFlag) {
        Result result = new Result();
        BaseUserDO baseUserDO = null;
        Long id = userCertDao.getUserId(sn);
        if(null != id){
            baseUserDO = baseUserDao.getBaseUserInfo(id, systemFlag);
        }
        if (null == baseUserDO || null == id) {
            logger.error("根据签名证书sn查询用户信息查询用户不存在");
            result.setError(ErrorEnum.THE_USER_IS_NOT_EXIT);
            return result;
        }
        String licenseNumberDecrypt = null;
        try {
            licenseNumberDecrypt = DecryptUserInfo.getDecryptString(baseUserDO.getLicenseNumber());
        } catch (Exception e) {
            throw new ServiceException("open-api 用户查询", e);
        }
        return userQuery(baseUserDO.getUserType(), baseUserDO.getLicenseType(), licenseNumberDecrypt, systemFlag);
    }

    /**
     * 查询用户ID
     *
     * @param userType
     * @param systemFlag
     * @param licenseType
     * @param licenseNumber
     * @return
     */
    @Override
    public Long queryUserId(int userType, String systemFlag, Integer licenseType, String licenseNumber) {
        try {
            licenseNumber = EncryptUserInfo.getEncryptString(licenseNumber);
            return baseUserDao.getUserId(userType, systemFlag, licenseType, licenseNumber);
        } catch (Exception e) {
            throw new ServiceException("查询用户ID失败", e);
        }
    }

    /**
     * open-api 用户更新
     *
     * @param userType
     * @param identType
     * @param licenseNumber
     * @return
     */
    @Override
    public Result userUpdate(Integer userType, Integer identType, String licenseNumber, UserInfo userInfo, String systemFlag) {
        try {
            Result result;
            if (userType == UserTypeEnum.PERSON_USER.id) {
                Long userId = queryUserId(UserTypeEnum.PERSON_USER.id, systemFlag, identType, licenseNumber);
                if (null == userId) {
                    return Result.failure(ErrorEnum.THE_USER_IS_NOT_EXIT);
                }
                PersonUserInfo personUserInfo = new PersonUserInfo();
                /**
                 * 数据转换
                 */
                PersonUser personUser = userInfo.getPersonUser();
                BeanUtils.copyProperties(personUser, personUserInfo);
                personUserInfo.setSystemFlag(systemFlag);
                if(null == personUserInfo.getLicenseType()){
                    personUserInfo.setLicenseType(identType);
                }
                if(null == personUserInfo.getLicenseNumber()){
                    personUserInfo.setLicenseNumber(licenseNumber);
                }
                result = personUserService.updatePersonUser(userId.intValue(), personUserInfo);
            } else if (userType == UserTypeEnum.ORGAN_USER.id) {
                Long userId = queryUserId(UserTypeEnum.ORGAN_USER.id, systemFlag, identType, licenseNumber);
                if (null == userId) {
                    return Result.failure(ErrorEnum.THE_USER_IS_NOT_EXIT);
                }
                OrganUserInfoDTO organUserInfoDTO = new OrganUserInfoDTO();
                UserInfoDTO userInfoDTO = new UserInfoDTO();
                OrganUser organUser = userInfo.getOrganUser();
                PersonUser personInfo = organUser.getUserInfo();
                /**
                 * 数据转换
                 */
                BeanUtils.copyProperties(organUser, organUserInfoDTO);
                if (null != personInfo) {
                    BeanUtils.copyProperties(personInfo, userInfoDTO);
                }
                organUserInfoDTO.setSystemFlag(systemFlag);
                if(null == organUserInfoDTO.getLicenseType()){
                    organUserInfoDTO.setLicenseType(identType);
                }
                if(null == organUserInfoDTO.getLicenseNumber()){
                    organUserInfoDTO.setLicenseNumber(licenseNumber);
                }
                result = organUserService.updateOrganUserInfo(userId, organUserInfoDTO, userInfoDTO);
            } else {
                DeviceUserInfo deviceUserInfo = new DeviceUserInfo();
                com.xdja.pki.ra.service.manager.deviceuser.bean.UserInfo userInfoDTO = new com.xdja.pki.ra.service.manager.deviceuser.bean.UserInfo();
                DeviceUser deviceUser = userInfo.getDeviceUser();
                PersonUser personInfo = deviceUser.getUserInfo();
                Long userId = queryUserId(UserTypeEnum.DEVICE_USER.id, systemFlag, identType, licenseNumber);
                if (null == userId) {
                    return Result.failure(ErrorEnum.THE_USER_IS_NOT_EXIT);
                }
                /**
                 * 数据转换
                 */
                BeanUtils.copyProperties(deviceUser, deviceUserInfo);
                if (null != personInfo) {
                    BeanUtils.copyProperties(personInfo, userInfo);
                }
                deviceUserInfo.setSystemFlag(systemFlag);
                if(null == deviceUserInfo.getLicenseType()){
                    deviceUserInfo.setLicenseType(identType);
                }
                if(null == deviceUserInfo.getLicenseNumber()){
                    deviceUserInfo.setLicenseNumber(licenseNumber);
                }
                result = deviceUserService.updateDeviceUser(userId, deviceUserInfo, userInfoDTO);
            }
            return result;
        } catch (BeansException e) {
            throw new ServiceException("open-api 用户查询", e);
        }
    }


}
