package com.xdja.pki.ra.service.manager.baseuser;

import com.xdja.pki.ra.core.commonenum.ErrorEnum;
import com.xdja.pki.ra.core.common.Result;
import com.xdja.pki.ra.core.commonenum.UserStatusEnum;
import com.xdja.pki.ra.core.commonenum.UserTypeEnum;
import com.xdja.pki.ra.core.constant.Constants;
import com.xdja.pki.ra.core.constant.PathConstants;
import com.xdja.pki.ra.manager.dao.BaseUserDao;
import com.xdja.pki.ra.manager.dao.ContactUserDao;
import com.xdja.pki.ra.manager.dao.CustomerDao;
import com.xdja.pki.ra.manager.dao.OrganUserDao;
import com.xdja.pki.ra.manager.dao.model.*;
import com.xdja.pki.ra.manager.dto.OrganUserDTO;
import com.xdja.pki.ra.manager.dto.OrganUserInfoDTO;
import com.xdja.pki.ra.manager.dto.UserInfoDTO;
import com.xdja.pki.ra.manager.page.PageInfo;
import com.xdja.pki.ra.service.manager.baseuser.bean.DecryptUserInfo;
import com.xdja.pki.ra.service.manager.baseuser.bean.EncryptUserInfo;
import com.xdja.pki.ra.service.manager.customer.CustomerSysService;
import com.xdja.pki.ra.service.manager.organuser.*;
import com.xdja.pki.ra.service.manager.organuser.bean.*;
import com.xdja.pki.ra.core.util.file.ErrorExcel;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.io.IOException;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 机构用户服务层-实现类
 *
 * @author syg
 */
@Service
public class OrganUserServiceImpl implements OrganUserService {
    private Logger logger = LoggerFactory.getLogger(this.getClass());
    @Autowired
    OrganUserDao organUserDao;
    @Autowired
    BaseUserDao baseUserDao;
    @Autowired
    ContactUserDao contactUserDao;
    @Autowired
    CustomerDao customerDao;
    @Autowired
    CustomerSysService customerSysService;
    @Autowired
    BaseUserService baseUserService;

    /**
     * 分页查询
     *
     * @param organName
     * @param licenseNumber
     * @param status
     * @param pageNo
     * @param pageSize
     * @return
     */
    @Override
    public Result listOrganUser(String organName, String licenseNumber, Integer status, Integer pageNo, Integer pageSize) {
        OrganUserResp organUserResp = new OrganUserResp();
        Result result = new Result();
        try {
            String licenseNumberEncrypt = EncryptUserInfo.getEncryptString(licenseNumber);
            PageInfo<OrganUserDTO> organUserDOPageInfo = organUserDao.listPageOrganUser(organName, licenseNumberEncrypt, status, pageNo, pageSize);
            if (organUserDOPageInfo == null) {
                result.setError(ErrorEnum.QUERY_ORGAN_USER_LIST_ERROR);
                return result;
            }
            List<OrganUserVO> organUserVOList = new ArrayList<>();

            List<OrganUserDTO> organUserDOList = organUserDOPageInfo.getList();
            if (CollectionUtils.isEmpty(organUserDOList)) {
                organUserResp.setRecordCount(0);
                organUserResp.setPageCount(0);
                organUserResp.setDatas(organUserVOList);
                result.setInfo(organUserResp);
                return result;
            }

            for (OrganUserDTO organUserDTO : organUserDOList) {
                OrganUserVO organUserVO = new OrganUserVO();
                BeanUtils.copyProperties(organUserDTO, organUserVO);
                Timestamp gmtCreate = organUserDTO.getGmtCreate();
                organUserVO.setGmtCreate(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(gmtCreate));
                /**
                 * 解密数据 DO 20200817
                 */
                organUserVO = DecryptUserInfo.getDecryptOrganUserVO(organUserVO);
                organUserVOList.add(organUserVO);
            }
            organUserResp.setDatas(organUserVOList);
            organUserResp.setPageCount(organUserDOPageInfo.getPageCount());
            organUserResp.setRecordCount(organUserDOPageInfo.getRecordCount());
            result.setInfo(organUserResp);
        } catch (Exception e) {
            result.setError(ErrorEnum.DECRYPT_ENCRYPT_INFO_ERROR);
            return result;
        }
        return result;
    }

    /**
     * 机构用户注册
     *
     * @param userInfoDTO
     * @param organUserInfoDTO
     * @return
     */
    @Transactional
    @Override
    public Result saveOrganUser(UserInfoDTO userInfoDTO, OrganUserInfoDTO organUserInfoDTO) {
        Result result = new Result();
        try {
            /**
             * 判断系统标识是否存在
             */
            result = customerSysService.verifyUserSystemFlag(organUserInfoDTO.getSystemFlag());
            if (!result.isSuccess()) {
                return result;
            }
            //base_user
            BaseUserDO baseUser = new BaseUserDO();
            Date date = new Date(System.currentTimeMillis());
            Timestamp time = new Timestamp(date.getTime());
            baseUser.setUserType(Constants.USER_TYPE_ORGAN_2);
            baseUser.setUserName(organUserInfoDTO.getOrganName());
            int queryOrganUser = baseUserService.queryUserExist(organUserInfoDTO.getLicenseType(), organUserInfoDTO.getLicenseNumber(), UserTypeEnum.ORGAN_USER.id, organUserInfoDTO.getSystemFlag());
            if (queryOrganUser > 0) {
                logger.info("机构用户信息已注册");
                result.setError(ErrorEnum.QUERY_BASE_USER_IS_EXISTENCE);
                return result;
            }
            baseUser.setLicenseNumber(organUserInfoDTO.getLicenseNumber());
            baseUser.setLicenseType(organUserInfoDTO.getLicenseType());
            baseUser.setGmtUpdate(time);
            baseUser.setGmtCreate(time);
            baseUser.setStatus(Constants.USER_STATUS_NORMAL_0);
            baseUser.setSystemFlag(organUserInfoDTO.getSystemFlag());
            baseUser = EncryptUserInfo.getEncryptBaseUserDO(baseUser);
            baseUserDao.saveBaseUserInfo(baseUser);
            //person_user
            ContactUserDO contactUser = new ContactUserDO();
            BeanUtils.copyProperties(userInfoDTO, contactUser);
            contactUser.setUserId(baseUser.getId());
            contactUser.setGmtUpdate(time);
            contactUser.setGmtCreate(time);
            if (null == contactUser.getSex()) {
                contactUser.setSex(0);
            }
            contactUser = EncryptUserInfo.getEncryptContactUserDO(contactUser);
            contactUserDao.saveContactUserInfo(contactUser);
            //organ_user
            OrganUserDO organUser = new OrganUserDO();
            BeanUtils.copyProperties(organUserInfoDTO, organUser);
            organUser.setPersonId(contactUser.getId());
            organUser.setUserId(baseUser.getId());
            organUser.setGmtCreate(time);
            organUser.setGmtUpdate(time);
            organUser = EncryptUserInfo.getEncryptOrganUserDO(organUser);
            OrganUserDO addOrganUserDO = organUserDao.saveOrganUserInfo(organUser);

            //审计日志
            result.setLogContent(", 用户ID=" + addOrganUserDO.getId());
        } catch (Exception e) {
            result.setError(ErrorEnum.DECRYPT_ENCRYPT_INFO_ERROR);
            return result;
        }
        return result;
    }

    /**
     * 机构用户批量注册
     *
     * @param userInfoDTO
     * @param organUserInfoDTO
     * @return
     */
    @Override
    public Result batchSaveOrganUser(UserInfoDTO userInfoDTO, OrganUserInfoDTO organUserInfoDTO) {
        Result result = new Result();
        try {
            // 获取当前系统时间
            Date date = new Date();
            Timestamp s = new Timestamp(date.getTime());
            //base_user
            BaseUserDO baseUser = new BaseUserDO();
            baseUser.setUserType(Constants.USER_TYPE_ORGAN_2);
            baseUser.setUserName(organUserInfoDTO.getOrganName());
            baseUser.setGmtCreate(s);
            baseUser.setGmtUpdate(s);
            baseUser.setStatus(Constants.USER_STATUS_NORMAL_0);
            baseUser.setLicenseNumber(organUserInfoDTO.getLicenseNumber());
            baseUser.setLicenseType(organUserInfoDTO.getLicenseType());
            baseUser.setSystemFlag(organUserInfoDTO.getSystemFlag());
            baseUser = EncryptUserInfo.getEncryptBaseUserDO(baseUser);
            baseUserDao.saveBaseUserInfo(baseUser);
            //person_user
            ContactUserDO contactUser = new ContactUserDO();
            contactUser.setUserId(baseUser.getId());
            contactUser.setPersonName(userInfoDTO.getPersonName());
            contactUser.setSex(userInfoDTO.getSex());
            contactUser.setLicenseType(userInfoDTO.getLicenseType());
            contactUser.setLicenseNumber(userInfoDTO.getLicenseNumber());
            contactUser.setTelNumber(userInfoDTO.getTelNumber());
            contactUser.setAddress(userInfoDTO.getAddress());
            contactUser.setEmail(userInfoDTO.getEmail());
            contactUser.setPostalCode(userInfoDTO.getPostalCode());
            contactUser.setRemark(userInfoDTO.getRemark());
            contactUser.setGmtCreate(s);
            contactUser.setGmtUpdate(s);
            if (null == contactUser.getSex()) {
                contactUser.setSex(0);
            }
            contactUser = EncryptUserInfo.getEncryptContactUserDO(contactUser);
            contactUserDao.saveContactUserInfo(contactUser);
            //organ_user
            OrganUserDO organUser = new OrganUserDO();
            organUser.setPersonId(contactUser.getId());
            organUser.setUserId(baseUser.getId());
            organUser.setOrganName(organUserInfoDTO.getOrganName());
            organUser.setLegalPerson(organUserInfoDTO.getLegalPerson());
            organUser.setLicenseType(organUserInfoDTO.getLicenseType());
            organUser.setLicenseNumber(organUserInfoDTO.getLicenseNumber());
            organUser.setAddress(organUserInfoDTO.getAddress());
            organUser.setEmail(organUserInfoDTO.getEmail());
            organUser.setPostalCode(organUserInfoDTO.getPostalCode());
            organUser.setGmtCreate(s);
            organUser.setGmtUpdate(s);
            organUser = EncryptUserInfo.getEncryptOrganUserDO(organUser);
            organUserDao.saveOrganUserInfo(organUser);
        } catch (Exception e) {
            result.setError(ErrorEnum.DECRYPT_ENCRYPT_INFO_ERROR);
            return result;
        }
        return result;
    }

    /**
     * 机构用户详情页面信息获取
     *
     * @param organId
     * @return
     */
    @Override
    public Result getOrganUserInfo(Long organId) {
        Result result = new Result();
        OrganUserVO organUserVO = new OrganUserVO();
        OrganUserDTO organUserDTO;
        try {

            organUserDTO = organUserDao.getOrganUserInfo(organId);
        } catch (Exception e) {
            logger.info("查询机构用户详情信息异常");
            result.setError(ErrorEnum.GET_ORGAN_USER_INFO_EXCEPTION);
            return result;
        }
        if (null == organUserDTO) {
            logger.info("获取机构用户详情信息为空");
            result.setError(ErrorEnum.GET_ORGAN_USER_IS_EMPTY);
            return result;
        }
        String systemName = customerDao.getSystemNameByFlag(organUserDTO.getSystemFlag());

        BeanUtils.copyProperties(organUserDTO, organUserVO);
        organUserVO.setSystemName(systemName);
        Timestamp gmtCreate = organUserDTO.getGmtCreate();
        organUserVO.setGmtCreate(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(gmtCreate));
        try {
            organUserVO = DecryptUserInfo.getDecryptOrganUserVO(organUserVO);
        } catch (Exception e) {
            result.setError(ErrorEnum.DECRYPT_ENCRYPT_INFO_ERROR);
            return result;
        }
        result.setInfo(organUserVO);
        return result;
    }


    /**
     * 根据机构类型和设备编号查询机构用户信息
     *
     * @param identType
     * @param licenseNumber
     * @return
     */
    @Override
    public Result getOrganUser(Integer identType, String licenseNumber, String systemFlag) {
        Result result = new Result();
        Map<String, Object> map = new HashMap<>();

        OrganUserDTO organUserDTO = null;
        try {
            organUserDTO = organUserDao.getOrganUserInfo(identType, EncryptUserInfo.getEncryptString(licenseNumber), systemFlag);
        } catch (Exception e) {
            result.setError(ErrorEnum.DECRYPT_ENCRYPT_INFO_ERROR);
            return result;
        }
        if (null == organUserDTO) {
            logger.info("获取机构用户表信息为空");
            return Result.failure(ErrorEnum.GET_ORGAN_USER_TABLE_IS_EMPTY);
        }
        String systemName = customerDao.getSystemNameByFlag(systemFlag);
        organUserDTO.setSystemName(systemName);
        ContactUserDO contactUserDO = contactUserDao.getContactUser(organUserDTO.getPersonId());
        map.put("organUserInfo", organUserDTO);
        map.put("userInfo", contactUserDO);
        result.setInfo(map);
        return result;
    }

    /**
     * 修改机构表详情
     *
     * @param organId
     * @param updateOrganUserInfoDTO
     * @param updateUserInfoDTO
     * @return
     */
    @Transactional
    @Override
    public Result updateOrganUserInfo(Long organId, OrganUserInfoDTO updateOrganUserInfoDTO, UserInfoDTO updateUserInfoDTO) {
        Result result = new Result();
        try {
            OrganUserDO organUserInfo = organUserDao.getOrganUser(organId);
            if (organUserInfo == null) {
                logger.info("查询机构用户信息为空");
                result.setError(ErrorEnum.GET_ORGAN_USER_TABLE_IS_EMPTY);
                return result;
            }
            /**
             * 判断系统标识是否存在
             */
            result = customerSysService.verifyUserSystemFlag(updateOrganUserInfoDTO.getSystemFlag());
            if (!result.isSuccess()) {
                return result;
            }
            //更新organ_user表
            String licenseNumberEncrypt = EncryptUserInfo.getEncryptString(updateOrganUserInfoDTO.getLicenseNumber());
            OrganUserDO organUserDO = organUserDao.getOrganUser(updateOrganUserInfoDTO.getLicenseType(), licenseNumberEncrypt, updateOrganUserInfoDTO.getSystemFlag());
            if (null != organUserDO && organUserDO.getUserId().intValue() != organId.intValue()) {
                logger.info("机构用户信息已注册");
                result.setError(ErrorEnum.QUERY_BASE_USER_IS_EXISTENCE);
                return result;
            }
            organUserInfo.setOrganName(updateOrganUserInfoDTO.getOrganName());
            organUserInfo.setLegalPerson(updateOrganUserInfoDTO.getLegalPerson());
            organUserInfo.setLicenseType(updateOrganUserInfoDTO.getLicenseType());
            organUserInfo.setLicenseNumber(updateOrganUserInfoDTO.getLicenseNumber());
            organUserInfo.setAddress(updateOrganUserInfoDTO.getAddress());
            organUserInfo.setEmail(updateOrganUserInfoDTO.getEmail());
            organUserInfo.setPostalCode(updateOrganUserInfoDTO.getPostalCode());

            // 获取当前系统时间
            Date date = new Date();
            Timestamp s = new Timestamp(date.getTime());
            organUserInfo.setGmtUpdate(s);

            organUserInfo = EncryptUserInfo.getEncryptOrganUserDO(organUserInfo);
            int updateOrganUserDO = organUserDao.updateOrganUserInfo(organUserInfo);
            if (updateOrganUserDO != 1) {
                logger.info("更新机构用户信息失败");
                result.setError(ErrorEnum.UPDATE_ORGAN_USER_INFO_ERROR);
                return result;
            }
            //contact_user表
            ContactUserDO contactUserInfo = contactUserDao.getContactUser(organUserInfo.getPersonId());
            if (contactUserInfo == null) {
                logger.info("查询联系人信息为空");
                result.setError(ErrorEnum.GET_CONTACT_USER_INFO_IS_EMPTY);
                return result;
            }
            contactUserInfo.setPersonName(updateUserInfoDTO.getPersonName());
            contactUserInfo.setSex(updateUserInfoDTO.getSex());
            contactUserInfo.setLicenseType(updateUserInfoDTO.getLicenseType());
            contactUserInfo.setTelNumber(updateUserInfoDTO.getTelNumber());
            contactUserInfo.setLicenseNumber(updateUserInfoDTO.getLicenseNumber());
            contactUserInfo.setAddress(updateUserInfoDTO.getAddress());
            contactUserInfo.setEmail(updateUserInfoDTO.getEmail());
            contactUserInfo.setPostalCode(updateUserInfoDTO.getPostalCode());
            contactUserInfo.setRemark(updateUserInfoDTO.getRemark());
            contactUserInfo.setGmtUpdate(s);
            if (null == contactUserInfo.getSex()) {
                contactUserInfo.setSex(0);
            }
            contactUserInfo = EncryptUserInfo.getEncryptContactUserDO(contactUserInfo);
            int updatePersonUserDO = contactUserDao.updateContactUserInfo(contactUserInfo);
            if (updatePersonUserDO != 1) {
                logger.info("更新机构用户联系人信息失败");
                result.setError(ErrorEnum.UPDATE_ORGAN_USER_INFO_ERROR);
                return result;
            }
            BaseUserDO baseUser = baseUserDao.getBaseUserInfo(organUserInfo.getUserId());
            if (baseUser == null) {
                logger.info("查询用户表信息为空");
                result.setError(ErrorEnum.GET_BASE_USER_INFO_IS_EMPTY);
                return result;
            }
            if (baseUser.getStatus() == UserStatusEnum.DISABLED.id) {
                result.setError(ErrorEnum.USER_STATUS_TYPE_IS_NOT_SUPPORT);
                return result;
            }
            baseUser.setUserName(organUserInfo.getOrganName());
            baseUser.setLicenseType(organUserInfo.getLicenseType());
            baseUser.setLicenseNumber(organUserInfo.getLicenseNumber());
            baseUser.setGmtUpdate(s);
            int updateBaseUserDO = baseUserDao.updateBaseUser(baseUser);
            if (updateBaseUserDO != 1) {
                logger.info("更新机构用户信息失败");
                result.setError(ErrorEnum.UPDATE_ORGAN_USER_INFO_ERROR);
                return result;
            }
        } catch (Exception e) {
            result.setError(ErrorEnum.DECRYPT_ENCRYPT_INFO_ERROR);
            return result;
        }

        return result;
    }

    /**
     * 批量注册
     *
     * @param fileList
     * @return
     */
    @Transactional
    @Override
    public Result batchSaveOrganUserByTemplateFile(ArrayList<ArrayList<String>> fileList) {
        Result result = new Result();
        int excelSize = fileList.size();
        ArrayList<ArrayList<String>> trueList = new ArrayList<>();
        ArrayList<ArrayList<String>> falseList = new ArrayList<>();
        //表格数据格式校验
        for (int i = 0; i < fileList.size(); i++) {
            Result organResult = new OrganParamsCheck().checkOrganParams(fileList.get(i));
            if (organResult.getCode() == 0) {
                //格式正确
                trueList.add((ArrayList<String>) organResult.getInfo());
            } else {
                //格式错误
                falseList.add((ArrayList<String>) organResult.getInfo());
            }
        }
        //infoList存放校验后的正确数据
        ArrayList<SaveOrganUserVO> infoList = new ArrayList<SaveOrganUserVO>();
        //list转实体类+本地和数据库校验唯一性
        HashSet hashSet = new HashSet();
        //处理错误信息暂存地
        ArrayList list;
        //遍历格式正确的
        for (int i = 0; i < trueList.size(); i++) {

            SaveOrganUserVO info = ToOrganUserVO.format(trueList.get(i));
            OrganUserInfoDTO organUser = info.getOrganUserInfoDTO();
            String licenseNumber = organUser.getLicenseNumber();
            Integer licenseType = organUser.getLicenseType();
            String s = licenseNumber + licenseType.toString();
            //系统标识校验 DO 0318

            String systemName = organUser.getSystemName();
            int systemFlagSize = 1; // 系统是否存在
            String systemFlag = null;
            if (StringUtils.isBlank(systemName) || Constants.SYSTEM_FLAG_DEFAULT.equals(systemName)) {
                systemFlag = Constants.SYSTEM_FLAG_DEFAULT;
            } else {
                if (Constants.SYSTEM_FLAG_V2X.equals(systemName)) {
                    systemFlag = Constants.SYSTEM_FLAG_V2X;
                } else {
                    systemFlagSize = baseUserDao.querySystemName(systemName);
                    if (systemFlagSize != 0) {
                        //系统标识一定有
                        systemFlag = baseUserDao.getSystemFlagByName(systemName);
                    }
                }
                //判断本地上传的用户是否重复加上系统名称标识
                s = s + systemFlag;
            }
            organUser.setSystemFlag(systemFlag);
            int queryOrganUser = 0;
            if (null != systemFlag) {
                try {
                    queryOrganUser = baseUserService.queryUserExist(licenseType, licenseNumber, UserTypeEnum.ORGAN_USER.id, systemFlag);
                } catch (Exception e) {
                    logger.info("查询已注册机构用户信息异常");
                    result.setError(ErrorEnum.GET_ORGAN_USER_INFO_EXCEPTION);
                    return result;
                }
            }

            //拿出机构信息的licenseType+licenseNumber放入HashSet,判断本地是否有重复,没有重复放到集合测数据库唯一性
            if (!hashSet.contains(s)) {
                hashSet.add(s);
                if (queryOrganUser == 0 && systemFlagSize == 1) {
                    infoList.add(info);
                } else {
                    list = trueList.get(i);
                    falseList.add(list);
                    if (queryOrganUser > 0 && systemFlagSize != 1) {
                        list.add("机构用户证件号码已存在，所属系统不存在");
                    } else if (queryOrganUser > 0) {
                        list.add("机构用户证件号码已存在");
                    } else {
                        list.add("所属系统不存在");
                    }
                }

            } else {
                list = trueList.get(i);
                list.add("本地上传机构用户重复");
                falseList.add(list);
            }
        }
       /* //校验错误打印
        for (int i = 0; i < falseList.size(); i++) {
            logger.info("校验错误输出:" + falseList.get(i).toString());
        }*/
        //成功注册
        for (int i = 0; i < infoList.size(); i++) {
            //插入数据库
            batchSaveOrganUser(infoList.get(i).getUserInfoDTO(), infoList.get(i).getOrganUserInfoDTO());
        }
        //错误写
        //错误数据文件名
        String filename;
        if (excelSize != infoList.size()) {
            //错误数据下载
            Result name = null;
            try {
                name = ErrorExcel.generateExcel(PathConstants.BATCH_REGISTER_FILE_PATH, Constants.ORGAN_1_ERROR, falseList);
            } catch (IOException e) {
                result.setError(ErrorEnum.UPDATE_ERROR_EXCEL_IS_ERROR);
                logger.info("生成错误信息文件失败：", e);
                return result;
            }
            filename = (String) name.getInfo();
        } else {

            filename = null;
        }
        //构造前端返回值
        BatchResp batchResp = new BatchResp(infoList.size(), falseList.size(), filename);
        //审计日志
        result.setLogContent("，成功条数=" + infoList.size() + "，失败条数=" + falseList.size());
        result.setInfo(batchResp);
        return result;
    }
}