package com.xdja.pki.ra.service.manager.baseuser;

import com.xdja.pki.ra.core.commonenum.ErrorEnum;
import com.xdja.pki.ra.core.common.Result;
import com.xdja.pki.ra.core.commonenum.UserStatusEnum;
import com.xdja.pki.ra.core.commonenum.UserTypeEnum;
import com.xdja.pki.ra.core.constant.Constants;
import com.xdja.pki.ra.core.constant.PathConstants;
import com.xdja.pki.ra.core.exception.ServiceException;
import com.xdja.pki.ra.manager.dao.BaseUserDao;
import com.xdja.pki.ra.manager.dao.CustomerDao;
import com.xdja.pki.ra.manager.dao.PersonUserDao;
import com.xdja.pki.ra.manager.dao.model.BaseUserDO;
import com.xdja.pki.ra.manager.dao.model.PersonUserDO;
import com.xdja.pki.ra.manager.dto.PersonUserDTO;
import com.xdja.pki.ra.manager.dto.PersonUserDetailsDTO;
import com.xdja.pki.ra.manager.page.PageInfo;
import com.xdja.pki.ra.service.manager.customer.CustomerSysService;
import com.xdja.pki.ra.service.manager.baseuser.bean.DecryptUserInfo;
import com.xdja.pki.ra.service.manager.baseuser.bean.EncryptUserInfo;
import com.xdja.pki.ra.service.manager.personuser.bean.PersonUserInfo;
import com.xdja.pki.ra.service.manager.personuser.bean.PersonUserResp;
import com.xdja.pki.ra.service.manager.personuser.bean.PersonUserVO;
import com.xdja.pki.ra.service.manager.personuser.bean.SavePersonUserVO;
import com.xdja.pki.ra.service.manager.organuser.bean.BatchResp;
import com.xdja.pki.ra.core.util.file.ErrorExcel;
import com.xdja.pki.ra.service.manager.personuser.bean.PersonParamsCheck;
import com.xdja.pki.ra.service.manager.personuser.bean.ToPersonUserVO;
import com.xdja.pki.ra.service.manager.personuser.PersonUserService;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;
import org.springframework.util.CollectionUtils;

import java.io.IOException;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;


/**
 * 个人用户服务层-实现类
 *
 * @author syg
 */
@Service
public class PersonUserServiceImpl implements PersonUserService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    PersonUserDao personUserDao;

    @Autowired
    BaseUserDao baseUserDao;

    @Autowired
    CustomerDao customerDao;

    @Autowired
    CustomerSysService customerSysService;

    @Autowired
    BaseUserService baseUserService;


    /**
     * 分页查询用户列表
     *
     * @param personName
     * @param licenseNumber 查询适配加密数据 改为全匹配查询 20200814
     * @param status
     * @param pageIndex
     * @param pageSize
     * @return
     */
    @Override
    public Result listPersonUser(String personName, String licenseNumber, int status, int pageIndex, int pageSize) {
        Result result = new Result();
        try {
            PersonUserResp personUserResp = new PersonUserResp();
            String licenseNumberEncrypt = EncryptUserInfo.getEncryptString(licenseNumber);
            PageInfo<PersonUserDTO> personUserDTOPageInfo = personUserDao.listPagePersonUser(personName, licenseNumberEncrypt, status, pageIndex, pageSize);
            if (personUserDTOPageInfo == null) {
                result.setError(ErrorEnum.QUERY_PERSON_USER_LIST_EMPTY);
                return result;
            }
            List<PersonUserVO> personUserVOList = new ArrayList<>();
            List<PersonUserDTO> personUserDTOList = personUserDTOPageInfo.getList();
            if (CollectionUtils.isEmpty(personUserDTOList)) {
                personUserResp.setRecordCount(0);
                personUserResp.setPageCount(0);
                personUserResp.setDatas(personUserVOList);
                result.setInfo(personUserResp);
                return result;
            }
            for (PersonUserDTO personUserDTO : personUserDTOList) {
                PersonUserVO personUserVO = new PersonUserVO();
                BeanUtils.copyProperties(personUserDTO, personUserVO);
                Timestamp gmtCreate = personUserDTO.getGmtCreate();
                personUserVO.setGmtCreate(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(gmtCreate));
                personUserVO = DecryptUserInfo.getDecryptPersonUserVO(personUserVO);
                personUserVOList.add(personUserVO);
            }
            personUserResp.setDatas(personUserVOList);
            personUserResp.setPageCount(personUserDTOPageInfo.getPageCount());
            personUserResp.setRecordCount(personUserDTOPageInfo.getRecordCount());
            result.setInfo(personUserResp);
        } catch (Exception e) {
            e.printStackTrace();
            result.setError(ErrorEnum.DECRYPT_ENCRYPT_INFO_ERROR);
            return result;
        }
        return result;
    }

    @Override
    public Result getPersonUserDetails(long personId) {
        Result result = new Result();
        try {
            PersonUserDetailsDTO personUserDetailsDTO = null;
            try {
                personUserDetailsDTO = personUserDao.getPersonDetailsInfoById(personId);
                //不存在抛此异常
            } catch (EmptyResultDataAccessException e) {
                result.setError(ErrorEnum.GET_PERSON_USER_INFO_IS_EMPTY);
                return result;
            }
            String systemName = customerDao.getSystemNameByFlag(personUserDetailsDTO.getSystemFlag());
            PersonUserVO personUserVO = new PersonUserVO();
            BeanUtils.copyProperties(personUserDetailsDTO, personUserVO);
            personUserVO.setSystemName(systemName);
            Timestamp gmtCreate = personUserDetailsDTO.getGmtCreate();
            personUserVO.setGmtCreate(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(gmtCreate));
            personUserVO = DecryptUserInfo.getDecryptPersonUserVO(personUserVO);
            result.setInfo(personUserVO);
        } catch (Exception e) {
            e.printStackTrace();
            result.setError(ErrorEnum.DECRYPT_ENCRYPT_INFO_ERROR);
            return result;
        }
        return result;
    }

    /**
     * 通过证件类型和证件号码查询个人用户详情
     *
     * @param identType
     * @param licenseNumber
     * @return
     */
    @Override
    public Result getPersonUserDetailsByLicenseTypeAndLicenseNumber(Integer identType, String licenseNumber, String systemFlag) {
        try {
            Result result = new Result();
            String licenseNumberEncrypt = EncryptUserInfo.getEncryptString(licenseNumber);
            PersonUserDTO personUserDTO = personUserDao.getPersonInfoByLicenseTypeAndLicenseNumber(identType, licenseNumberEncrypt, systemFlag);
            if (null == personUserDTO) {
                return Result.failure(ErrorEnum.GET_PERSON_USER_INFO_IS_EMPTY);
            }
            result.setInfo(personUserDTO);
            return result;
        } catch (Exception e) {
            throw new ServiceException("通过证件类型和证件号码查询个人用户详情", e);
        }
    }


    @Transactional
    @Override
    public Result updatePersonUser(int personId, PersonUserInfo personUserInfo) {
        Result result = new Result();
        try {
            PersonUserDO personInfo;
            // 获取当前系统时间
            Date date = new Date();
            Timestamp s = new Timestamp(date.getTime());
            personInfo = personUserDao.getPersonInfoById(personId);
            if (null == personInfo) {
                logger.info("更新用户不存在");
                result.setError(ErrorEnum.GET_PERSON_USER_INFO_IS_EXCEPTION);
                return result;
            }
            /**
             * 判断系统标识是否存在
             */
            result = customerSysService.verifyUserSystemFlag(personUserInfo.getSystemFlag());
            if (!result.isSuccess()) {
                return result;
            }
            String licenseNumberEncrypt = EncryptUserInfo.getEncryptString(personUserInfo.getLicenseNumber());
            PersonUserDTO personUserDTO = personUserDao.getPersonInfoByLicenseTypeAndLicenseNumber(personUserInfo.getLicenseType(), licenseNumberEncrypt, personUserInfo.getSystemFlag());
            /**
             * 判断将要更新的licenseNumber是否已经存在
             */
            if (null != personUserDTO && personId != personUserDTO.getUserId()) {
                logger.info("用户信息已经注册");
                result.setError(ErrorEnum.QUERY_BASE_USER_IS_EXISTENCE);
                return result;
            }
            if (null == personUserInfo.getSex()) {
                personUserInfo.setSex(0);
            }
            BeanUtils.copyProperties(personUserInfo, personInfo);
            personInfo.setPersonUserType(0);
            personInfo.setGmtUpdate(s);
            personInfo = EncryptUserInfo.getEncryptPersonUserDO(personInfo);
            int i = personUserDao.updatePersonUser(personInfo);
            if (i != 1) {
                result.setError(ErrorEnum.UPDATE_PERSON_INFO_FAIL);
                return result;
            }

            long userId = personInfo.getUserId();
            BaseUserDO baseUser = baseUserDao.getBaseUserInfo(userId);
            if (baseUser == null) {
                result.setError(ErrorEnum.GET_BASE_USER_INFO_IS_EMPTY);
                return result;
            }
            if (baseUser.getStatus() == UserStatusEnum.DISABLED.id) {
                result.setError(ErrorEnum.USER_STATUS_TYPE_IS_NOT_SUPPORT);
                return result;
            }
            baseUser.setLicenseNumber(personUserInfo.getLicenseNumber());
            baseUser.setLicenseType(personUserInfo.getUserType());
            baseUser.setUserName(personUserInfo.getPersonName());
            baseUser.setGmtUpdate(s);
            baseUser = EncryptUserInfo.getEncryptBaseUserDO(baseUser);
            try {
                baseUserDao.updateBaseUser(baseUser);
            } catch (Exception e) {
                logger.error("更新用户基本信息表异常{}", e);
                result.setError(ErrorEnum.UPDATE_BASE_USER_INFO_ERROR);
                return result;
            }
        } catch (Exception e) {
            result.setError(ErrorEnum.DECRYPT_ENCRYPT_INFO_ERROR);
            return result;
        }

        return result;
    }


    /**
     * 注册用户
     *
     * @param personUserInfo
     * @return
     */
    @Transactional
    @Override
    public Result savePersonUser(PersonUserDTO personUserInfo) {
        Result result = new Result();
        try {
            //baser_user
            BaseUserDO baseUserDO = new BaseUserDO();
            /**
             * 判断系统标识是否存在
             */
            result = customerSysService.verifyUserSystemFlag(personUserInfo.getSystemFlag());
            if (!result.isSuccess()) {
                return result;
            }
            int queryUser = baseUserService.queryUserExist(personUserInfo.getLicenseType(), personUserInfo.getLicenseNumber(), UserTypeEnum.PERSON_USER.id, personUserInfo.getSystemFlag());
            if (queryUser > 0) {
                logger.info("个人用户证件号码已注册");
                result.setError(ErrorEnum.QUERY_BASE_USER_IS_EXISTENCE);
                return result;
            }
            baseUserDO.setUserType(Constants.USER_TYPE_PERSON_1);
            baseUserDO.setUserName(personUserInfo.getPersonName());
            baseUserDO.setLicenseNumber(personUserInfo.getLicenseNumber());
            baseUserDO.setLicenseType(personUserInfo.getLicenseType());
            baseUserDO.setStatus(Constants.USER_STATUS_NORMAL_0);

            // 获取当前系统时间
            Date date = new Date();
            Timestamp s = new Timestamp(date.getTime());
            baseUserDO.setGmtCreate(s);
            baseUserDO.setGmtUpdate(s);
            baseUserDO.setSystemFlag(personUserInfo.getSystemFlag());
            baseUserDO = EncryptUserInfo.getEncryptBaseUserDO(baseUserDO);
            baseUserDao.saveBaseUserInfo(baseUserDO);
            //person_user
            PersonUserDO personUserDO = new PersonUserDO();
            BeanUtils.copyProperties(personUserInfo, personUserDO);
            personUserDO.setUserId(baseUserDO.getId());
            personUserDO.setGmtUpdate(s);
            personUserDO.setGmtCreate(s);
            personUserDO.setPersonUserType(0);
            if (null == personUserDO.getSex()) {
                personUserDO.setSex(0);
            }
            personUserDO = EncryptUserInfo.getEncryptPersonUserDO(personUserDO);
            PersonUserDO personUser = personUserDao.savePersonUserInfo(personUserDO);
            //审计日志
            result.setLogContent(", 用户ID=" + personUser.getId());
        } catch (Exception e) {
            result.setError(ErrorEnum.DECRYPT_ENCRYPT_INFO_ERROR);
            return result;
        }

        return result;
    }

    /**
     * 注册个人用户--批量注册调用
     *
     * @param personUserInfo
     * @return
     */
    @Override
    public Result saveBatchPersonUser(PersonUserDTO personUserInfo) {
        Result result = new Result();
        try {
            // 获取当前系统时间
            Date date = new Date();
            Timestamp s = new Timestamp(date.getTime());
            //baser_user
            BaseUserDO baseUserDO = new BaseUserDO();
            baseUserDO.setUserType(Constants.USER_TYPE_PERSON_1);
            baseUserDO.setUserName(personUserInfo.getPersonName());

            int queryUser = baseUserService.queryUserExist(personUserInfo.getLicenseType(), personUserInfo.getLicenseNumber(), UserTypeEnum.PERSON_USER.id, personUserInfo.getSystemFlag());
            if (queryUser > 0) {
                logger.info("个人用户信息已注册");
                result.setError(ErrorEnum.QUERY_BASE_USER_IS_EXISTENCE);
                return result;
            }
            baseUserDO.setLicenseNumber(personUserInfo.getLicenseNumber());
            baseUserDO.setLicenseType(personUserInfo.getLicenseType());
            baseUserDO.setSystemFlag(personUserInfo.getSystemFlag());
            baseUserDO.setStatus(Constants.USER_STATUS_NORMAL_0);
            baseUserDO.setGmtCreate(s);
            baseUserDO.setGmtUpdate(s);
            baseUserDO = EncryptUserInfo.getEncryptBaseUserDO(baseUserDO);
            baseUserDao.saveBaseUserInfo(baseUserDO);
            //person_user
            PersonUserDO personUserDO = new PersonUserDO();
            BeanUtils.copyProperties(personUserInfo, personUserDO);
            personUserDO.setUserId(baseUserDO.getId());
            personUserDO.setGmtUpdate(s);
            personUserDO.setGmtCreate(s);
            personUserDO.setPersonUserType(0);
            if (null == personUserDO.getSex()) {
                personUserDO.setSex(0);
            }
            personUserDO = EncryptUserInfo.getEncryptPersonUserDO(personUserDO);
            personUserDao.savePersonUserInfo(personUserDO);
            result.setInfo(true);
        } catch (Exception e) {
            result.setError(ErrorEnum.DECRYPT_ENCRYPT_INFO_ERROR);
            return result;
        }
        return result;
    }

    @Transactional
    @Override
    public Result batchSavePersonUserByTemplateFile(ArrayList<ArrayList<String>> fileList) {
        Result result = new Result();
        int excelSize = fileList.size();
        ArrayList<ArrayList<String>> trueList = new ArrayList<>();
        ArrayList<ArrayList<String>> falseList = new ArrayList<>();
        //表格数据格式校验
        for (int i = 0; i < fileList.size(); i++) {
            Result personResult = new PersonParamsCheck().checkPersonParams(fileList.get(i));
            if (personResult.getCode() == 0) {
                //格式正确
                trueList.add((ArrayList<String>) personResult.getInfo());
            } else {
                //格式错误
                falseList.add((ArrayList<String>) personResult.getInfo());
            }
        }
        //infoList存放校验后的正确数据
        ArrayList<SavePersonUserVO> infoList = new ArrayList<SavePersonUserVO>();
        //list转实体类+本地和数据库校验唯一性
        HashSet hashSet = new HashSet();
        //处理错误信息暂存地
        ArrayList list;
        //遍历格式正确的
        for (int i = 0; i < trueList.size(); i++) {
            SavePersonUserVO personUser = ToPersonUserVO.format(trueList.get(i));

            String licenseNumber = personUser.getLicenseNumber();
            Integer licenseType = personUser.getLicenseType();
            String s = licenseNumber + licenseType.toString();
            int queryUser = 0;
            // DO 用户填写的为系统名称 需要查询系统标识入库

            //系统标识校验 DO 0318
            String systemName = personUser.getSystemName();
            int systemFlagSize = 1;
            String systemFlag = null;
            if (StringUtils.isBlank(systemName) || Constants.SYSTEM_FLAG_DEFAULT.equals(systemName)) {
                systemFlag = Constants.SYSTEM_FLAG_DEFAULT;
            } else {
                if (Constants.SYSTEM_FLAG_V2X.equals(systemName)) {
                    systemFlag = Constants.SYSTEM_FLAG_V2X;
                } else {
                    systemFlagSize = baseUserDao.querySystemName(systemName);
                    if (systemFlagSize != 0) {
                        //系统标识一定有
                        systemFlag = baseUserDao.getSystemFlagByName(systemName);
                    }
                }
                //判断本地上传的用户是否重复加上系统名称标识
                s = s + systemFlag;
            }
            personUser.setSystemFlag(systemFlag);
            if (null != systemFlag) {
                try {
                    queryUser = baseUserService.queryUserExist(licenseType, licenseNumber, UserTypeEnum.PERSON_USER.id, systemFlag);
                } catch (Exception e) {
                    logger.info("查询已注册个人用户信息异常");
                    result.setError(ErrorEnum.GET_BASE_USER_INFO_IS_EXCEPTION);
                    TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
                    return result;
                }
            }


            //拿出用户信息的licenseType+licenseNumber放入HashSet,判断本地是否有重复
            if (!hashSet.contains(s)) {
                hashSet.add(s);

                if (queryUser == 0 && systemFlagSize == 1) {
                    infoList.add(personUser);
                } else {
                    list = trueList.get(i);
                    falseList.add(list);
                    if (queryUser > 0 && systemFlagSize != 1) {
                        list.add("个人用户证件号码已存在，所属系统不存在");
                    } else if (queryUser > 0) {
                        list.add("个人用户证件号码已存在");
                    } else {
                        list.add("所属系统不存在");
                    }
                }

            } else {
                list = trueList.get(i);
                list.add("本地上传个人用户重复");
                falseList.add(list);
            }


        }
       /* //校验错误打印
        for (int i = 0; i < falseList.size(); i++) {
            logger.info("校验错误输出:" + falseList.get(i).toString());
        }*/
        //成功注册
        addBatchSuccessInfo(infoList);

        //错误写出
        //错误数据写出文件名
        String filename;
        if (excelSize != infoList.size()) {
            //错误数据下载
            Result name = null;
            try {
                name = ErrorExcel.generateExcel(PathConstants.BATCH_REGISTER_FILE_PATH, Constants.PERSON_1_ERROR, falseList);
            } catch (IOException e) {
                result.setError(ErrorEnum.UPDATE_ERROR_EXCEL_IS_ERROR);
                logger.info("生成错误信息文件失败：", e);
                return result;
            }
            filename = (String) name.getInfo();
        } else {
            filename = null;
        }
        //构造前端返回值
        BatchResp batchResp = new BatchResp(infoList.size(), falseList.size(), filename);
        //审计日志
        result.setLogContent("，成功条数=" + infoList.size() + "，失败条数=" + falseList.size());
        result.setInfo(batchResp);
        return result;
    }

    /**
     * 成功信息 --- 插入
     *
     * @param list 成功数据集合
     * @return
     */
    public Result addBatchSuccessInfo(ArrayList<SavePersonUserVO> list) {
        Result result = new Result();
        for (int i = 0; i < list.size(); i++) {
            SavePersonUserVO savePersonUserVO = list.get(i);
            PersonUserDTO personUserDTO = new PersonUserDTO();
            BeanUtils.copyProperties(savePersonUserVO, personUserDTO);
            //插入数据库
            result = saveBatchPersonUser(personUserDTO);
        }
        result.setInfo(true);
        return result;
    }
}
