package com.xdja.pki.ra.service.manager.certapply;

import com.xdja.pki.ra.core.common.Result;
import com.xdja.pki.ra.core.commonenum.ApplyTypeEnum;
import com.xdja.pki.ra.core.commonenum.ErrorEnum;
import com.xdja.pki.ra.core.constant.Constants;
import com.xdja.pki.ra.core.util.cert.CertUtils;
import com.xdja.pki.ra.manager.dao.CaCertDao;
import com.xdja.pki.ra.manager.dao.RaCertDao;
import com.xdja.pki.ra.manager.dao.UserCertDao;
import com.xdja.pki.ra.manager.dao.model.CaCertDO;
import com.xdja.pki.ra.manager.dao.model.RaCertDO;
import com.xdja.pki.ra.manager.dao.model.UserCertDO;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.Map;

/**
 * 证书申请帮助
 *
 * @author yangmenghao
 * @date 2020-03-24 16:34:09
 */
@Service
public class CertApplyHelperServiceImpl implements CertApplyHelperService {
    private Logger logger = LoggerFactory.getLogger(CertApplyHelperServiceImpl.class);
    @Autowired
    private RaCertDao raCertDao;
    @Autowired
    private CaCertDao caCertDao;
    @Autowired
    private UserCertDao userCertDao;

    @Override
    public Result checkBeforeSaveApplyRecord(String signSn, Integer applyType) {
        Result result = new Result();

        // 获取当前RA服务器签名证书DN
        String raServiceDnName = this.getRAServiceDnName();
        if (StringUtils.isBlank(raServiceDnName)) {
            result.setError(ErrorEnum.GET_RA_SERVICE_DN_NAME_ERROR);
            return result;
        }

        // 获取目标CA证书DN
        String caServiceDnName = this.getCAServiceDnName();
        if (StringUtils.isBlank(caServiceDnName)) {
            result.setError(ErrorEnum.GET_CA_SERVICE_DN_NAME_ERROR);
            return result;
        }

        // 判断当前用户证书状态  是否支持发起撤销申请
        UserCertDO userCertBaseInfo = userCertDao.getUserCertBaseInfo(signSn);
        if (userCertBaseInfo == null) {
            logger.info("获取用户证书信息为空");
            result.setError(ErrorEnum.GET_USER_CERT_INFO_IS_EMPTY);
            return result;
        }
        // BUG #80696 【第三方接口-发起冻结申请】基于已过期的证书（审核策略是自动审核），发起冻结申请，返回错误码提示，但仍生成了冻结申请 - CA
        long time = userCertBaseInfo.getFailureTime().getTime() - System.currentTimeMillis();
        if (Constants.CERT_STATUS_NO_CONFIRM_0 == userCertBaseInfo.getCertStatus() || Constants.CERT_STATUS_EXPIRED_2 == userCertBaseInfo.getCertStatus() || Constants.CERT_STATUS_REVOKED_3 == userCertBaseInfo.getCertStatus() || time < 0) {
            logger.info("用户证书不是正常状态");
            result.setError(ErrorEnum.USER_CERT_IS_NOT_NORMAL_STATUS);
            return result;
        }
        if (applyType.equals(ApplyTypeEnum.UNFREEZE_APPLY.id)) {
            // 只有冻结状态的证书才能解冻
            if (Constants.CERT_STATUS_FREEZE_4 != userCertBaseInfo.getCertStatus()) {
                logger.info("用户证书不是冻结状态");
                result.setError(ErrorEnum.USER_CERT_IS_NOT_FREEZE_STATUS);
                return result;
            }
        } else {
            if (Constants.CERT_STATUS_NORMAL_1 != userCertBaseInfo.getCertStatus()) {
                logger.info("用户证书不是正常状态");
                result.setError(ErrorEnum.USER_CERT_IS_NOT_NORMAL_STATUS);
                return result;
            }
        }

        Map<String, String> dnNameMap = new HashMap<>(4);
        dnNameMap.put("raServiceDnName", raServiceDnName);
        dnNameMap.put("caServiceDnName", caServiceDnName);
        result.setInfo(dnNameMap);
        return result;
    }

    @Override
    public String getRAServiceDnName() {
        RaCertDO newRaCertInfo = null;
        try {
            newRaCertInfo = raCertDao.getNewRaCertInfo();
        } catch (Exception e) {
            logger.error("获取RA服务器证书异常", e);
            return null;
        }

        if (newRaCertInfo == null) {
            return null;
        }
        String certInfo = newRaCertInfo.getCertInfo();
        X509Certificate certFromStr = CertUtils.getCertFromStr(certInfo);
        return CertUtils.getSubjectByX509Cert(certFromStr);
    }

    @Override
    public String getCAServiceDnName() {
        CaCertDO newCaCertInfo = null;
        try {
            newCaCertInfo = caCertDao.getNewCaCertInfo();
        } catch (Exception e) {
            logger.error("获取CA服务器证书异常", e);
            return null;
        }

        if (newCaCertInfo == null) {
            return null;
        }
        String certInfo = newCaCertInfo.getCertInfo();
        X509Certificate certFromStr = CertUtils.getCertFromStr(certInfo);
        return CertUtils.getSubjectByX509Cert(certFromStr);
    }
}
