package com.xdja.pki.ra.service.manager.certapply;

import com.xdja.ca.vo.UserCertInfo;
import com.xdja.pki.ra.core.common.CommonVariable;
import com.xdja.pki.ra.core.common.Result;
import com.xdja.pki.ra.core.commonenum.DoubleCodeUseEnum;
import com.xdja.pki.ra.core.commonenum.ErrorEnum;
import com.xdja.pki.ra.core.constant.Constants;
import com.xdja.pki.ra.core.util.cert.CertUtils;
import com.xdja.pki.ra.core.util.json.JsonUtils;
import com.xdja.pki.ra.manager.dao.*;
import com.xdja.pki.ra.manager.dao.model.*;
import com.xdja.pki.ra.manager.sdk.cmp.CertLifeCycleManager;
import com.xdja.pki.ra.service.manager.certapply.bean.ApplyVariable;
import com.xdja.pki.ra.service.manager.certapply.bean.DoubleCode;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.security.cert.X509Certificate;
import java.sql.Timestamp;
import java.util.Date;
import java.util.Map;
import java.util.concurrent.ThreadLocalRandom;

/**
 * 第三方证书在线申请服务层-实现类
 * @author wangtf
 * @date 2020年8月31日09:10:30
 */
@Service
public class CertApplyServiceIssueImpl implements CertApplyServiceIssue {
    private Logger logger = LoggerFactory.getLogger(CertApplyServiceIssueImpl.class);
    private static final String LOCK_RA_CERT_REQ_ID="raCertReqId";
    private static final String LOCK_PAIR_CERT_INDEX="pairCertIndex";

    @Autowired
    private BaseUserDao baseUserDao;
    @Autowired
    private CertApplyDao certApplyDao;
    @Autowired
    private DoubleCodeDao doubleCodeDao;
    @Autowired
    private IssueApplyDao issueApplyDao;
    @Autowired
    private UpdateApplyDao updateApplyDao;
    @Autowired
    private CertTempDao certTempDao;

    @Autowired
    private CaCertDao caCertDao;
    @Autowired
    private UserCertDao userCertDao;

    @Autowired
    private CertLifeCycleManager certLifeCycleManager;
    @Autowired
    private CertApplyService certApplyService;

    @Override
    public Result certApplyCarry(int applyType, String systemFlag, String applyNo, DoubleCode doubleCode, byte[] issueBytes, Integer keyFormat) {
        String cardNo=null;
        Result result = new Result();

        // 获取当前RA服务器签名证书DN
        String raServiceDnName = this.getRAServiceDnName();
        if (StringUtils.isBlank(raServiceDnName)) {
            logger.error("获取RA服务器证书DN名字错误");
            return Result.failure(ErrorEnum.GET_RA_SERVICE_DN_NAME_ERROR);
        }
        // 获取目标CA证书DN
        String caServiceDnName = this.getCAServiceDnName();
        if (StringUtils.isBlank(caServiceDnName)) {
            logger.error("获取CA服务器证书DN名字错误");
            return Result.failure(ErrorEnum.GET_CA_SERVICE_DN_NAME_ERROR);
        }
        // 判断只有除RSA算法外的Nist和SM2算法提供0016格式的私钥格式
        if (Constants.KEY_ALG_NAME_RSA.equalsIgnoreCase(CommonVariable.getKeyAlgName()) && Constants.KEY_FORMAT_0016_2.equals(keyFormat)) {
            logger.error("RSA算法不提供0016的私钥格式 keyAlgName:{}", CommonVariable.getKeyAlgName());
            return Result.failure(ErrorEnum.RSA_ALG_CANNOT_BUILD_0016_KEY_FORMAT);
        }
        // 检查第三方用户是否存在
        BaseUserDO baseUserDO = baseUserDao.getBaseUserInfo(Integer.parseInt(doubleCode.getRefCode()));
        if (baseUserDO == null || !systemFlag.equals(baseUserDO.getSystemFlag())) {
            logger.error("此用户不存在");
            return Result.failure(ErrorEnum.THE_USER_IS_NOT_EXIT);
        }
        // 查询申请的状态为待签发时，可以发起签发操作
        CertApplyDO certApplyInfo = certApplyDao.getCertApplyInfo(applyNo);
        if (certApplyInfo == null || applyType != certApplyInfo.getApplyType()) {
            logger.error("不存在当前申请编号对应的申请记录 applyNo:{}", applyNo);
            return Result.failure(ErrorEnum.CANNOT_FIND_APPLY_BY_NO);
        }
        if (Constants.CERT_APPLY_STATUS_NOT_ISSUE_3 != certApplyInfo.getApplyStatus()) {
            logger.error("当前申请状态不可发起签发 applyStatus:{}", certApplyInfo.getApplyStatus());
            return Result.failure(ErrorEnum.APPLY_STATUS_NOT_SUPPORT_ISSUE_CERT);
        }
        //校验两码有效性
        DoubleCodeDO doubleCodeDO = doubleCodeDao.getDoubleCode(applyNo);
        if (doubleCodeDO == null || DoubleCodeUseEnum.IS_USE.id == doubleCodeDO.getIsUse()
                || !String.valueOf(doubleCodeDO.getRefCode()).equals(doubleCode.getRefCode())) {
            return Result.failure(ErrorEnum.DOUBLE_CODE_HAS_USED);
        }

//        String raTransId = String.valueOf(System.nanoTime()+ (int) (Math.random() * 900000) + 10000 );
        String raTransId;
        synchronized (LOCK_RA_CERT_REQ_ID){
            raTransId = String.valueOf(System.nanoTime())+ (ThreadLocalRandom.current().nextLong(0,9999999) + 90000000);
        }
        Map<String, String> raMap = ApplyVariable.getRaMap();
        raMap.put(applyNo, raTransId);
        if(Constants.CERT_APPLY_TYPE_ISSUE_1==applyType) {
            // 签发证书申请流程
            result = this.issueApplyHandler(applyNo, systemFlag, cardNo,
                    issueBytes, raServiceDnName, caServiceDnName, raTransId,  true,
                    keyFormat, certApplyInfo);
        } else if(Constants.CERT_APPLY_TYPE_UPDATE_2==applyType) {
            // 签发证书申请流程
            result = this.updateApplyHandler(applyNo, systemFlag, cardNo,
                    issueBytes, raServiceDnName, caServiceDnName, raTransId, true,
                    keyFormat, certApplyInfo );
        }
        if (result.isSuccess()) {
            //更新两码有效性，两码使用后停用
            doubleCodeDao.updateStatus(doubleCode.getRefCode(), doubleCode.getAuthCode());
        } else{
            logger.error("签发证书申请处理失败");
            raMap.remove(applyNo);
        }
        return result;
    }

    /**
     * 处理签发申请
     * @param applyNo        申请编号
     * @param cardNo         卡号
     * @param issueBytes     签发公钥信息(针对Tbox方式签发做了兼容，如果是Tbox的申请该内容为公钥，否则为P10信息)
     * @param raDN           ra的DN
     * @param caDN           ca的DN
     * @param transId        事务ID
     * @param isOnlineIssue  是否在线签发- issueBytes 在线签发情况下为公钥信息，离线签发为P10信息
     * @return Result
     */
    private Result issueApplyHandler(String applyNo, String systemFlag,
                                     String cardNo, byte[] issueBytes, String raDN, String caDN, String transId, boolean isOnlineIssue,
                                     Integer keyFormat, CertApplyDO certApplyInfo) {
        Result result;
        IssueApplyDO issueApplyDO = issueApplyDao.getIssueApplyInfoByApplyId(certApplyInfo.getId());
        CertTempDO certTempDO = certTempDao.getCertTempInfoByTempId(certApplyInfo.getTempId());
        if(issueApplyDO ==null || certTempDO== null) {
            logger.error("获取签发证书申请基本信息为空");
            return Result.failure(ErrorEnum.GET_ISSUE_APPLY_INFO_IS_EMPTY);
        }

        int certValidity = issueApplyDO.getCertValidity();
        String signAlg = issueApplyDO.getSignAlg();
        String tempNo = certTempDO.getTempNo();
        String tempParas = null;
        if (StringUtils.isNotBlank(issueApplyDO.getTempParas())) {
            tempParas = issueApplyDO.getTempParas();
        }

        result = certLifeCycleManager.issueUserCert(applyNo, cardNo, issueBytes, raDN, caDN, transId, tempNo, tempParas,
                signAlg, certValidity, certApplyInfo.getCertDn(), keyFormat);
        if (result.isSuccess()) {
            UserCertInfo userCertInfo = (UserCertInfo) result.getInfo();
            // 用户证书入库
            Result certInsertResult = this.insertUserCertInfo(certApplyInfo.getUserId(), issueApplyDO.getApplyId(), certApplyInfo.getTempId(),
                    certTempDO.getTempNo(), signAlg, issueApplyDO.getPrivateKeyLength(), userCertInfo);
            if (!certInsertResult.isSuccess()) {
                logger.error("将用户证书插入数据库失败");
                return certInsertResult;
            }
        } else {
            // 更新证书申请的证书状态以及操作管理员信息
            certApplyInfo.setAdminId(666L);
            certApplyInfo.setAdminCertDn("CN=当前登录的管理员，O=**省公安厅，C=CN");
            certApplyInfo.setApplyStatus(Constants.CERT_APPLY_STATUS_ISSUE_FAIL_4);
            certApplyInfo.setGmtUpdate(new Timestamp(System.currentTimeMillis()));
            int updateCertApplyStatusResult = certApplyDao.updateCertApply(certApplyInfo);
            if (updateCertApplyStatusResult <= 0) {
                logger.error("更新申请基本信息失败");
                return Result.failure(ErrorEnum.UPDATE_CERT_APPLY_INFO_FAIL);
            } else {
                // 添加申请记录
                certApplyService.insertCertApplyRecord(Constants.CERT_APPLY_TYPE_ISSUE_1, Constants.OPERATE_TYPE_ISSUE_5, applyNo, systemFlag,
                        Constants.CERT_APPLY_STATUS_ISSUE_FAIL_4, result.getErrorBean().getErrMsg(), Constants.CERT_APPLY_OPERATE_TYPE_ISSUE_FAIL_7,
                        false, isOnlineIssue);
                logger.error("调用CA，签发证书失败");
            }
        }
        return result;
    }
    private Result updateApplyHandler( String applyNo, String systemFlag,
                                      String cardNo, byte[] issueBytes, String raDN, String caDN, String transId, boolean isOnlineIssue,
                                      Integer keyFormat, CertApplyDO certApplyInfo) {
        Result result;
        UpdateApplyDO updateApplyDO = updateApplyDao.getUpdateApplyInfoByApplyId(certApplyInfo.getId());
        CertTempDO certTempDO = certTempDao.getCertTempInfoByTempId(certApplyInfo.getTempId());
        if(updateApplyDO ==null || certTempDO== null) {
            logger.error("获取签发证书申请基本信息为空");
            return Result.failure(ErrorEnum.GET_ISSUE_APPLY_INFO_IS_EMPTY);
        }

        int certValidity = updateApplyDO.getCertValidity();
        String signAlg = updateApplyDO.getSignAlg();
        String tempNo = certTempDO.getTempNo();
        String tempParas = null;
        if (StringUtils.isNotBlank(updateApplyDO.getTempParas())) {
            tempParas = updateApplyDO.getTempParas();
        }

        boolean updateKey = updateApplyDO.getIsUpdateKey() == 1;
        logger.debug("updateApplyDO.getIsUpdateKey() :[{}]", updateApplyDO.getIsUpdateKey() );
        logger.debug("updateKey :[{}]", updateKey);

        // END 2020/01/08
        result = certLifeCycleManager.updateUserCert(applyNo, cardNo, issueBytes, raDN, caDN, transId, tempNo, tempParas,
                signAlg, certValidity, certApplyInfo.getCertDn(), updateApplyDO.getSignSn(), updateKey, keyFormat);
        if (result.isSuccess()) {
            UserCertInfo userCertInfo = (UserCertInfo) result.getInfo();

            // 用户证书入库
            Result certInsertResult = this.insertUserCertInfo(certApplyInfo.getUserId(), updateApplyDO.getApplyId(), certApplyInfo.getTempId(),
                    tempNo, signAlg, updateApplyDO.getPrivateKeyLength(), userCertInfo);
            if (!certInsertResult.isSuccess()) {
                logger.info("将用户证书插入数据库失败");
                return certInsertResult;
            }
        } else {
            // 更新申请状态
            certApplyInfo.setAdminId(666L);
            certApplyInfo.setAdminCertDn("CN=当前登录的管理员，O=**省公安厅，C=CN");
            certApplyInfo.setApplyStatus(Constants.CERT_APPLY_STATUS_ISSUE_FAIL_4);
            certApplyInfo.setGmtUpdate(new Timestamp(System.currentTimeMillis()));
            int updateCertApplyStatusResult = certApplyDao.updateCertApply(certApplyInfo);
            if (updateCertApplyStatusResult <= 0) {
                logger.info("操作签发失败:" + JsonUtils.object2Json(certApplyInfo));
                throw new RuntimeException();
            }
            // 添加申请记录
            certApplyService.insertCertApplyRecord(Constants.CERT_APPLY_TYPE_UPDATE_2, Constants.OPERATE_TYPE_ISSUE_5, applyNo, systemFlag,
                    Constants.CERT_APPLY_STATUS_ISSUE_FAIL_4, updateApplyDO.getApplyReason(), Constants.CERT_APPLY_OPERATE_TYPE_ISSUE_FAIL_7,
                    false, isOnlineIssue);
            logger.info("调用CA，更新证书失败");
        }
        return result;
    }

    /**
     * 将CA返回的用户证书插入到数据表中
     * @param userId 用户ID
     * @param applyId 申请ID
     * @param tempId 模板ID
     * @param signAlg 签名密钥类型
     * @param privateKeyLength 私钥长度
     * @param userCertInfo 用户证书信息
     * @return Result
     */
    private Result insertUserCertInfo(long userId, long applyId, long tempId, String tempNo, String signAlg, int privateKeyLength, UserCertInfo userCertInfo) {
        Result result = new Result();
        String signCert = userCertInfo.getSignCert();
        if (StringUtils.isBlank(signCert)) {
            logger.info("CA返回的用户证书信息中，签名证书为空");
            result.setError(ErrorEnum.CA_RESPONSE_USER_SIGN_CERT_INFO_IS_EMPTY);
            return result;
        }
        X509Certificate signCertStr = CertUtils.getCertFromStr(signCert);
        if (signCertStr == null) {
            logger.info("CA返回的用户证书信息中，签名证书错误");
            result.setError(ErrorEnum.CA_RESPONSE_USER_SIGN_CERT_ERROR);
            return result;
        }

        long pairCertIndex;
        synchronized (LOCK_PAIR_CERT_INDEX){
            pairCertIndex = System.nanoTime() + (ThreadLocalRandom.current().nextLong(0,9999999) + 90000000);
        }

        UserCertDO signCertDO = new UserCertDO();
        signCertDO.setPairCertIndex(pairCertIndex);
        if (StringUtils.isBlank(userCertInfo.getEncCert())) {
            signCertDO.setCertType(Constants.CERT_TYPE_SINGLE_1);
        } else {
            signCertDO.setCertType(Constants.CERT_TYPE_SIGN_2);
        }
        signCertDO.setCertStatus(Constants.CERT_STATUS_NO_CONFIRM_0);

        signCertDO.setUserId(userId);
        signCertDO.setApplyId(applyId);
        signCertDO.setTempId(tempId);
        signCertDO.setTempNo(tempNo);
        signCertDO.setSignAlg(signAlg);
        signCertDO.setPrivateKeyLength(privateKeyLength);

        //TODO 数据库中有一份，本地有一份，内存有一份，内容不尽相同，以谁为准
        CaCertDO newCaCertInfo = caCertDao.getNewCaCertInfo();
        if (newCaCertInfo == null) {
            logger.info("获取CA证书信息为空");
            result.setError(ErrorEnum.GET_CA_CERT_INFO_IS_EMPTY);
            return result;
        }
        signCertDO.setCaCertId(newCaCertInfo.getId());


        signCertDO.setCertSn(signCertStr.getSerialNumber().toString(16).toLowerCase());
        signCertDO.setCertDn(CertUtils.getSubjectByX509Cert(signCertStr));

        Date startDate = signCertStr.getNotBefore();
        Date endDate = signCertStr.getNotAfter();
        signCertDO.setEffectiveTime(new Timestamp(startDate.getTime()));
        signCertDO.setFailureTime(new Timestamp(endDate.getTime()));

        // 用户证书有效期
        int certDays = (int) ((endDate.getTime() - startDate.getTime()) / (1000 * 3600 * 24));
        signCertDO.setCertValidity(certDays);

        // 用户证书中对应的加密公私钥的有效天数
        int priKeyDays = (int) ((newCaCertInfo.getFailureTime().getTime() - startDate.getTime()) / (1000 * 3600 * 24));
        signCertDO.setEncKeyValidity(priKeyDays);

        Date date = new Date();
        signCertDO.setGmtCreate(new Timestamp(date.getTime()));
        signCertDO.setGmtUpdate((new Timestamp(date.getTime())));
        UserCertDO userSignCertDO = userCertDao.insertUserCertInfo(signCertDO);

        // 加密
        String encCert = userCertInfo.getEncCert();
        if (StringUtils.isNotBlank(encCert)) {

            UserCertDO encCertDO = new UserCertDO();
            encCertDO.setPairCertIndex(pairCertIndex);
            encCertDO.setCertType(Constants.CERT_TYPE_ENC_3);
            encCertDO.setCertStatus(Constants.CERT_STATUS_NO_CONFIRM_0);

            encCertDO.setUserId(userId);
            encCertDO.setApplyId(applyId);
            encCertDO.setTempId(tempId);
            encCertDO.setTempNo(tempNo);
            encCertDO.setSignAlg(signAlg);
            encCertDO.setPrivateKeyLength(privateKeyLength);

            encCertDO.setCaCertId(newCaCertInfo.getId());

            encCertDO.setEffectiveTime(new Timestamp(startDate.getTime()));
            encCertDO.setFailureTime(new Timestamp(endDate.getTime()));

            encCertDO.setCertValidity(certDays);
            encCertDO.setEncKeyValidity(priKeyDays);

            X509Certificate encCertStr = CertUtils.getCertFromStr(encCert);
            if (encCertStr == null) {
                logger.info("CA返回的用户证书信息中，加密证书错误");
                result.setError(ErrorEnum.CA_RESPONSE_USER_ENC_CERT_ERROR);
                return result;
            }

            encCertDO.setGmtCreate(new Timestamp(date.getTime()));
            encCertDO.setGmtUpdate((new Timestamp(date.getTime())));

            encCertDO.setCertSn(encCertStr.getSerialNumber().toString(16).toLowerCase());
            encCertDO.setSignCertSn(userSignCertDO.getCertSn());
            try {
                encCertDO.setCertDn(CertUtils.getSubjectByX509Cert(encCertStr));
                userCertDao.insertUserCertInfo(encCertDO);
            } catch (Exception e) {
                logger.info("手动处理manager层的插入异常");
                userCertDao.deleteUserCert(userSignCertDO.getId());
            }
        }
        return result;
    }

    /**
     * 获取系统RA服务器签名证书DN
     * @return String
     */
    //TODO 是否可直接获取DNname，不用每次都解析，减少CPU消耗
    private String getRAServiceDnName() {
        X509Certificate certFromStr = CommonVariable.getRaServiceCert();
        return CertUtils.getSubjectByX509Cert(certFromStr);
    }

    /**
     * 获取系统CA服务器证书DN
     * @return String
     */
    //TODO 是否可直接获取DNname，不用每次都解析，减少CPU消耗
    private String getCAServiceDnName() {
        X509Certificate certFromStr = CommonVariable.getCaServiceCert();
        return CertUtils.getSubjectByX509Cert(certFromStr);
    }

}
