package com.xdja.pki.ra.service.manager.certapply;

import com.alibaba.fastjson.JSON;
import com.fasterxml.jackson.core.type.TypeReference;
import com.xdja.pki.core.exception.JSONException;
import com.xdja.pki.core.utils.JsonMapper;
import com.xdja.pki.ra.core.common.CommonVariable;
import com.xdja.pki.ra.core.common.Result;
import com.xdja.pki.ra.core.commonenum.*;
import com.xdja.pki.ra.core.constant.Constants;
import com.xdja.pki.ra.core.exception.ServiceException;
import com.xdja.pki.ra.manager.dao.*;
import com.xdja.pki.ra.manager.dao.model.*;
import com.xdja.pki.ra.manager.dto.*;
import com.xdja.pki.ra.service.manager.CommonService;
import com.xdja.pki.ra.service.manager.baseuser.BaseUserService;
import com.xdja.pki.ra.service.manager.certapply.bean.*;
import com.xdja.pki.ra.service.manager.template.CertTempService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.security.cert.X509Certificate;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;


/**
 * @author: ggp
 * @Date: 2019/10/25 08:39
 * @Description:
 */
@Service
public class CertApplyServiceManagerImpl implements CertApplyManagerService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private static final String LOCK_APPLY_NO="applyNo";

    @Autowired
    BaseUserService baseUserService;

    @Autowired
    private IssueApplyService issueApplyService;

    @Autowired
    private UpdateApplyService updateApplyService;

    @Autowired
    private RevokeApplyService revokeApplyService;
    @Autowired
    private RecoveryApplyService recoveryApplyService;
    @Autowired
    private CertApplyService certApplyService;

    @Autowired
    private CertApplyServiceIssue certApplyServiceIssue;

    @Autowired
    private FreezeApplyService freezeApplyService;

    @Autowired
    private CertTempDao certTempDao;

    @Autowired
    private BaseUserDao baseUserDao;

    @Autowired
    private UserCertDao userCertDao;

    @Autowired
    private DoubleCodeDao doubleCodeDao;

    @Autowired
    private CertApplyDao certApplyDao;

    @Autowired
    private CommonService commonService;

    @Autowired
    private CustomerDao customerDao;

    @Autowired
    private CertTempService certTempService;

    @Autowired
    UpdateApplyDao updateApplyDao;

    @Autowired
    RevokeApplyDao revokeApplyDao;

    @Autowired
    ApplyRecordDao applyRecordDao;

    @Autowired
    IssueApplyDao issueApplyDao;

    @Autowired
    CaCertDao caCertDao;

    /**
     * 获取RA系统的BaseDn
     *
     * @return
     */
    @Override
    public Result getRABaseDn() {
        Result result = certApplyService.getRaBaseDn();
        return result;
    }

    /**
     * 获取用户模板信息
     *
     * @param systemName
     * @return
     */
    @Override
    public Result getTemplateInfoList(String systemName) {
        Result result = new Result();
        List<CertTempDO> certTempDOList = null;
        try {
            certTempDOList = customerDao.getTempNoList(systemName);
            if (null == certTempDOList || CollectionUtils.isEmpty(certTempDOList)) {
                result.setError(ErrorEnum.CERT_TEMP_LIST_IS_EMPTY);
                return result;
            }
        } catch (Exception e) {
            logger.error("查询第三方系统模板列表异常", e);
            result.setError(ErrorEnum.CERT_TEMP_LIST_IS_EMPTY);
            return result;
        }

        List<TemplateInfo> templateInfos = new ArrayList<>();
        for (CertTempDO certTempDO : certTempDOList) {
            /**
             * 基本数据转换
             */
            TemplateInfo templateInfo = new TemplateInfo();
            templateInfo.setKeyLength(certTempDO.getPrivateKeyLength());
            templateInfo.setSignAlg(certTempDO.getSignAlg());
            templateInfo.setTempCertValidity(certTempDO.getMaxValidity());
            templateInfo.setTempNo(certTempDO.getTempNo());
            templateInfo.setCheckStrategy(certTempDO.getCheckStrategy());
            templateInfo.setBaseDn(certTempDO.getBaseDn());
            /**
             * 扩展项数据转换
             */
            List<Extension> extensions = new ArrayList<>();
            List<ExtensionAttr> extensionAttrs = formatExtensionJon(certTempDO.getTempParas());
            for (ExtensionAttr extensionAttr : extensionAttrs) {
                Extension extension = new Extension();
                extension.setAttrOid(extensionAttr.getAttrOid());
                extension.setAttrName(extensionAttr.getAttrName());
                extension.setAttrRequired(extensionAttr.getAttrRequired());
                extension.setAttrCustomize(extensionAttr.getAttrCustomize());
                /**
                 * 扩展项属性转换
                 */
                List<AttrType> attrTypes = new ArrayList<>();
                String[] types = extensionAttr.getAttrType().split("#");
                /**
                 * 只有个人身份标识和主体替换名称有多个属性
                 */
                for (int i = 0; i < types.length; i++) {
                    AttrType attrType = new AttrType();
                    attrType.setAttrTypeNo(types[i]);
                    if (extensionAttr.getAttrOid().equals(ExtensionAttrTypeEnum.PERSONAL_IDENTIFICATION_CODE.oid)) {
                        attrType.setAttrTypeName(PersonIdentificationCodeEnum.getNameFormId(types[i]));
                    } else if (extensionAttr.getAttrOid().equals(ExtensionAttrTypeEnum.SUBJECT_REPLACEMENT_NAME.oid)) {
                        attrType.setAttrTypeName(SubjectReplacementNameEnum.getNameFormId(types[i]));
                    } else {
                        attrType.setAttrTypeName("");
                    }
                    attrType.setAttrTypeValue("");
                    attrTypes.add(attrType);
                }
                extension.setAttrTypeList(attrTypes);
                extensions.add(extension);
            }
            templateInfo.setExtensions(extensions);
            templateInfos.add(templateInfo);
        }
        result.setInfo(templateInfos);
        return result;
    }

    /**
     * 签发申请-发起
     *
     * @param issueApply
     * @return
     */
    @Override
    @Transactional
    public Result issueApplyLaunch(IssueApply issueApply, String systemFlag) {
        ApplyLaunch applyLaunch = new ApplyLaunch();
        applyLaunch.setApplyType(ApplyTypeEnum.ISSUE_APPLY.id);
        Result result = checkValidity(issueApply.getCertValidity(), true, null, issueApply.getTempNo());
        if (!result.isSuccess()) {
            return result;
        }
        // 将check之后的结果赋值给申请的证书有效期字段
        issueApply.setCertValidity((Integer) result.getInfo());

        result = checkParameter(issueApply.getPrivateKeyLength(), issueApply.getSignAlg());

        if (!result.isSuccess()) {
            return result;
        }
        /**
         * 判断该模板是否授权给该第三方系统
         */
        List<CertTempDO> certTempDOList = customerDao.getTempNoList(systemFlag);
        boolean flag = false;
        String baseDn = null;
        for (CertTempDO certTempDO : certTempDOList) {
            if (certTempDO.getTempNo().equals(issueApply.getTempNo())) {
                flag = true;
                baseDn = certTempDO.getBaseDn();
            }
        }
        if (!flag) {
            logger.error("模板:[{}]未授权给该第三方系统:[{}]", issueApply.getTempNo(), systemFlag);
            return Result.failure(ErrorEnum.CERT_TEMPLATE_IS_NOT_AUTHORIZATION);
        }

        int count = baseUserService.queryUserExist(issueApply.getLicenseType(), issueApply.getLicenseNumber(), issueApply.getUserType(), systemFlag);
        if (count == 0) {
            logger.error("此用户不存在");
            return Result.failure(ErrorEnum.THE_USER_IS_NOT_EXIT);
        }
        applyLaunch.setIssueApply(issueApply);
        result = getApplyLaunchResult(systemFlag, applyLaunch, baseDn);
        return result;
    }

    /**
     * 更新申请-发起
     * @param updateApply 更新申请信息
     * @param systemFlag 系统表示
     * @return Result
     * modify by wangtf 2020年9月2日08:46:58【压测-优化】
     */
    @Override
    @Transactional
    public Result updateApplyLaunch(UpdateApply updateApply, String systemFlag) {
        Result result;
        String signSn = updateApply.getSignSn();
        //todo
        //获取用户证书信息
        UserCertDO userCertDO = userCertDao.getUserCertBaseInfo(signSn);
        if (userCertDO == null) {
            return Result.failure(ErrorEnum.THE_CERT_IS_NOT_EXIT);
        }
        long time = userCertDO.getFailureTime().getTime() - System.currentTimeMillis();
        if (Constants.CERT_STATUS_NORMAL_1 != userCertDO.getCertStatus() || time < 0) {
            return Result.failure(ErrorEnum.USER_CERT_IS_NOT_NORMAL_STATUS);
        }

        String tempNo = userCertDO.getTempNo();
        Long userId = userCertDO.getUserId();

        //获取模板信息
        CertTempDO certTempInfo = certTempDao.getCertTempInfoByTempNo(tempNo);
        if (certTempInfo == null){
            return Result.failure(ErrorEnum.GET_CERT_TEMP_INFO_IS_EMPTY);
        }
        if (certTempInfo.getTempStatus() != Constants.TEMP_STATUS_RUN_2 || certTempInfo.getTempBound() != Constants.TEMP_BOUND_STATUS_1) {
            return Result.failure(ErrorEnum.CERT_TEMP_STATUS_IS_NOT_NORMAL);
        }

        if (updateApply.getUpdateValidity()) {
            Integer validity = updateApply.getCertValidity();
            if (validity == null || validity <= 0) {
                return Result.failure(ErrorEnum.APPLY_VALIDITY_IS_NOT_POSITIVE_INTEGER);
            }

            boolean isUpdateKey = updateApply.getUpdateKey();
            //获取最新的CA证书信息//
            // CaCertDO newCaCertInfo = caCertDao.getNewCaCertInfo();
            X509Certificate newCaCertInfo = CommonVariable.getCaServiceCert();
            if (newCaCertInfo == null){
                return Result.failure(ErrorEnum.GET_CA_CERT_INFO_IS_EMPTY);
            }
            int maxValidity =  this.getCertApplyMaxValidity(tempNo, isUpdateKey ? 1 : 0,
                    certTempInfo.getMaxValidity(), userCertDO.getEncKeyValidity(),newCaCertInfo.getNotAfter().getTime());
            if (validity > maxValidity) {
                return Result.failure(ErrorEnum.APPLY_VALIDITY_IS_TOO_LARGE);
            }
        }

        //一个证书只能存在一个正在流转的相同类型的证书申请，不能重复发起申请。
        int unClosedApplyNum = updateApplyDao.getUnClosedUpdateApplyNum(signSn);
        if (unClosedApplyNum > 0) {
            return Result.failure(ErrorEnum.SIGN_SN_HAS_UPDATE_APPLY_NOT_CLOSED);
        }

        //判断用户不存在
        BaseUserDO baseUserDO = baseUserDao.getBaseUserInfo(userId);
        if (!baseUserDO.getSystemFlag().equals(systemFlag)) {
            return Result.failure(ErrorEnum.THE_USER_IS_NOT_EXIT);
        }
        if (baseUserDO.getStatus() == Constants.USER_STATUS_STOP_1) {
            return Result.failure(ErrorEnum.UPDATE_APPLY_USER_STATUS_IS_STOP);
        }

        String baseDn = certTempInfo.getBaseDn();
        String certDn = updateApply.getCertDn();
        result = this.checkDn(baseDn, certDn, userId, null);
        if (!result.isSuccess()) {
            logger.error("用户更新申请DN错误，DN[{}]", certDn);
            return result;
        }

        UpdateApplyDTO updateApplyDTO = new UpdateApplyDTO();
        BeanUtils.copyProperties(updateApply, updateApplyDTO);
        updateApplyDTO.setUserId(userId);
        updateApplyDTO.setTempNo(tempNo);
        updateApplyDTO.setSignAlg(certTempInfo.getSignAlg());
        updateApplyDTO.setPrivateKeyLength(certTempInfo.getPrivateKeyLength());
        String encSn = userCertDao.getEncSnBySignSn(signSn);
        updateApplyDTO.setEncSn(encSn == null ? "" : encSn);//申请编号

//        String applyNo = DateUtils.getCurrDate(DateUtils.FORMAT_FIVE) + (int) (Math.random() * 900000) + 100000;
        String applyNo;
        synchronized (LOCK_APPLY_NO){
            applyNo = String.valueOf(System.nanoTime())+ (ThreadLocalRandom.current().nextLong(0,9999999) + 90000000);
        }
        updateApplyDTO.setApplyNo(applyNo);
        result = updateApplyService.insertCertUpdateApply(updateApplyDTO, userCertDO, certTempInfo,true);
        // 添加申请记录信息
        certApplyService.insertCertApplyRecord(Constants.CERT_APPLY_TYPE_UPDATE_2, Constants.OPERATE_TYPE_ENTRY_1, applyNo,
                systemFlag, Constants.CERT_APPLY_STATUS_NOT_CHECK_1, updateApplyDTO.getApplyReason(),
                Constants.CERT_APPLY_OPERATE_TYPE_SUBMIT_SUCCESS_1, false, true);

        // 如果为自动审核的模板，则添加自动审核记录
        if (Constants.TEMP_CHECK_STRATEGY_AUTO_1 == certTempInfo.getCheckStrategy()) {
            certApplyService.insertCertApplyRecord(Constants.CERT_APPLY_TYPE_UPDATE_2, Constants.OPERATE_TYPE_CHECK_3, applyNo,
                    systemFlag, Constants.CERT_APPLY_STATUS_NOT_ISSUE_3, "自动审核类模板-审核成功",
                    Constants.CERT_APPLY_OPERATE_TYPE_CHECK_SUCCESS_4,true, true);
        }

        boolean certTempIsAutoStrategy = (certTempInfo.getCheckStrategy() == Constants.TEMP_CHECK_STRATEGY_AUTO_1);
        ApplyRep applyRep = new ApplyRep();
        applyRep.setNeedCheck(!certTempIsAutoStrategy);
        if (result.isSuccess()) {
            applyRep.setApplyNo(applyNo);
            if (certTempIsAutoStrategy) {
                DoubleCodeDO doubleCodeDO = new DoubleCodeDO();
                doubleCodeDO.setRefCode(userId);
                doubleCodeDO.setAuthCode(applyNo);
                doubleCodeDO.setIsUse(DoubleCodeUseEnum.NOT_USE.id);
                doubleCodeDao.InsertDouble(doubleCodeDO);

                DoubleCode doubleCode = new DoubleCode();
                doubleCode.setRefCode(String.valueOf(userId));
                doubleCode.setAuthCode(applyNo);
                applyRep.setDoubleCode(doubleCode);
            }
            result.setInfo(applyRep);
        }
        return result;

    }



    /**
     * @author wangtf 2020年9月2日09:45:59
     * @param tempNo 模板编号
     * @param updateKey 是否更新密钥
     * @param tempMaxValidity 模板最大有效期
     * @param priKeyValidity 私钥有效期
     * @param failTime 失效时间
     * @return int
     */
    private int getCertApplyMaxValidity(String tempNo,int updateKey,int tempMaxValidity, int priKeyValidity,long failTime) {
//        long failTime = failureTime.getTime();
        long nowTime = System.currentTimeMillis();
        int caValidity = (int) ((failTime - nowTime) / (1000 * 3600 * 24));

        int min = Math.min(caValidity, tempMaxValidity);
        // 如果不更新密钥  则需要将用户证书的加密公私钥的有效期计算在内
        if (updateKey == Constants.CANNOT_UPDATE_CERT_KEY_0) {
            // 取CA证书有效期、模板有效期、加密密钥有效期的最小值
            min = Math.min(priKeyValidity, min);
        }
        logger.debug("用户模板:"+tempNo+" CA最大有效期:"+caValidity+" 模板最大有效期:"+tempMaxValidity+" 加密密钥最大有效期:"+priKeyValidity);
        return min;
    }

    private boolean certIsExit(String signSn, String systemFlag) {
        boolean flag = baseUserDao.isAuthorize(systemFlag, signSn);
        if (!flag) {
            logger.error("此证书不存在");
            return true;
        }
        return false;
    }

    /**
     * 一个证书只能存在一个正在流转的相同类型的证书申请，不能重复发起申请。
     *
     * @param signSn
     * @param applyType
     * @return
     */
    private Result checkCertApply(String signSn, Integer applyType) {
        return certApplyService.verifyUserCertApply(signSn, applyType);
    }

    /**
     * 校验sn是否存在
     *
     * @param signSn
     * @return
     */
    private Result checkSn(String signSn) {
        UserCertDO userCertDO = userCertDao.getUserCertBaseInfoBySignSn(signSn);
        if (null == userCertDO) {
            return Result.failure(ErrorEnum.THE_SN_IS_INVALID);
        }
        return new Result();
    }

    /**
     * 撤销申请-发起
     *
     * @param revokeApply
     * @return
     */
    @Override
    public Result revokeApplyLaunch(RevokeApply revokeApply, String systemFlag) {
        ApplyLaunch applyLaunch = new ApplyLaunch();
        Result result = checkSn(revokeApply.getSignSn());
        if (!result.isSuccess()) {
            return result;
        }
        result = checkCertApply(revokeApply.getSignSn(), ApplyTypeEnum.REVOKE_APPLY.id);
        if (!result.isSuccess()) {
            return result;
        }
        if (certIsExit(revokeApply.getSignSn(), systemFlag)) {
            return Result.failure(ErrorEnum.THE_CERT_IS_NOT_EXIT);
        }
        applyLaunch.setApplyType(ApplyTypeEnum.REVOKE_APPLY.id);
        applyLaunch.setRevokeApply(revokeApply);
        return getApplyLaunchResult(systemFlag, applyLaunch, null);
    }

    /**
     * 恢复申请-发起
     *
     * @param recoveryApply RecoveryApply
     * @param systemFlag    systemFlag
     * @return ApplyRep类型的签发申请返回对象
     */
    @Override
    public Result recoveryApplyLaunch(RecoveryApply recoveryApply, String systemFlag) {
        ApplyLaunch applyLaunch = new ApplyLaunch();
        Result result = checkSn(recoveryApply.getSignSn());
        if (!result.isSuccess()) {
            return result;
        }
        result = checkCertApply(recoveryApply.getSignSn(), ApplyTypeEnum.RECOVERY_APPLY.id);
        if (!result.isSuccess()) {
            return result;
        }
        if (certIsExit(recoveryApply.getSignSn(), systemFlag)) {
            return Result.failure(ErrorEnum.THE_CERT_IS_NOT_EXIT);
        }
        applyLaunch.setApplyType(ApplyTypeEnum.RECOVERY_APPLY.id);
        applyLaunch.setRecoveryApply(recoveryApply);
        return getApplyLaunchResult(systemFlag, applyLaunch, null);
    }

    /**
     * 冻结申请-发起
     *
     * @param freezeApply FreezeApply
     * @param systemFlag  systemFlag
     * @return ApplyRep类型的签发申请返回对象
     */
    @Override
    public Result freezeApplyLaunch(FreezeApply freezeApply, String systemFlag) {
        ApplyLaunch applyLaunch = new ApplyLaunch();
        Result result = checkSn(freezeApply.getSignSn());
        if (!result.isSuccess()) {
            return result;
        }
        int applyType;
        applyLaunch.setFreezeApply(freezeApply);
        if (freezeApply.getFreeze()) {
            applyType = ApplyTypeEnum.FREEZE_APPLY.id;
        } else {
            applyType = ApplyTypeEnum.UNFREEZE_APPLY.id;
        }
        applyLaunch.setApplyType(applyType);
        result = checkCertApply(freezeApply.getSignSn(), applyType);
        if (!result.isSuccess()) {
            return result;
        }
        if (certIsExit(freezeApply.getSignSn(), systemFlag)) {
            return Result.failure(ErrorEnum.THE_CERT_IS_NOT_EXIT);
        }
        // RSA联调环境接口bug-基于已停用的用户可以发起冻结解冻申请，应限制不能发起
        Long id = userCertDao.getUserId(freezeApply.getSignSn());
        if (id == null) {
            return Result.failure(ErrorEnum.THE_CERT_IS_NOT_EXIT);
        }
        BaseUserDO baseUserDO = baseUserDao.getBaseUserInfo(id, systemFlag);
        if (baseUserDO == null) {
            return Result.failure(ErrorEnum.THE_CERT_IS_NOT_EXIT);
        } else {
            Integer status = baseUserDO.getStatus();
        }
        return getFreezeApplyLaunchResult(systemFlag, applyLaunch);
    }


    /**
     * 申请审核
     *
     * @param applyType  申请类型	1-签发申请 2-更新申请 3-撤销申请 4-恢复申请
     * @param applyNo    申请编号	发起申请时，返回的申请编号
     * @param checkApply 审核申请对象	审核各种申请审核类对象
     * @param systemFlag systemFlag
     * @return
     */
    @Override
    public Result applyCheck(Integer applyType, String applyNo, CheckApply checkApply, String systemFlag) {
        if (applyIsExit(applyNo, systemFlag, applyType)) {
            return Result.failure(ErrorEnum.THE_APPLY_IS_NOT_EXIT);
        }
        // BUG #80420 【第三方接口-申请审核】冻结解冻申请进行审核通过后无需生成、返回两码 - CA yangmenghao 2020-04-27
        Result result = certApplyService.checkUserCertApply(applyNo, systemFlag, applyType, checkApply.getPass(), checkApply.getCheckInfo(), true);
        if (result.isSuccess() && checkApply.getPass() && applyType != ApplyTypeEnum.REVOKE_APPLY.id && applyType != ApplyTypeEnum.FREEZE_APPLY.id && applyType != ApplyTypeEnum.UNFREEZE_APPLY.id) {
            result.setInfo(commonService.createDoubleCode(applyNo));
        }
        return result;
    }

    private boolean applyIsExit(String applyNo, String systemFlag, Integer applyType) {
        boolean flag = baseUserDao.isAuthorizeByApplyNo(systemFlag, applyNo, applyType);
        if (!flag) {
            logger.error("此申请不存在");
            return true;
        }
        return false;
    }


    /**
     * 申请制证执行
     * @param applyType
     * @param applyNo
     * @param doubleCode
     * @return
     */
    @Override
    @Transactional
    public Result certApplyCarry(String systemFlag, Integer applyType, String applyNo, DoubleCode doubleCode, byte[] publickey, Integer keyFormat) {
        if(Constants.CERT_APPLY_TYPE_ISSUE_1 == applyType || Constants.CERT_APPLY_TYPE_UPDATE_2 == applyType) {
            return certApplyServiceIssue.certApplyCarry(applyType, systemFlag, applyNo, doubleCode, publickey, keyFormat);
        }
        boolean flag = baseUserDao.isAuthorize(systemFlag, Integer.parseInt(doubleCode.getRefCode()));
        if (!flag) {
            logger.error("此用户不存在");
            Result result = new Result();
            result.setError(ErrorEnum.THE_USER_IS_NOT_EXIT);
            return result;
        }
        flag = baseUserDao.isAuthorizeByApplyNo(systemFlag, applyNo, applyType);
        if (!flag) {
            logger.error("此申请不存在");
            Result result = new Result();
            result.setError(ErrorEnum.THE_APPLY_IS_NOT_EXIT);
            return result;
        }
        /**
         * 校验两码有效性
         */
        boolean checkResult = doubleCodeDao.checkDoubleCode(doubleCode.getRefCode(), doubleCode.getAuthCode());
        if (!checkResult) {
            return Result.failure(ErrorEnum.DOUBLE_CODE_HAS_USED);
        }
        /**
         * 用户证书-签发
         */
        Result result = certApplyService.issueUserCert(null, null, Constants.P10_ISSUE_TYPE_P7B_4, applyType, applyNo, systemFlag
                , null, publickey, Constants.APPLY_CERT_TYPE_BY_P10_2, true, false, null, keyFormat);
        /**
         * 更新两码有效性，两码使用后停用
         */
        if (result.isSuccess()) {
            doubleCodeDao.updateStatus(doubleCode.getRefCode(), doubleCode.getAuthCode());
        }
        return result;
    }

    /**
     * 证书签发确认消息
     *
     * @param applyNo
     * @return
     */
    @Override
    public Result certIssueConfirmMsg(String applyNo) {
        try {
            return certApplyService.issueUserCertResp(applyNo, null, true);
        } catch (Exception e) {
            throw new ServiceException("证书签发确认消息失败", e);
        }
    }

    /**
     * 证书签发错误消息
     *
     * @param applyNo
     * @param errorMsg
     * @return
     */
    @Override
    public Result certApplyErrorMsg(String applyNo, ErrorMsg errorMsg) {
        try {
            return certApplyService.genErrorMsgContent(applyNo, errorMsg.getErrorCode(), errorMsg.getErrorMsg(), true);
        } catch (Exception e) {
            throw new ServiceException("证书签发错误消息", e);
        }
    }

    /**
     * 查询证书申请信息
     *
     * @param applyType
     * @param applyNo
     * @return
     */
    @Override
    public Result certApplyQuery(Integer applyType, String applyNo, String systemFlag) {
        CertApplyInfo certApplyInfo = new CertApplyInfo();
        Result result;
        if (applyIsExit(applyNo, systemFlag, applyType)) {
            return Result.failure(ErrorEnum.THE_APPLY_IS_NOT_EXIT);
        }
        if (applyType == ApplyTypeEnum.ISSUE_APPLY.id) {
            result = issueApplyService.getIssueApplyInfo(applyNo);
            if (!result.isSuccess()) {
                return result;
            }
            IssueApplyInfo issueApplyInfo = new IssueApplyInfo();
            BeanUtils.copyProperties(result.getInfo(), issueApplyInfo);
            certApplyInfo.setIssueApplyInfo(issueApplyInfo);
        } else if (applyType == ApplyTypeEnum.UPDATE_APPLY.id) {
            result = updateApplyService.getUpdateApplyInfo(applyNo);
            if (!result.isSuccess()) {
                return result;
            }
            UpdateApplyInfo updateApplyInfo = new UpdateApplyInfo();
            BeanUtils.copyProperties(result.getInfo(), updateApplyInfo);
            certApplyInfo.setUpdateApplyInfo(updateApplyInfo);
        } else if (applyType == ApplyTypeEnum.REVOKE_APPLY.id) {
            result = revokeApplyService.getRevokeApplyInfo(applyNo);
            if (!result.isSuccess()) {
                return result;
            }
            RevokeApplyInfo revokeApplyInfo = new RevokeApplyInfo();
            BeanUtils.copyProperties(result.getInfo(), revokeApplyInfo);
            certApplyInfo.setRevokeApplyInfo(revokeApplyInfo);
        } else if (applyType == ApplyTypeEnum.RECOVERY_APPLY.id) {
            result = recoveryApplyService.getRecoveryApplyInfo(applyNo);
            if (!result.isSuccess()) {
                return result;
            }
            RecoveryApplyInfo recoveryApplyInfo = new RecoveryApplyInfo();
            BeanUtils.copyProperties(result.getInfo(), recoveryApplyInfo);
            certApplyInfo.setRecoveryApplyInfo(recoveryApplyInfo);
        }
        /*else if (applyType == ApplyTypeEnum.FREEZE_APPLY.id || applyType == ApplyTypeEnum.UNFREEZE_APPLY.id)*/
        else {
            result = freezeApplyService.getFreezeApplyInfo(applyNo);
            if (!result.isSuccess()) {
                return result;
            }
            FreezeApplyInfo freezeApplyInfo = new FreezeApplyInfo();
            BeanUtils.copyProperties(result.getInfo(), freezeApplyInfo);
            certApplyInfo.setFreezeApplyInfo(freezeApplyInfo);
        }

        result.setInfo(certApplyInfo);
        return result;
    }

    /**
     * 查询证书申请信息
     *
     * @param applyNo
     * @param systemFlag
     * @return
     */
    @Override
    public Result certApplyQuery(String applyNo, String systemFlag) {
        BaseCertApply baseCertApply = new BaseCertApply();
        Result result = new Result();
        if (applyIsExit(applyNo, systemFlag,null)) {
            return Result.failure(ErrorEnum.THE_APPLY_IS_NOT_EXIT);
        }
        CertApplyDO certApplyDO = certApplyDao.getCertApplyInfo(applyNo);
        Integer applyStatus = certApplyDO.getApplyStatus();
        baseCertApply.setApplyStatus(applyStatus);
        if (applyStatus == ApplyStatusEnum.PENDING_ISSUANCE.id) {
            DoubleCodeDO doubleCode = doubleCodeDao.getDoubleCode(applyNo);
            DoubleCode doubleCode1 = new DoubleCode();
            doubleCode1.setRefCode(String.valueOf(doubleCode.getRefCode()));
            doubleCode1.setAuthCode(String.valueOf(doubleCode.getAuthCode()));
            baseCertApply.setDoubleCode(doubleCode1);
        }

        // 如果申请状态为某个失败的申请状态，则查询相应的失败原因，并返回
        if (applyStatus == ApplyStatusEnum.REVIEW_FAILED.id || applyStatus == ApplyStatusEnum.ISSUING_FAILED.id ||
                applyStatus == ApplyStatusEnum.CANCELLATION_FAILED.id || applyStatus == ApplyStatusEnum.FREEZE_FAILED.id ||
                applyStatus == ApplyStatusEnum.UN_FREEZE_FAILED.id) {
            String applyFailedReason = applyRecordDao.getApplyFailedReason(applyNo, applyStatus);
            baseCertApply.setFailReason(applyFailedReason);
        }

        result.setInfo(baseCertApply);
        return result;
    }

    /**
     * 修改证书申请信息
     *
     * @param applyType
     * @param applyNo
     * @param editCertApplyInfo
     * @return
     */
    @Override
    public Result certApplyUpdate(Integer applyType, String applyNo, EditCertApplyInfo editCertApplyInfo, String systemFlag) {
        Result result;
        if (applyIsExit(applyNo, systemFlag, applyType)) {
            return Result.failure(ErrorEnum.THE_APPLY_IS_NOT_EXIT);
        }
        long userId = certApplyDao.getUserIdByApplyNo(applyNo);
        if (applyType == ApplyTypeEnum.ISSUE_APPLY.id) {
            IssueApplyDTO issueApplyDTO = new IssueApplyDTO();
            BeanUtils.copyProperties(editCertApplyInfo.getEditIssueApplyInfo(), issueApplyDTO);
            result = checkValidity(issueApplyDTO.getCertValidity(), true, null, issueApplyDTO.getTempNo());
            if (!result.isSuccess()) {
                return result;
            }
            // 将check之后的结果赋值给申请的证书有效期字段
            issueApplyDTO.setCertValidity((Integer) result.getInfo());

            IssueApplyDTO issueApplyInfoByApplyNo = issueApplyDao.getIssueApplyInfoByApplyNo(applyNo);
            result = checkDn(issueApplyInfoByApplyNo.getBaseDn(), issueApplyDTO.getCertDn(), userId, applyNo);
            if (!result.isSuccess()) {
                logger.error("用户修改签发申请DN错误，DN[{}]", issueApplyDTO.getCertDn());
                return result;
            }
            issueApplyDTO.setUserId(userId);
            issueApplyDTO.setTempParas(conversion(editCertApplyInfo.getEditIssueApplyInfo().getExtensionList()));
            result = issueApplyService.updateUserCertIssueApply(systemFlag, applyNo, issueApplyDTO, false, true);
        } else if (applyType == ApplyTypeEnum.UPDATE_APPLY.id) {
            UpdateApplyDTO updateApplyDTO = new UpdateApplyDTO();
            BeanUtils.copyProperties(editCertApplyInfo.getEditUpdateApplyInfo(), updateApplyDTO);
            UpdateApplyDTO updateApplyInfoByApplyNo = updateApplyDao.getUpdateApplyInfoByApplyNo(applyNo);
            updateApplyDTO.setTempNo(updateApplyInfoByApplyNo.getTempNo());
            updateApplyDTO.setSignSn(updateApplyInfoByApplyNo.getSignSn());
            updateApplyDTO.setEncSn(updateApplyInfoByApplyNo.getEncSn());
            updateApplyDTO.setUserId(userId);
            if (editCertApplyInfo.getEditUpdateApplyInfo().getUpdateValidity()) {
                result = checkValidity(updateApplyDTO.getCertValidity(), updateApplyDTO.getUpdateKey(), null, updateApplyDTO.getTempNo());
                if (!result.isSuccess()) {
                    return result;
                }
                // 将check之后的结果赋值给申请的证书有效期字段
                updateApplyDTO.setCertValidity((Integer) result.getInfo());

            }
            result = checkDn(updateApplyInfoByApplyNo.getBaseDn(), updateApplyDTO.getCertDn(), userId, applyNo);
            if (!result.isSuccess()) {
                logger.error("用户修改更新申请DN错误，DN[{}]", updateApplyDTO.getCertDn());
                return result;
            }
            result = updateApplyService.updateUserCertUpdateApply(systemFlag, applyNo, updateApplyDTO, true);
        } else {
            RevokeApplyDTO revokeApplyDTO = new RevokeApplyDTO();
            BeanUtils.copyProperties(editCertApplyInfo.getEditRevokeApplyInfo(), revokeApplyDTO);
            if (!RevokeReasonEnum.contain(revokeApplyDTO.getRevokeReason())) {
                return Result.failure(ErrorEnum.REVOKE_REASON_IS_NOT_SUPPORT);
            }
            RevokeApplyDTO revokeApplyInfoByApplyNo = revokeApplyDao.getRevokeApplyInfoByApplyNo(applyNo);
            revokeApplyDTO.setTempNo(revokeApplyInfoByApplyNo.getTempNo());
            revokeApplyDTO.setSignSn(revokeApplyInfoByApplyNo.getSignSn());
            revokeApplyDTO.setEncSn(revokeApplyInfoByApplyNo.getEncSn());
            revokeApplyDTO.setUserId(userId);
            result = revokeApplyService.updateUserCertRevokeApply(systemFlag, applyNo, revokeApplyDTO, true);
        }

        return result;
    }

    private Result getFreezeApplyLaunchResult(String systemFlag, ApplyLaunch applyLaunch) {
        /*if (applyLaunch.getApplyType().equals(ApplyTypeEnum.FREEZE_APPLY.id)) {
            // 证书冻结申请
        } else {
            // 证书解冻申请
        }*/
        Result result;
        ApplyRep applyRep = new ApplyRep();
        CertTempDO certTempInfo;
        FreezeApply freezeApply = applyLaunch.getFreezeApply();
        String signSn = freezeApply.getSignSn();
        FreezeApplyDTO freezeApplyDTO = new FreezeApplyDTO();
        freezeApplyDTO.setSignSn(signSn);
        freezeApplyDTO.setApplyReason(freezeApply.getApplyReason());
        freezeApplyDTO.setUserId(userCertDao.getUserId(signSn));
        Result tempNoResult = this.getTempNo(signSn);
        if (!tempNoResult.isSuccess()) {
            return tempNoResult;
        }
        freezeApplyDTO.setTempNo(tempNoResult.getInfo().toString());
        certTempInfo = certTempDao.getCertTempInfoByTempNo(freezeApplyDTO.getTempNo());
        freezeApplyDTO.setPrivateKeyLength(certTempInfo.getPrivateKeyLength());
        freezeApplyDTO.setSignAlg(certTempInfo.getSignAlg());
        String encSn = userCertDao.getEncSnBySignSn(signSn);
        freezeApplyDTO.setEncSn(encSn == null ? "" : encSn);
        freezeApplyDTO.setApplyType(applyLaunch.getApplyType());
        //  插入冻结解冻记录
        result = freezeApplyService.insertUserCertFreezeApply(systemFlag, freezeApplyDTO, true, true);
        boolean certTempIsAutoStrategy = (certTempInfo.getCheckStrategy() == Constants.TEMP_CHECK_STRATEGY_AUTO_1);
        applyRep.setNeedCheck(!certTempIsAutoStrategy);
        if (result.isSuccess()) {
            applyRep.setApplyNo(result.getInfo().toString());
            result.setInfo(applyRep);
        }
        return result;
    }

    private Result getApplyLaunchResult(String systemFlag, ApplyLaunch applyLaunch, String baseDn) {
        Long userId;
        Result result;
        ApplyRep applyRep = new ApplyRep();
        CertTempDO certTempInfo;
        if (applyLaunch.getApplyType() == ApplyTypeEnum.ISSUE_APPLY.id) {
            IssueApply issueApply = applyLaunch.getIssueApply();
            IssueApplyDTO issueApplyDTO = new IssueApplyDTO();
            BeanUtils.copyProperties(issueApply, issueApplyDTO);
            userId = baseUserService.queryUserId(issueApply.getUserType(), systemFlag, issueApply.getLicenseType(), issueApply.getLicenseNumber());
            if (null == userId) {
                logger.error("该用户不存在：userType:[{}],identType:[{}],licenseNumber:[{}]", issueApply.getUserType(),
                        issueApply.getLicenseType(), issueApply.getLicenseNumber());
                return Result.failure(ErrorEnum.GET_PERSON_USER_INFO_IS_EMPTY);
            }
            result = this.checkDn(baseDn, issueApply.getCertDn(), userId, null);
            if (!result.isSuccess()) {
                logger.error("用户签发申请DN错误，DN[{}]", issueApply.getCertDn());
                return result;
            }
            issueApplyDTO.setUserId(userId);
            issueApplyDTO.setTempParas(conversion(issueApply.getExtensionList()));
            certTempInfo = certTempDao.getCertTempInfoByTempNo(issueApplyDTO.getTempNo());
            issueApplyDTO.setSignAlg(certTempInfo.getSignAlg());
            issueApplyDTO.setPrivateKeyLength(certTempInfo.getPrivateKeyLength());
            issueApplyDTO.setCertValidity(issueApply.getCertValidity());
            result = issueApplyService.insertUserCertIssueApply(systemFlag, issueApplyDTO, true, true);
        } else if (applyLaunch.getApplyType() == ApplyTypeEnum.UPDATE_APPLY.id) {
            UpdateApply updateApply = applyLaunch.getUpdateApply();
            UpdateApplyDTO updateApplyDTO = new UpdateApplyDTO();
            BeanUtils.copyProperties(updateApply, updateApplyDTO);
            userId = userCertDao.getUserId(updateApply.getSignSn());
            updateApplyDTO.setUserId(userId);
            Result tempNoResult = this.getTempNo(updateApply.getSignSn());
            if (!tempNoResult.isSuccess()) {
                return tempNoResult;
            }
            updateApplyDTO.setTempNo(tempNoResult.getInfo().toString());
            certTempInfo = certTempDao.getCertTempInfoByTempNo(updateApplyDTO.getTempNo());

            baseDn = certTempInfo.getBaseDn();
            result = checkDn(baseDn, updateApply.getCertDn(), userId, null);
            if (!result.isSuccess()) {
                logger.error("用户更新申请DN错误，DN[{}]", updateApply.getCertDn());
                return result;
            }

            updateApplyDTO.setSignAlg(certTempInfo.getSignAlg());
            updateApplyDTO.setPrivateKeyLength(certTempInfo.getPrivateKeyLength());
            String encSn = userCertDao.getEncSnBySignSn(updateApply.getSignSn());
            updateApplyDTO.setEncSn(encSn == null ? "" : encSn);
            result = updateApplyService.insertUserCertUpdateApply(systemFlag, updateApplyDTO, true, true);
        } else if (applyLaunch.getApplyType() == ApplyTypeEnum.REVOKE_APPLY.id) {
            RevokeApply revokeApply = applyLaunch.getRevokeApply();
            RevokeApplyDTO revokeApplyDTO = new RevokeApplyDTO();
            BeanUtils.copyProperties(revokeApply, revokeApplyDTO);
            revokeApplyDTO.setUserId(userCertDao.getUserId(revokeApply.getSignSn()));
            Result tempNoResult = this.getTempNo(revokeApply.getSignSn());
            if (!tempNoResult.isSuccess()) {
                return tempNoResult;
            }
            revokeApplyDTO.setTempNo(tempNoResult.getInfo().toString());
            certTempInfo = certTempDao.getCertTempInfoByTempNo(revokeApplyDTO.getTempNo());
            revokeApplyDTO.setPrivateKeyLength(certTempInfo.getPrivateKeyLength());
            revokeApplyDTO.setSignAlg(certTempInfo.getSignAlg());

            String encSn = userCertDao.getEncSnBySignSn(revokeApply.getSignSn());
            revokeApplyDTO.setEncSn(encSn == null ? "" : encSn);
            result = revokeApplyService.insertUserCertRevokeApply(systemFlag, revokeApplyDTO, true, true);
        } else {
            RecoveryApply recoveryApply = applyLaunch.getRecoveryApply();
            RecoveryApplyDTO recoveryApplyDTO = new RecoveryApplyDTO();
            recoveryApplyDTO.setSignSn(recoveryApply.getSignSn());
            recoveryApplyDTO.setApplyReason(recoveryApply.getApplyReason());
            recoveryApplyDTO.setUserId(userCertDao.getUserId(recoveryApply.getSignSn()));
            Result tempNoResult = this.getTempNo(recoveryApply.getSignSn());
            if (!tempNoResult.isSuccess()) {
                return tempNoResult;
            }
            recoveryApplyDTO.setTempNo(tempNoResult.getInfo().toString());
            certTempInfo = certTempDao.getCertTempInfoByTempNo(recoveryApplyDTO.getTempNo());
            recoveryApplyDTO.setPrivateKeyLength(certTempInfo.getPrivateKeyLength());
            recoveryApplyDTO.setSignAlg(certTempInfo.getSignAlg());
            String encSn = userCertDao.getEncSnBySignSn(recoveryApply.getSignSn());
            recoveryApplyDTO.setEncSn(encSn == null ? "" : encSn);
            result = recoveryApplyService.insertUserCertRecoveryApply(systemFlag, recoveryApplyDTO, true);
        }
        boolean certTempIsAutoStrategy = (certTempInfo.getCheckStrategy() == Constants.TEMP_CHECK_STRATEGY_AUTO_1);
        /*boolean flag = certTempInfo.getCheckStrategy() == Constants.TEMP_CHECK_STRATEGY_AUTO_1 ? false : true;*/
        applyRep.setNeedCheck(!certTempIsAutoStrategy);
        if (result.isSuccess()) {
            applyRep.setApplyNo(result.getInfo().toString());
            if (certTempIsAutoStrategy && (applyLaunch.getApplyType() != ApplyTypeEnum.REVOKE_APPLY.id)) {
                applyRep.setDoubleCode(commonService.createDoubleCode(result.getInfo().toString()));
            }
            result.setInfo(applyRep);
        }
        return result;
    }

    /**
     * 扩展项转换
     *
     * @param extensionList
     * @return
     */
    private String conversion(List<Extension> extensionList) {
        if (null == extensionList || extensionList.isEmpty()) {
            return "[]";
        }
        List<ExtensionAttr> extensionAttrs = new ArrayList<>();
        for (Extension extension : extensionList) {
            ExtensionAttr extensionAttr = new ExtensionAttr();
            BeanUtils.copyProperties(extension, extensionAttr);
            String type = "";
            String value = "";
            for (AttrType attrType : extension.getAttrTypeList()) {

                type = type + "#" + attrType.getAttrTypeNo();
                value = value + "#" + attrType.getAttrTypeValue();
            }
            extensionAttr.setAttrType(type.substring(1));
            extensionAttr.setAttrValue(value.substring(1));
            extensionAttrs.add(extensionAttr);
        }
        return JSON.toJSONString(extensionAttrs);
    }

    /**
     * 获取模板编号
     *
     * @param signSn
     * @returnt
     */
    private Result getTempNo(String signSn) {
        UserCertDO userCertDO = userCertDao.getUserCertBaseInfo(signSn);
        if (userCertDO == null) {
            return Result.failure(ErrorEnum.THE_CERT_IS_NOT_EXIT);
        }
        return Result.success(userCertDO.getTempNo());
    }

    /**
     * 校验DN
     *
     * @param dn      待验证证书主体
     * @param baseDn  模板baseDn
     * @param userId  用户id
     * @param applyNo 申请编号
     * @return
     */
    private Result checkDn(String baseDn, String dn, long userId, String applyNo) {
        logger.info("模板获取baseDn为 ： " + baseDn);
        if (null != baseDn && dn.endsWith(baseDn)) {
            return certApplyService.verifyUserCertApplyDn(dn, userId, applyNo);
        } else {
            if (!dn.endsWith(CommonVariable.getRaBaseDn())) {
                return Result.failure(ErrorEnum.CERT_APPLY_DN_IS_ERROR);
            }
            return certApplyService.verifyUserCertApplyDn(dn, userId, applyNo);
        }
    }

    /**
     * 进行格式转换
     *
     * @param json
     * @return
     */
    public List<ExtensionAttr> formatExtensionJon(String json) {
        try {
            List<ExtensionAttr> atrList = JsonMapper.alwaysMapper().fromJson(json, new TypeReference<List<ExtensionAttr>>() {
            });
            return atrList;
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 参数校验
     *
     * @param privateKeyLength
     * @param signAlg
     * @return
     */
    private Result checkParameter(Integer privateKeyLength, String signAlg) {
        Result result = new Result();
        if (null != privateKeyLength) {
            if (!KeyLengthEnum.contain(privateKeyLength)) {
                return Result.failure(ErrorEnum.KEY_LENGTH_IS_NOT_SUPPORT);
            }
        }
        if (null != signAlg) {
            if (!SignAlgEnum.contain(signAlg)) {
                return Result.failure(ErrorEnum.SIGN_ALG_IS_NOT_SUPPORT);
            }
        }
        return result;
    }

    /**
     * 校验申请有效期
     *
     * @param validity
     * @param isUpdateKey
     * @param signSn
     * @param tempNo
     * @return
     */
    private Result checkValidity(Integer validity, boolean isUpdateKey, String signSn, String tempNo) {
        int maxValidity;
        Result result;
        if (validity == null || validity <= 0) {
            return Result.failure(ErrorEnum.APPLY_VALIDITY_IS_NOT_POSITIVE_INTEGER);
        }
        if (null != tempNo) {
            result = certTempService.getCertApplyMaxValidity(tempNo, 1, null);
        } else {
            Result tempNoResult = this.getTempNo(signSn);
            if (!tempNoResult.isSuccess()) {
                return tempNoResult;
            }
            result = certTempService.getCertApplyMaxValidity(tempNoResult.getInfo().toString(), isUpdateKey ? 1 : 0, signSn);
        }
        if (!result.isSuccess()) {
            logger.error("校验申请有效期错误，signSn[{}]", signSn);
            return result;
        }
        maxValidity = (int) result.getInfo();
        if (validity > maxValidity) {
            //return Result.failure(ErrorEnum.APPLY_VALIDITY_IS_TOO_LARGE);
            // 后台自动调整有效期  update by syg 20200922
            logger.info("申请的有效期为:" + validity + " 最大有效期为:" + maxValidity + " 程序自动将申请的最大有效期更新为:" + maxValidity);
            validity = maxValidity;
        }
        return Result.success(validity);
    }
}
