package com.xdja.pki.ra.service.manager.certapply;

import com.xdja.ca.utils.DnUtil;
import com.xdja.pki.ra.core.commonenum.ErrorEnum;
import com.xdja.pki.ra.core.common.Result;
import com.xdja.pki.ra.core.constant.Constants;
import com.xdja.pki.ra.core.util.json.JsonUtils;
import com.xdja.pki.ra.core.util.time.DateUtils;
import com.xdja.pki.ra.manager.dao.ApplyRecordDao;
import com.xdja.pki.ra.manager.dao.BaseUserDao;
import com.xdja.pki.ra.manager.dao.CertApplyDao;
import com.xdja.pki.ra.manager.dao.CertTempDao;
import com.xdja.pki.ra.manager.dao.IssueApplyDao;
import com.xdja.pki.ra.manager.dao.model.*;
import com.xdja.pki.ra.manager.dto.IssueApplyDTO;
import com.xdja.pki.ra.security.bean.Operator;
import com.xdja.pki.ra.security.util.OperatorUtil;
import com.xdja.pki.ra.service.manager.baseuser.bean.DecryptUserInfo;
import com.xdja.pki.ra.service.manager.certapply.bean.IssueApplyVO;
import com.xdja.pki.ra.service.manager.login.bean.CurrentAdminInfo;
import org.bouncycastle.asn1.x500.X500Name;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.naming.NamingException;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;

import static java.lang.System.currentTimeMillis;

/**
 * 签发证书申请服务层-实现类
 *
 * @author syg
 */
@Service
public class IssueApplyServiceImpl implements IssueApplyService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private static final String LOCK_APPLY_NO="applyNo";

    @Autowired
    CertApplyDao certApplyDao;

    @Autowired
    IssueApplyDao issueApplyDao;

    @Autowired
    BaseUserDao baseUserDao;

    @Autowired
    CertTempDao certTempDao;

    @Autowired
    ApplyRecordDao applyRecordDao;

    @Autowired
    CertApplyService certApplyService;

    @Override
    public Result getIssueApplyInfo(String applyNo) {
        Result result = new Result();
        IssueApplyVO issueApplyVO = new IssueApplyVO();
        IssueApplyDTO issueApplyInfoByApplyNo = null;
        try{
            issueApplyInfoByApplyNo = issueApplyDao.getIssueApplyInfoByApplyNo(applyNo);
        }catch (EmptyResultDataAccessException e){
            logger.info("getIssueApplyInfo.applyNo:"+applyNo +" 查询签发证书申请实体为空");
        }
        if (issueApplyInfoByApplyNo == null) {
            logger.info("获取签发证书申请详细信息为空");
            result.setError(ErrorEnum.GET_ISSUE_APPLY_INFO_IS_EMPTY);
            return result;
        }

        BeanUtils.copyProperties(issueApplyInfoByApplyNo,issueApplyVO);
        String licenseNumber = issueApplyVO.getLicenseNumber();
        String decryptLicenseNumber = null;
        try {
            decryptLicenseNumber = DecryptUserInfo.getDecryptString(licenseNumber);
        } catch (Exception e) {
            result.setError(ErrorEnum.DECRYPT_ENCRYPT_INFO_ERROR);
            return result;
        }
        issueApplyVO.setLicenseNumber(decryptLicenseNumber);
        Timestamp gmtCreate = issueApplyInfoByApplyNo.getGmtCreate();
        issueApplyVO.setGmtCreate(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(gmtCreate));
        result.setInfo(issueApplyVO);
        return result;
    }

    @Override
    public Result insertCertIssueApply(IssueApplyDTO issueApplyDTO, CertTempDO certTempInfo) {
        Result result = new Result();
        CertApplyDO certApplyDO = new CertApplyDO();
        // 添加申请基本信息
        //申请编号
        certApplyDO.setApplyNo(issueApplyDTO.getApplyNo());
        certApplyDO.setUserId(issueApplyDTO.getUserId());
        certApplyDO.setApplyType(Constants.CERT_APPLY_TYPE_ISSUE_1);
        certApplyDO.setCertDn(issueApplyDTO.getCertDn());
        certApplyDO.setAdminCertDn("V2X");
        certApplyDO.setGmtCreate(new Timestamp(currentTimeMillis()));
        certApplyDO.setGmtUpdate(new Timestamp(currentTimeMillis()));

        // TBOX申请状态直接为待签发
        certApplyDO.setApplyStatus(Constants.CERT_APPLY_STATUS_NOT_ISSUE_3);
        certApplyDO.setTempId(certTempInfo.getId());
        CertApplyDO addCertApply = certApplyDao.insertCertApply(certApplyDO);
        if (addCertApply == null) {
            logger.info("添加证书申请基本信息失败");
            result.setError(ErrorEnum.INSERT_CERT_APPLY_INFO_FAIL);
            return result;
        }
        // 添加签发申请信息
        IssueApplyDO issueApplyDO = new IssueApplyDO();
        issueApplyDO.setApplyId(addCertApply.getId());
        issueApplyDO.setSignAlg(issueApplyDTO.getSignAlg());
        issueApplyDO.setPrivateKeyLength(issueApplyDTO.getPrivateKeyLength());
        issueApplyDO.setCertValidity(issueApplyDTO.getCertValidity());
        issueApplyDO.setApplyReason(issueApplyDTO.getApplyReason());
        issueApplyDO.setTempParas(issueApplyDTO.getTempParas());
        Date date = new Date();
        issueApplyDO.setGmtUpdate(new Timestamp(date.getTime()));
        issueApplyDO.setGmtCreate(new Timestamp(date.getTime()));
        IssueApplyDO addIssueApply = issueApplyDao.insertIssueApply(issueApplyDO);
        if (addIssueApply == null) {
            logger.info("添加签发申请基本信息失败");
            result.setError(ErrorEnum.INSERT_ISSUE_CERT_APPLY_INFO_FAIL);
            throw new RuntimeException();
        }
        result.setInfo(addCertApply.getId());
        return result;
    }



    @Transactional
    @Override
    public Result insertUserCertIssueApply(String systemFlag, IssueApplyDTO issueApplyDTO, boolean isOnlineIssue, boolean isNormal) {
        Result result = new Result();
        CertApplyDO certApplyDO = new CertApplyDO();
        // 添加申请基本信息
        //申请编号 modify by wangtf nanotime+随机数，解决高并发下申请编号重复问题
        String applyNo;
        synchronized (LOCK_APPLY_NO){
            applyNo = String.valueOf(System.nanoTime())+ (ThreadLocalRandom.current().nextLong(0,9999999) + 90000000);
        }
        certApplyDO.setApplyNo(applyNo);
//        certApplyDO.setApplyNo(DateUtils.getCurrDate(DateUtils.FORMAT_FIVE) + ((int) (Math.random() * 900000) + 100000));
        certApplyDO.setUserId(issueApplyDTO.getUserId());
        certApplyDO.setApplyType(Constants.CERT_APPLY_TYPE_ISSUE_1);

        X500Name x500Name = null;
        try{
            x500Name = DnUtil.getRFC4519X500Name(issueApplyDTO.getCertDn());
        }catch (Exception e){
            logger.info("certDn不正确{}",e.getMessage());
            result.setError(ErrorEnum.CERT_APPLY_DN_IS_ERROR);
            return result;
        }
        certApplyDO.setCertDn(x500Name.toString());
        //审计日志
        try {
            result.setLogContent("，证书主体=" + x500Name.toString() + "，申请编号=" + certApplyDO.getApplyNo());
        } catch (Exception e) {
            logger.info("加入日志审计失败", e);
        }
        // 获取当前登录管理员的信息
        if (!isOnlineIssue){
            Operator operator = OperatorUtil.getOperator();
            if (operator == null || operator.getCurrUser() == null) {
                result.setError(ErrorEnum.CANNOT_FIND_CURRENT_LOGIN_ADMIN);
                return result;
            }
            CurrentAdminInfo currentAdminInfo = operator.getCurrUser();
            List<Integer> roleList = currentAdminInfo.getRoleList();
            // 判断当前登录的管理员是不是拥有录入员角色
            if (!roleList.contains(Constants.ADMIN_ROLE_OPERATOR_INPUT_3)) {
                result.setError(ErrorEnum.CURRENT_ADMIN_ROLE_IS_ERROR);
                return result;
            }
            certApplyDO.setAdminId(currentAdminInfo.getId());
            certApplyDO.setAdminCertDn(currentAdminInfo.getCertDn());
        }
        // 判断当前的用户状态是否支持签发申请
        BaseUserDO baseUserDO = null;
        try {
            baseUserDO = baseUserDao.getBaseUserInfo(issueApplyDTO.getUserId());
        } catch (Exception e) {
            logger.info("获取用户表信息异常");
            result.setError(ErrorEnum.GET_BASE_USER_INFO_IS_EXCEPTION);
            return result;
        }
        if (baseUserDO == null) {
            logger.info("获取用户表信息为空");
            result.setError(ErrorEnum.GET_BASE_USER_INFO_IS_EMPTY);
            return result;
        }
        //审计日志
        result.setLogContent(result.getLogContent() + "，用户ID=" + baseUserDO.getId());
        if (baseUserDO.getStatus() == Constants.USER_STATUS_STOP_1) {
            logger.info("发起证书签发申请的用户已停用");
            result.setError(ErrorEnum.ISSUE_APPLY_USER_STATUS_IS_STOP);
            return result;
        }

        certApplyDO.setGmtCreate(new Timestamp(currentTimeMillis()));
        certApplyDO.setGmtUpdate(new Timestamp(currentTimeMillis()));
        CertTempDO certTempInfo = certTempDao.getCertTempInfoByTempNo(issueApplyDTO.getTempNo());
        if (certTempInfo == null) {
            logger.info("获取证书模板信息为空");
            result.setError(ErrorEnum.GET_CERT_TEMP_INFO_IS_EMPTY);
            return result;
        }
        if (certTempInfo.getTempStatus() != Constants.TEMP_STATUS_RUN_2 || certTempInfo.getTempBound() != Constants.TEMP_BOUND_STATUS_1){
            logger.info("获取证书模板状态不可用"+ JsonUtils.object2Json(certTempInfo));
            result.setError(ErrorEnum.CERT_TEMP_STATUS_IS_NOT_NORMAL);
            return result;
        }

        // 如果审核策略是自动的，则申请状态直接为待签发，否则是待审核状态
        int applyStatus = Constants.CERT_APPLY_STATUS_NOT_CHECK_1;
        if (Constants.TEMP_CHECK_STRATEGY_AUTO_1 == certTempInfo.getCheckStrategy() || (isOnlineIssue&&!isNormal)){
            applyStatus = Constants.CERT_APPLY_STATUS_NOT_ISSUE_3;
        }
        certApplyDO.setApplyStatus(applyStatus);
        certApplyDO.setTempId(certTempInfo.getId());
        CertApplyDO addCertApply = certApplyDao.insertCertApply(certApplyDO);
        if (addCertApply == null) {
            logger.info("添加证书申请基本信息失败");
            result.setError(ErrorEnum.INSERT_CERT_APPLY_INFO_FAIL);
            return result;
        }
        // 添加签发申请信息
        IssueApplyDO issueApplyDO = new IssueApplyDO();
        issueApplyDO.setApplyId(addCertApply.getId());
        issueApplyDO.setSignAlg(issueApplyDTO.getSignAlg());
        issueApplyDO.setPrivateKeyLength(issueApplyDTO.getPrivateKeyLength());
        issueApplyDO.setCertValidity(issueApplyDTO.getCertValidity());
        issueApplyDO.setApplyReason(issueApplyDTO.getApplyReason());
        issueApplyDO.setTempParas(issueApplyDTO.getTempParas());
        Date date = new Date();
        issueApplyDO.setGmtUpdate(new Timestamp(date.getTime()));
        issueApplyDO.setGmtCreate(new Timestamp(date.getTime()));
        IssueApplyDO addIssueApply = issueApplyDao.insertIssueApply(issueApplyDO);
        if (addIssueApply == null) {
            logger.info("添加签发申请基本信息失败");
            result.setError(ErrorEnum.INSERT_ISSUE_CERT_APPLY_INFO_FAIL);
            throw new RuntimeException();
        }

        // 添加申请记录信息
        certApplyService.insertCertApplyRecord(Constants.CERT_APPLY_TYPE_ISSUE_1, Constants.OPERATE_TYPE_ENTRY_1, addCertApply.getApplyNo(),
                systemFlag, Constants.CERT_APPLY_STATUS_NOT_CHECK_1, issueApplyDTO.getApplyReason(), Constants.CERT_APPLY_OPERATE_TYPE_SUBMIT_SUCCESS_1, false,isOnlineIssue);

        // 如果为自动审核的模板，则添加自动审核记录
        if (Constants.TEMP_CHECK_STRATEGY_AUTO_1 == certTempInfo.getCheckStrategy()) {
            certApplyService.insertCertApplyRecord(Constants.CERT_APPLY_TYPE_ISSUE_1, Constants.OPERATE_TYPE_CHECK_3, addCertApply.getApplyNo(),
                    systemFlag, Constants.CERT_APPLY_STATUS_NOT_ISSUE_3, "自动审核类模板-审核成功", Constants.CERT_APPLY_OPERATE_TYPE_CHECK_SUCCESS_4, true,isOnlineIssue);

        }
        result.setInfo(addCertApply.getApplyNo());
        return result;
    }

    @Transactional
    @Override
    public Result updateUserCertIssueApply(String systemFlag, String applyNo, IssueApplyDTO issueApplyDTO,boolean isIgnoreApplyStatus,boolean isOnlineIssue) {
        Result result = new Result();

        // 获取当前登录管理员的信息
        if(!isOnlineIssue){
            Operator operator = OperatorUtil.getOperator();
            if (operator == null || operator.getCurrUser() == null) {
                result.setError(ErrorEnum.CANNOT_FIND_CURRENT_LOGIN_ADMIN);
                return result;
            }
            CurrentAdminInfo currentAdminInfo = operator.getCurrUser();
            List<Integer> roleList = currentAdminInfo.getRoleList();
            // 判断当前登录的管理员是不是拥有录入员角色
            if (!roleList.contains(Constants.ADMIN_ROLE_OPERATOR_INPUT_3)){
                result.setError(ErrorEnum.CURRENT_ADMIN_ROLE_IS_ERROR);
                return result;
            }
        }

        // 判断当前的申请状态是否支持发起更新签发申请
        CertApplyDO certApplyDO = certApplyDao.getCertApplyInfo(applyNo);
        if (certApplyDO == null) {
            logger.info("获取证书申请基本信息为空");
            result.setError(ErrorEnum.GET_CERT_APPLY_INFO_IS_EMPTY);
            return result;
        }
        int nowApplyStatus = certApplyDO.getApplyStatus();
        if (nowApplyStatus != Constants.CERT_APPLY_STATUS_CHECK_FAIL_2){
            logger.info("当前申请状态，不可修改申请信息");
            result.setError(ErrorEnum.APPLY_STATUS_CANNOT_UPDATE_INFO);
            return result;
        }

        // 判断当前的用户状态是否支持签发申请
        BaseUserDO baseUserDO = null;
        try {
            baseUserDO = baseUserDao.getBaseUserInfo(issueApplyDTO.getUserId());
        }catch (Exception e){
            logger.info("获取用户表信息异常");
            result.setError(ErrorEnum.GET_BASE_USER_INFO_IS_EXCEPTION);
            return result;
        }
        if (baseUserDO == null){
            logger.info("获取用户表信息为空");
            result.setError(ErrorEnum.GET_BASE_USER_INFO_IS_EMPTY);
            return result;
        }

        //审计日志
        result.setLogContent("，用户ID="+baseUserDO.getId());

        if (baseUserDO.getStatus() == Constants.USER_STATUS_STOP_1){
            logger.info("修改证书签发申请的用户已停用");
            result.setError(ErrorEnum.UPDATE_ISSUE_APPLY_USER_STATUS_IS_STOP);
            return result;
        }

        long applyId = certApplyDO.getId();

        // 更新签发申请信息
        IssueApplyDO issueApplyDO = issueApplyDao.getIssueApplyInfoByApplyId(applyId);
        if (issueApplyDO == null) {
            logger.info("获取签发证书申请基本信息为空");
            result.setError(ErrorEnum.GET_ISSUE_APPLY_INFO_IS_EMPTY);
            return result;
        }
//        issueApplyDO.setSignAlg(issueApplyDTO.getSignAlg());
//        issueApplyDO.setPrivateKeyLength(issueApplyDTO.getPrivateKeyLength());
        issueApplyDO.setCertValidity(issueApplyDTO.getCertValidity());
        issueApplyDO.setApplyReason(issueApplyDTO.getApplyReason());
        issueApplyDO.setTempParas(issueApplyDTO.getTempParas());
        issueApplyDO.setGmtUpdate(new Timestamp(System.currentTimeMillis()));
        int updateIssueApplyResult = issueApplyDao.updateIssueApply(issueApplyDO);
        if (updateIssueApplyResult <= 0 ) {
            logger.info("更新签发申请基本信息失败");
            result.setError(ErrorEnum.UPDATE_ISSUE_CERT_APPLY_INFO_FAIL);
            return result;
        }

        // 更新申请信息和记录更新记录
        CertTempDO certTempInfo = certTempDao.getCertTempInfoByTempNo(issueApplyDTO.getTempNo());
        if (certTempInfo == null){
            logger.info("获取证书模板信息为空");
            result.setError(ErrorEnum.GET_CERT_TEMP_INFO_IS_EMPTY);
            return result;
        }

        if (certTempInfo.getTempStatus() != Constants.TEMP_STATUS_RUN_2||certTempInfo.getTempBound() != Constants.TEMP_BOUND_STATUS_1){
            logger.info("获取证书模板状态不可用"+ JsonUtils.object2Json(certTempInfo));
            result.setError(ErrorEnum.CERT_TEMP_STATUS_IS_NOT_NORMAL);
            return result;
        }

        // 如果审核策略是自动的，则申请状态直接为待签发，否则是待审核状态
        int applyStatus = Constants.CERT_APPLY_STATUS_NOT_CHECK_1;
        if (Constants.TEMP_CHECK_STRATEGY_AUTO_1 == certTempInfo.getCheckStrategy() || isIgnoreApplyStatus){
            applyStatus = Constants.CERT_APPLY_STATUS_NOT_ISSUE_3;
        }
        String dnName;
        try {
            dnName = DnUtil.getRFC4519X500Name(issueApplyDTO.getCertDn()).toString();
        } catch (NamingException e) {
            result.setError(ErrorEnum.CERT_DN_IS_NOT_FORMAT);
            return result;
        }
        // 更新申请状态
        Result updateCertApplyResult = certApplyService.updateCertApplyInfo(dnName,applyNo,applyStatus,certTempInfo.getId());
        if (updateCertApplyResult.getCode() != 0) {
            logger.info("更新申请信息失败:" + JsonUtils.object2Json(updateCertApplyResult));
            throw new RuntimeException();
        }

        // 添加申请记录
        certApplyService.insertCertApplyRecord(Constants.CERT_APPLY_TYPE_ISSUE_1,Constants.OPERATE_TYPE_UPDATE_2,applyNo, systemFlag, Constants.CERT_APPLY_STATUS_NOT_CHECK_1,issueApplyDTO.getApplyReason(),Constants.CERT_APPLY_OPERATE_TYPE_UPDATE_SUCCESS_2,false,isOnlineIssue);

        // 如果为自动审核的模板，则添加自动审核记录
        if (Constants.TEMP_CHECK_STRATEGY_AUTO_1 == certTempInfo.getCheckStrategy()){
            certApplyService.insertCertApplyRecord(Constants.CERT_APPLY_TYPE_ISSUE_1,Constants.OPERATE_TYPE_CHECK_3,applyNo,systemFlag, Constants.CERT_APPLY_STATUS_NOT_ISSUE_3,"自动审核类模板-审核成功",Constants.CERT_APPLY_OPERATE_TYPE_CHECK_SUCCESS_4,true,isOnlineIssue);
        }
        return result;
    }

}
