package com.xdja.pki.ra.service.manager.certapply;

import com.xdja.ca.utils.DnUtil;
import com.xdja.pki.ra.core.common.Result;
import com.xdja.pki.ra.core.commonenum.ErrorEnum;
import com.xdja.pki.ra.core.constant.Constants;
import com.xdja.pki.ra.core.util.cert.CertUtils;
import com.xdja.pki.ra.core.util.time.DateUtils;
import com.xdja.pki.ra.manager.dao.*;
import com.xdja.pki.ra.manager.dao.model.*;
import com.xdja.pki.ra.manager.dto.RecoveryApplyDTO;
import com.xdja.pki.ra.security.bean.Operator;
import com.xdja.pki.ra.security.util.OperatorUtil;
import com.xdja.pki.ra.service.manager.baseuser.bean.DecryptUserInfo;
import com.xdja.pki.ra.service.manager.certapply.bean.RecoveryApplyVO;
import com.xdja.pki.ra.service.manager.login.bean.CurrentAdminInfo;
import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.style.RFC4519Style;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.env.Environment;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.naming.NamingException;
import java.security.cert.X509Certificate;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;

/**
 * 恢复证书申请实现类
 *
 * @author wly
 */
@Service
public class RecoveryApplyServiceImpl implements RecoveryApplyService {
    private Logger logger = LoggerFactory.getLogger(this.getClass());
    private static final String LOCK_APPLY_NO="applyNo";

    @Autowired
    RecoveryApplyDao recoveryApplyDao;

    @Autowired
    CaCertDao caCertDao;

    @Autowired
    RaCertDao raCertDao;

    @Autowired
    UserCertDao userCertDao;

    @Autowired
    BaseUserDao baseUserDao;

    @Autowired
    CertTempDao certTempDao;

    @Autowired
    CertApplyService certApplyService;

    @Autowired
    CertApplyDao certApplyDao;


    /**
     * 获取恢复证书申请详情
     *
     * @param applyNo
     * @return
     */
    @Override
    public Result getRecoveryApplyInfo(String applyNo) {
        Result result = new Result();

        RecoveryApplyVO recoveryApplyVO = new RecoveryApplyVO();
        RecoveryApplyDTO recoveryApplyDTO = null;
        try{
            recoveryApplyDTO = recoveryApplyDao.getRecoveryApplyInfoByApplyNo(applyNo);
        } catch (EmptyResultDataAccessException e) {
            logger.info("getIssueApplyInfoByApplyNo.applyNo:" + applyNo + " 查询恢复证书申请实体为空");
        }
        if (recoveryApplyDTO == null) {
            logger.info("获取恢复证书申请基本信息为空");
            result.setError(ErrorEnum.GET_RECOVERY_APPLY_INFO_IS_EMPTY);
            return result;
        }
        BeanUtils.copyProperties(recoveryApplyDTO, recoveryApplyVO);
        String decryptLicenseNumber;
        try {
            decryptLicenseNumber = DecryptUserInfo.getDecryptString(recoveryApplyVO.getLicenseNumber());
        } catch (Exception e) {
            result.setError(ErrorEnum.DECRYPT_ENCRYPT_INFO_ERROR);
            return result;
        }
        recoveryApplyVO.setLicenseNumber(decryptLicenseNumber);
        Timestamp gmtCreate = recoveryApplyDTO.getGmtCreate();
        recoveryApplyVO.setGmtCreate(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(gmtCreate));
        result.setInfo(recoveryApplyVO);
        return result;
    }

    /**
     * 插入恢复证书申请记录
     *
     * @param recoveryApplyDTO
     * @return
     */
    @Override
    @Transactional
    public Result insertUserCertRecoveryApply(String systemFlag, RecoveryApplyDTO recoveryApplyDTO, boolean isOnlineRecovery) {
        Result result = new Result();

        CertApplyDO certApplyDO = new CertApplyDO();

        // 获取当前登录管理员的信息
        if (!isOnlineRecovery) {
            Operator operator = OperatorUtil.getOperator();
            CurrentAdminInfo currentAdminInfo = new CurrentAdminInfo();
            if (operator == null || operator.getCurrUser() == null) {
                result.setError(ErrorEnum.CANNOT_FIND_CURRENT_LOGIN_ADMIN);
                return result;
            } else {
                currentAdminInfo = operator.getCurrUser();
                List<Integer> roleList = currentAdminInfo.getRoleList();
                // 判断当前登录的管理员是不是拥有录入员角色
                if (!roleList.contains(Constants.ADMIN_ROLE_OPERATOR_INPUT_3)) {
                    result.setError(ErrorEnum.CURRENT_ADMIN_ROLE_IS_ERROR);
                    return result;
                }
                long adminId = currentAdminInfo.getId();
                String adminCertDn = currentAdminInfo.getCertDn();
                certApplyDO.setAdminId(adminId);
                certApplyDO.setAdminCertDn(adminCertDn);
            }
        }
        // 获取当前RA服务器签名证书DN
        String raServiceDnName = this.getRAServiceDnName();
        if (StringUtils.isBlank(raServiceDnName)) {
            result.setError(ErrorEnum.GET_RA_SERVICE_DN_NAME_ERROR);
            return result;
        }

        // 获取目标CA证书DN
        String caServiceDnName = this.getCAServiceDnName();
        if (StringUtils.isBlank(caServiceDnName)) {
            result.setError(ErrorEnum.GET_CA_SERVICE_DN_NAME_ERROR);
            return result;
        }
        // 判断用户状态 是否支持发起更新申请
        BaseUserDO baseUserDO = null;
        try {
            baseUserDO = baseUserDao.getBaseUserInfo(recoveryApplyDTO.getUserId());
        } catch (Exception e) {
            logger.error("获取用户表信息异常",e);
            result.setError(ErrorEnum.GET_BASE_USER_INFO_IS_EXCEPTION);
            return result;
        }
        if (baseUserDO == null) {
            logger.info("获取用户表信息为空");
            result.setError(ErrorEnum.GET_BASE_USER_INFO_IS_EMPTY);
            return result;
        }

        if (baseUserDO.getStatus() == Constants.USER_STATUS_STOP_1) {
            logger.info("已停用用户不支持密钥恢复申请");
            result.setError(ErrorEnum.RECOVERY_APPLY_USER_STATUS_IS_STOP);
            return result;
        }
        //审计日志
        result.setLogContent("，用户ID=" + baseUserDO.getId());

        // 判断当前用户证书状态  是否支持发起恢复申请
        UserCertDO userCertBaseInfo = userCertDao.getUserCertBaseInfo(recoveryApplyDTO.getSignSn());
        if (userCertBaseInfo == null) {
            logger.info("获取用户证书信息为空");
            result.setError(ErrorEnum.GET_USER_CERT_INFO_IS_EMPTY);
            return result;
        }
        //失效时间 - 现在时间 小于0  已过期
        long time = userCertBaseInfo.getFailureTime().getTime() - System.currentTimeMillis();

        if (Constants.CERT_STATUS_NORMAL_1 != userCertBaseInfo.getCertStatus() || time < 0) {
            logger.info("用户证书不是正常状态");
            result.setError(ErrorEnum.USER_CERT_IS_NOT_NORMAL_STATUS);
            return result;
        }



        // 添加申请基本信息
        //申请编号 modify by wangtf nanotime+随机数，解决高并发下申请编号重复问题
        String applyDO;
        synchronized (LOCK_APPLY_NO){
            applyDO = String.valueOf(System.nanoTime())+ (ThreadLocalRandom.current().nextLong(0,9999999) + 90000000);
        }
        certApplyDO.setApplyNo(applyDO);
//        certApplyDO.setApplyNo(DateUtils.getCurrDate(DateUtils.FORMAT_FIVE) + String.valueOf((int) (Math.random() * 900000) + 100000));
        certApplyDO.setUserId(recoveryApplyDTO.getUserId());
        certApplyDO.setApplyType(Constants.CERT_APPLY_TYPE_RECOVERY_4);
        //certApplyDO.setCertDn(new X500Name(recoveryApplyDTO.getCertDn()).toString());
        try {
            certApplyDO.setCertDn(DnUtil.getRFC4519X500Name(userCertBaseInfo.getCertDn()).toString());
        }catch (NamingException e){
            result.setError(ErrorEnum.CERT_APPLY_DN_IS_ERROR);
            return result;
        }

        result.setInfo(certApplyDO.getApplyNo());
        result.setLogContent("，申请编号=" + certApplyDO.getApplyNo());

        long current = System.currentTimeMillis();
        certApplyDO.setGmtCreate(new Timestamp(current));
        certApplyDO.setGmtUpdate(new Timestamp(current));

        CertTempDO certTempInfo = certTempDao.getCertTempInfoByTempNo(recoveryApplyDTO.getTempNo());
        if (certTempInfo == null) {
            logger.info("证书发起恢复时，获取证书模板信息为空");
            result.setError(ErrorEnum.GET_CERT_TEMP_INFO_IS_EMPTY);
            return result;
        }

        logger.info("证书发起恢复时，对应模板编号为{},审核策略为【{}】", certTempInfo.getTempNo(), certTempInfo.getCheckStrategy() == 1 ? "自动" : "手动");

        // 如果审核策略是自动的，则申请状态直接为待签发，否则是待审核状态
        int applyStatus = Constants.CERT_APPLY_STATUS_NOT_CHECK_1;
        if (Constants.TEMP_CHECK_STRATEGY_AUTO_1 == certTempInfo.getCheckStrategy()) {
            applyStatus = Constants.CERT_APPLY_STATUS_NOT_ISSUE_3;
        }
        certApplyDO.setApplyStatus(applyStatus);
        certApplyDO.setTempId(certTempInfo.getId());

        CertApplyDO addCertApply = certApplyDao.insertCertApply(certApplyDO);
        if (addCertApply == null) {
            logger.info("添加证书申请基本信息失败");
            result.setError(ErrorEnum.INSERT_CERT_APPLY_INFO_FAIL);
            return result;
        }

        RecoveryApplyDO recoveryApplyDO = new RecoveryApplyDO();
        BeanUtils.copyProperties(recoveryApplyDTO, recoveryApplyDO);
        recoveryApplyDO.setApplyId(addCertApply.getId());
        recoveryApplyDO.setGmtCreate(new Timestamp(System.currentTimeMillis()));
        recoveryApplyDO.setGmtUpdate(new Timestamp(System.currentTimeMillis()));
        recoveryApplyDO.setSignAlg(userCertBaseInfo.getSignAlg());
        recoveryApplyDO.setCertValidity(userCertBaseInfo.getCertValidity());
        recoveryApplyDO.setPrivateKeyLength(userCertBaseInfo.getPrivateKeyLength());

        RecoveryApplyDO addRecoveryApply = recoveryApplyDao.insertRecoveryApply(recoveryApplyDO);

        if (addRecoveryApply == null) {
            logger.info("添加恢复申请基本信息失败");
            result.setError(ErrorEnum.INSERT_RECOVERY_CERT_APPLY_INFO_FAIL);
            throw new RuntimeException();
        }

        // 添加申请记录信息
        certApplyService.insertCertApplyRecord(Constants.CERT_APPLY_TYPE_RECOVERY_4, Constants.OPERATE_TYPE_ENTRY_1, addCertApply.getApplyNo(), systemFlag, Constants.CERT_APPLY_STATUS_NOT_CHECK_1, recoveryApplyDTO.getApplyReason(), Constants.CERT_APPLY_OPERATE_TYPE_SUBMIT_SUCCESS_1, false, isOnlineRecovery);
        // 如果为自动审核的模板，则添加自动审核记录
        if (Constants.TEMP_CHECK_STRATEGY_AUTO_1 == certTempInfo.getCheckStrategy()) {
            certApplyService.insertCertApplyRecord(Constants.CERT_APPLY_TYPE_RECOVERY_4, Constants.OPERATE_TYPE_CHECK_3, addCertApply.getApplyNo(), systemFlag, Constants.CERT_APPLY_STATUS_NOT_ISSUE_3, "自动审核类模板-审核成功", Constants.CERT_APPLY_OPERATE_TYPE_CHECK_SUCCESS_4, true, isOnlineRecovery);
        }

        return result;
    }


    /**
     * 获取系统RA服务器签名证书DN
     *
     * @return
     */
    private String getRAServiceDnName() {
        RaCertDO newRaCertInfo = null;
        try {
            newRaCertInfo = raCertDao.getNewRaCertInfo();
        } catch (Exception e) {
            logger.error("获取RA服务器证书异常{}", e);
            return null;
        }

        if (newRaCertInfo == null) {
            return null;
        }
        String certInfo = newRaCertInfo.getCertInfo();
        X509Certificate certFromStr = CertUtils.getCertFromStr(certInfo);
        String raServiceDnName = CertUtils.getSubjectByX509Cert(certFromStr);
        return raServiceDnName;
    }

    /**
     * 获取系统CA服务器证书DN
     *
     * @return
     */
    private String getCAServiceDnName() {
        CaCertDO newCaCertInfo = null;
        try {
            newCaCertInfo = caCertDao.getNewCaCertInfo();
        } catch (Exception e) {
            logger.error("获取CA服务器证书异常{}", e);
            return null;
        }

        if (newCaCertInfo == null) {
            return null;
        }
        String certInfo = newCaCertInfo.getCertInfo();
        X509Certificate certFromStr = CertUtils.getCertFromStr(certInfo);
        String caServiceDnName = CertUtils.getSubjectByX509Cert(certFromStr);
        return caServiceDnName;
    }

}
