package com.xdja.pki.ra.service.manager.customer;

import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.X509Certificate;
import java.security.spec.PKCS8EncodedKeySpec;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.regex.Pattern;

import com.xdja.ca.constant.SdkConstants;
import com.xdja.ca.utils.DnUtil;
import com.xdja.pki.ra.core.common.OperatorCertCache;
import com.xdja.pki.ra.core.util.cert.CertDnVerifyUtils;
import com.xdja.pki.ra.manager.dao.RaCertDao;
import com.xdja.pki.ra.manager.dao.model.RaCertDO;
import com.xdja.pki.ra.manager.dto.CustomerCertDTO;
import com.xdja.pki.ra.manager.sdk.business.CaBusinessManager;
import com.xdja.pki.ra.service.manager.certapply.CertApplyService;
import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import com.xdja.ca.vo.UserCertInfo;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.ra.core.asn1.RsaObjectIdentifiers;
import com.xdja.pki.ra.core.common.CommonVariable;
import com.xdja.pki.ra.core.common.Result;
import com.xdja.pki.ra.core.commonenum.ErrorEnum;
import com.xdja.pki.ra.core.constant.Constants;
import com.xdja.pki.ra.core.constant.PathConstants;
import com.xdja.pki.ra.core.exception.DAOException;
import com.xdja.pki.ra.core.exception.ServiceException;
import com.xdja.pki.ra.core.util.cert.CertUtils;
import com.xdja.pki.ra.core.util.cert.KeyStoreUtils;
import com.xdja.pki.ra.core.util.cert.KeyUtils;
import com.xdja.pki.ra.core.util.cert.RandomUtils;
import com.xdja.pki.ra.core.util.json.JsonUtils;
import com.xdja.pki.ra.manager.dao.CaCertDao;
import com.xdja.pki.ra.manager.dao.CertTempDao;
import com.xdja.pki.ra.manager.dao.CustomerCertDao;
import com.xdja.pki.ra.manager.dao.CustomerDao;
import com.xdja.pki.ra.manager.dao.CustomerTempDao;
import com.xdja.pki.ra.manager.dao.model.CaCertDO;
import com.xdja.pki.ra.manager.dao.model.CertTempDO;
import com.xdja.pki.ra.manager.dao.model.CustomerSysCertDO;
import com.xdja.pki.ra.manager.dao.model.CustomerSysDO;
import com.xdja.pki.ra.manager.dao.model.CustomerSysTempDO;
import com.xdja.pki.ra.manager.page.PageInfo;
import com.xdja.pki.ra.manager.sdk.cmp.CertLifeCycleManager;
import com.xdja.pki.ra.service.manager.customer.bean.CustomerCertIssueApplyVO;
import com.xdja.pki.ra.service.manager.customer.bean.CustomerSysCertVO;
import com.xdja.pki.ra.service.manager.customer.bean.CustomerSysResp;
import com.xdja.pki.ra.service.manager.customer.bean.CustomerSysVO;
import com.xdja.pki.ra.service.manager.customer.bean.CustomerTemplateRepVO;
import com.xdja.pki.ra.service.manager.customer.bean.CustomerTemplateSaveVO;
import com.xdja.pki.ra.service.manager.organuser.bean.OrganParamsCheck;
import com.xdja.pki.ra.service.manager.system.CaService;
import com.xdja.pki.ra.service.manager.system.RaServer;

/**
 * @Classname CustomerSysServiceImpl
 * @Description TODO
 * @Date 2019/10/31 9:57
 * @Author cl
 */
@Service
public class CustomerSysServiceImpl implements CustomerSysService {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private CustomerDao customerDao;

    @Autowired
    private CustomerCertDao customerCertDao;

    @Autowired
    private CertTempDao certTempDao;

    @Autowired
    private CustomerTempDao customerTempDao;

    @Autowired
    private RaServer raServer;

    @Autowired
    private CaService caService;

    @Autowired
    private CertApplyService certApplyService;

    @Autowired
    private CaCertDao caCertDao;


    @Autowired
    RaCertDao raCertDao;

    @Autowired
    private CertLifeCycleManager certLifeCycleManager;

    @Autowired
    private CaBusinessManager caBusinessManager;

    @Override
    public Result registerCustomerInfo(CustomerSysVO customerSysVO) {
        Result result = new Result();
        //系统标识符校验
        int count = customerDao.getSystemIdentifier(customerSysVO.getCustomerSysNumber());
        if (count >= 1) {
            result.setError(ErrorEnum.SYS_ID_HAS_EXISTED);
            return result;
        }
        //联系电话校验
        if(!StringUtils.isAnyBlank(customerSysVO.getCustomerSysContact())) {
        	boolean check = OrganParamsCheck.isCheck(OrganParamsCheck.TELNUMBER_NUMBER_PATTERN, customerSysVO.getCustomerSysContact());
        	if (!check) {
        		logger.info("联系电话格式错误");
        		return Result.failure(ErrorEnum.ILLEGAL_REQUEST_PARAMETER);
        	}
        }
        CustomerSysDO customerSysDO = new CustomerSysDO();
        BeanUtils.copyProperties(customerSysVO, customerSysDO);
        Date date = new Date(System.currentTimeMillis());
        customerSysDO.setGmtCreate(date);
        customerSysDO.setGmtUpdate(date);
        customerDao.insertSystemInfo(customerSysDO);
        return Result.success();
    }

    @Override
    public Result getCustomerInfo(Integer id) {
        Result result = new Result();
        CustomerSysDO systemInfo = customerDao.getSystemInfo(id);
        result.setInfo(systemInfo);
        return result;
    }

    @Override
    public Result updateCustomerInfo(Integer id, CustomerSysVO customerSysVO) {
        Result result = new Result();
        
        CustomerSysDO customerSysDO = customerDao.getSystemInfo(id);
        
        if(!customerSysDO.getCustomerSysNumber().equals(customerSysVO.getCustomerSysNumber())) {
        	return Result.failure(ErrorEnum.ILLEGAL_REQUEST_PARAMETER);
        }
        
        //联系电话校验
        if(!StringUtils.isAnyBlank(customerSysVO.getCustomerSysContact())) {
        	boolean check = OrganParamsCheck.isCheck(OrganParamsCheck.TELNUMBER_NUMBER_PATTERN, customerSysVO.getCustomerSysContact());
        	if (!check){
        		logger.info("联系电话格式错误");
        		return Result.failure(ErrorEnum.ILLEGAL_REQUEST_PARAMETER);
        	}
        }
        //获取原信息
        CustomerSysDO systemInfo = customerDao.getSystemInfo(id);
        //更新数据
        BeanUtils.copyProperties(customerSysVO, systemInfo);
        Date date = new Date(System.currentTimeMillis());
        systemInfo.setGmtUpdate(date);
        int i = customerDao.updateSystemInfo(systemInfo);
        if (i == 0) {
            result.setError(ErrorEnum.SYS_UPDATE_IS_ERROR);
            return result;
        }
        return Result.success();
    }

    @Override
    public Result getCustomerInfoList(String systemNumber, String systemName, int pageSize, int pageNo) {
        CustomerSysResp customerSysResp = new CustomerSysResp();
        Result result = new Result();
        PageInfo<CustomerSysDO> systemInfoPageList = customerDao.getSystemInfoList(systemNumber, systemName, pageSize, pageNo);

        List<CustomerSysVO> customerSysVOList = new ArrayList<>();

        List<CustomerSysDO> customerPagelist = systemInfoPageList.getList();
        if (CollectionUtils.isEmpty(customerPagelist)) {
            customerSysResp.setRecordCount(0);
            customerSysResp.setPageCount(0);
            customerSysResp.setDatas(customerSysVOList);
            result.setInfo(customerSysResp);
            return result;
        }
        for (CustomerSysDO customerSysDO : customerPagelist) {
            CustomerSysVO deviceUserVO = new CustomerSysVO();
            BeanUtils.copyProperties(customerSysDO, deviceUserVO);
            Date gmtCreate = customerSysDO.getGmtCreate();
            deviceUserVO.setId(Integer.valueOf((int) customerSysDO.getId()));
            deviceUserVO.setGmtCreate(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(gmtCreate));
            customerSysVOList.add(deviceUserVO);
        }
        customerSysResp.setDatas(customerSysVOList);
        customerSysResp.setPageCount(systemInfoPageList.getPageCount());
        customerSysResp.setRecordCount(systemInfoPageList.getRecordCount());
        result.setInfo(customerSysResp);
        return result;
    }

    @Override
    public Result getCustomerCertInfoList(String systemName, String certDn, String status, String order, int pageSize, int pageNo) {
        CustomerSysResp customerSysResp = new CustomerSysResp();
        Result result = new Result();
        PageInfo<CustomerCertDTO> systemInfoPageList = customerCertDao.getSystemCertInfoList(systemName, certDn, status, order, pageSize, pageNo);

        List<CustomerSysCertVO> customerSysVOList = new ArrayList<>();

        List<CustomerCertDTO> customerPagelist = systemInfoPageList.getList();
        if (CollectionUtils.isEmpty(customerPagelist)) {
            customerSysResp.setRecordCount(0);
            customerSysResp.setPageCount(0);
            customerSysResp.setDatas(customerSysVOList);
            result.setInfo(customerSysResp);
            return result;
        }
        for (CustomerCertDTO customerSysCertDO : customerPagelist) {
            CustomerSysCertVO certVO = new CustomerSysCertVO();
            BeanUtils.copyProperties(customerSysCertDO, certVO);

            //当前时间
            long nowDateMillis = System.currentTimeMillis();
            long time = customerSysCertDO.getFailureTime().getTime();
            if (time < nowDateMillis) {
                certVO.setCertStatus(Constants.CERT_STATUS_EXPIRED_2);
            }

            //判断是否一个月内过期
            Calendar ca = Calendar.getInstance();
            long timeInMillis = ca.getTimeInMillis();
            ca.add(Calendar.MONTH, 1);
            if (time > timeInMillis && ca.getTimeInMillis() > time) {
                certVO.setOutStatus(true);
            } else {
                certVO.setOutStatus(false);
            }

            certVO.setEffectiveTime(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(customerSysCertDO.getEffectiveTime()));
            certVO.setFailureTime(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(customerSysCertDO.getFailureTime()));
            customerSysVOList.add(certVO);
        }
        customerSysResp.setDatas(customerSysVOList);
        customerSysResp.setPageCount(systemInfoPageList.getPageCount());
        customerSysResp.setRecordCount(systemInfoPageList.getRecordCount());
        result.setInfo(customerSysResp);
        return result;
    }

    @Override
    public Result getCustomerTemplateById(Long id) {
        List<CustomerTemplateRepVO> customerTemplateRepVOList = new ArrayList<CustomerTemplateRepVO>();
        List<CertTempDO> allNewCertTemplateList = null;
        List<CustomerSysTempDO> customerTempInfoById = null;

        try {
            //获取最新模板信息
            allNewCertTemplateList = certTempDao.getAllNewCertTemplateList();
            //获取第三方系统绑定的模板信息
            customerTempInfoById = customerTempDao.getCustomerTempInfoById(id);
        } catch (DAOException e) {
            throw new ServiceException("查询第三方授权模板信息异常", e);
        }


        if (null != allNewCertTemplateList && !allNewCertTemplateList.isEmpty()) {
            for (CertTempDO certTempDO : allNewCertTemplateList) {
                CustomerTemplateRepVO customerTemplateRepVO = new CustomerTemplateRepVO();
                customerTemplateRepVO.setId(certTempDO.getId());
                customerTemplateRepVO.setTempNo(certTempDO.getTempNo());
                customerTemplateRepVO.setName(certTempDO.getTempName());
                customerTemplateRepVO.setStatus(certTempDO.getTempStatus());

                int status = certTempDO.getTempStatus();
                boolean isNotCancel = true;
                if (status == Constants.TEMP_STATUS_STOP_3) {
                    isNotCancel = false;
                }

                if (null != customerTempInfoById && !customerTempInfoById.isEmpty()) {
                    for (CustomerSysTempDO customerTempInfo : customerTempInfoById) {
                        if (customerTempInfo.getTempNo().equals(certTempDO.getTempNo())) {
                            customerTemplateRepVO.setBound(true);
                            isNotCancel = true;
                        }
                    }
                }
                if (isNotCancel) {
                    customerTemplateRepVOList.add(customerTemplateRepVO);
                }

            }
        }
        return Result.success(customerTemplateRepVOList);
    }

    @Override
    public Result saveCustomerTemplates(CustomerTemplateSaveVO customerTemplateSaveVO) {
        Result result =new Result();
        long customsysId = customerTemplateSaveVO.getCustomsysId();
        String templateNos = customerTemplateSaveVO.getTemplateNos();
        String[] customerTemplateNos = null;
        String customerTemplateNoStr = "";

        try {
            if (!StringUtils.isEmpty(templateNos)) {
                customerTemplateNos = templateNos.split("#");
                if (customerTemplateNos.length > 0) {
                    customerTemplateNoStr = Arrays.toString(customerTemplateNos);

                    //获取所有用户模板信息

                    List<CertTempDO> allNewCertTemplateList = certTempDao.getAllNewCertTemplateList();

                    for (String customerTemplateNo : customerTemplateNos) {
                        boolean isCustomerTemplateExist = false;
                        if (null != allNewCertTemplateList && !allNewCertTemplateList.isEmpty()) {
                            for (CertTempDO certTempDO : allNewCertTemplateList) {
                                if (customerTemplateNo.equals(certTempDO.getTempNo())) {
                                    isCustomerTemplateExist = true;
                                    break;
                                }
                            }

                            if (!isCustomerTemplateExist) {
                                return Result.failure(ErrorEnum.TEMPLATEID_NOT_EXIST);
                            }
                        }
                    }
                }
            }

            //获取第三方模板中已停用的模板no
            List<CustomerSysTempDO> customerSysTempDOStopeds = customerTempDao.getCustomerTemplateByStatus(customsysId, Constants.TEMP_STATUS_STOP_3);

            for (CustomerSysTempDO customerSysTempDOStoped : customerSysTempDOStopeds) {
                String customerSysTempDOStopedTempNo = customerSysTempDOStoped.getTempNo();
                boolean ishasTemplateStop = customerTemplateNoStr.contains(customerSysTempDOStopedTempNo);
                if (!ishasTemplateStop) {
                    return Result.failure(ErrorEnum.TEMPLATE_STOPED_CANNOT_CANCEL);
                }
            }

            //重置第三方授权模板
            customerTempDao.resetCustomerTemplate(customsysId);

            //更新授权模板信息
            if (null != customerTemplateNos && customerTemplateNos.length > 0) {
                CustomerSysTempDO customerSysTempDO = new CustomerSysTempDO();
                customerSysTempDO.setCustomerSysId(customsysId);
                Arrays.asList(customerTemplateNos).forEach(customerTemplateNo -> {
                    customerSysTempDO.setTempNo(customerTemplateNo);
                    customerSysTempDO.setGmtCreate(new Date(System.currentTimeMillis()));
                    customerTempDao.saveCustomerTemplate(customerSysTempDO);
                });
            }

            CustomerSysDO systemInfo = customerDao.getSystemInfo((int) customsysId);
            String customerSysNumber = systemInfo.getCustomerSysNumber();

            result.setLogContent("，系统标识=" + customerSysNumber + "，模板编号tempNos=" + templateNos);
        } catch (DAOException e) {
            throw new ServiceException("保存第三方授权模板服务失败，", e);
        }

        return result;
    }

    @Override
    public Result issCustomerCertByKeyStore(CustomerCertIssueApplyVO customerCertIssueApplyVO) {
        Result result = new Result();

        // 获取当前RA服务器签名证书DN
        String raServiceDnName = raServer.getRAServiceDnName();
        if (StringUtils.isBlank(raServiceDnName)) {
            result.setError(ErrorEnum.GET_RA_SERVICE_DN_NAME_ERROR);
            return result;
        }

        // 获取目标CA证书DN
        String caServiceDnName = caService.getCAServiceDnName();
        if (StringUtils.isBlank(caServiceDnName)) {
            result.setError(ErrorEnum.GET_CA_SERVICE_DN_NAME_ERROR);
            return result;
        }

        int certValidity = customerCertIssueApplyVO.getCertValidity();
        String signAlg = customerCertIssueApplyVO.getSignAlg();
        String tempNo = customerCertIssueApplyVO.getTempNo();
        String tempParas = null;
        if (StringUtils.isNotBlank(customerCertIssueApplyVO.getTempParas())) {
            tempParas = customerCertIssueApplyVO.getTempParas();
        }

        String customerCertDN = customerCertIssueApplyVO.getCertDn();

        KeyPair signKeyPair = null;
        String keyAlgName = null;
        int keyLength = 0;
        if (Constants.SIGN_ALG_NAME_SM3_WHIT_SM2.equalsIgnoreCase(signAlg)) {
            keyAlgName = Constants.KEY_ALG_NAME_SM2;
            keyLength = 256;
        }  else if(Constants.SIGN_ALG_NAME_SHA256_WHIT_ECDSA.equalsIgnoreCase(signAlg)){
            keyAlgName = Constants.KEY_ALG_NAME_NIST;
            keyLength = 256;
        }else {
            keyAlgName = Constants.KEY_ALG_NAME_RSA;
            keyLength = customerCertIssueApplyVO.getPrivateKeyLength();
        }


        logger.info("密钥算法：" + keyAlgName + " 密钥长度：" + keyLength);

        String customerTransId = RandomUtils.getUUID();

        try {
            signKeyPair = KeyUtils.genKeyPair(keyAlgName, keyLength);
        } catch (Exception e) {
            logger.info("生成用户签名公私钥失败");
            result.setError(ErrorEnum.GEN_USER_CERT_SIGN_KEY_FAIL);
            return result;
        }

        Result issueResult = certLifeCycleManager.issueUserCert(null, null, signKeyPair.getPublic().getEncoded(), raServiceDnName, caServiceDnName, customerTransId, tempNo, tempParas, signAlg, certValidity, customerCertDN,null);
        if (!issueResult.isSuccess()) {
            logger.info("签发第三方系统证书失败:【" + "code:" + issueResult.getErrorBean().getErrCode() + ";desc:" + issueResult.getErrorBean().getErrMsg() + "】");
            result.setErrorBean(issueResult.getErrorBean());
            return result;
        }

        UserCertInfo customerCertInfo = (UserCertInfo) issueResult.getInfo();

        // 第三方系统证书入库
        Result certInsertResult = this.insertCustomerCertInfo(customerCertIssueApplyVO, customerCertInfo);
        if (!certInsertResult.isSuccess()) {
            logger.info("将用户证书插入数据库失败");
            result.setError(certInsertResult.getError());
            return result;
        }

        CertTempDO certTempDO = certTempDao.getCertTempInfoByTempNo(tempNo);
        PrivateKey encPriKey = null;
        if(SdkConstants.CERT_TYPE_SINGLE_1 != certTempDO.getCertPatterm()) {
            // 用私钥解密用户加密私钥信封
            try {
                logger.info("签名私钥：" + Base64.toBase64String(signKeyPair.getPrivate().getEncoded()) + " 私钥信封：" + customerCertInfo.getEncPriKey());
                String encPriKeyStr = CertUtils.getDataFromSignedAndEnvelopedDataByBc(signKeyPair.getPrivate().getEncoded(), customerCertInfo.getEncPriKey().getBytes(), CommonVariable.getKeyAlgName());
                logger.info("加密私钥：" + encPriKeyStr);
                byte[] decode = Base64.decode(encPriKeyStr);
                if (CommonVariable.getKeyAlgName().equalsIgnoreCase(Constants.KEY_ALG_NAME_SM2)) {
                    byte[] data = new byte[32];
                    System.arraycopy(decode, 32, data, 0, 32);
                    encPriKey = GMSSLX509Utils.convertSM2PrivateKey(data);
                } else if (CommonVariable.getKeyAlgName().equalsIgnoreCase(Constants.KEY_ALG_NAME_RSA)){
                    KeyFactory keyFactory = KeyFactory.getInstance(RsaObjectIdentifiers.rsaAlg.getId(), new BouncyCastleProvider());
                    encPriKey = keyFactory.generatePrivate(new PKCS8EncodedKeySpec(decode));
                } else {
                    byte[] data = new byte[32];
                    System.arraycopy(decode, 32, data, 0, 32);
                    encPriKey = GMSSLX509Utils.convertPrivateKeyEncode(data);
                }
            } catch (Exception e) {
                logger.error("从P7b格式中获取加密私钥失败", e);
                result.setError(ErrorEnum.GET_ENC_PUBLIC_KEY_FROM_P7B_FAIL);
                return result;
            }
        }
        // 组装用户PFX或JKS证书
        try {
            List<X509Certificate> trustCaCerts = CommonVariable.getTrustCaCerts();
            for (X509Certificate trustCaCert : trustCaCerts) {
                logger.info("trustCaCert====issuer:" + CertUtils.getIssuerByX509Cert(trustCaCert) + " subjcet:" + CertUtils.getSubjectByX509Cert(trustCaCert));
            }
            int length = trustCaCerts.size();
            X509Certificate[] caCerts = new X509Certificate[length];
            trustCaCerts.toArray(caCerts);

            for (int i = 0; i < caCerts.length; i++) {
                X509Certificate caCert = caCerts[i];
                logger.info("caCert====issuer:" + CertUtils.getIssuerByX509Cert(caCert) + " subjcet:" + CertUtils.getSubjectByX509Cert(caCert));
            }

            X509Certificate[] signCert = new X509Certificate[caCerts.length + 1];
            X509Certificate customerSignCert = CertUtils.getCertFromStr(customerCertInfo.getSignCert());
            signCert[0] = CertUtils.getCertFromStr(customerCertInfo.getSignCert());

            for (int i = 1; i < signCert.length; i++) {
                signCert[i] = caCerts[i - 1];
            }

            for (int i = 0; i < signCert.length; i++) {
                X509Certificate x509Certificate = signCert[i];
                logger.info("signCert====issuer:" + CertUtils.getIssuerByX509Cert(x509Certificate) + " subjcet:" + CertUtils.getSubjectByX509Cert(x509Certificate));
            }

            X509Certificate[] encCert = null;
            if (SdkConstants.CERT_TYPE_SINGLE_1 != certTempDO.getCertPatterm()) {
                encCert = new X509Certificate[caCerts.length + 1];
                encCert[0] = CertUtils.getCertFromStr(customerCertInfo.getEncCert());
                for (int i = 1; i < encCert.length; i++) {
                    encCert[i] = caCerts[i - 1];
                }
            }
            KeyStoreUtils.generateDoubleCertByKeyStore(true, signCert, encCert, signKeyPair.getPrivate(), encPriKey, Constants.PFX_PRIVATE_KEY_STORE_TYPE_1, customerSignCert.getSerialNumber().toString(16), PathConstants.USER_CERT_KEYSTORE_FILE_PATH);

            Map<String, String> resultMap = new HashMap<String, String>();
            resultMap.put("transId", customerTransId);
            resultMap.put("customerSystemFlag", customerCertIssueApplyVO.getCustomerSysNumber());
            resultMap.put("certSn", customerSignCert.getSerialNumber().toString(16));
            result.setInfo(resultMap);

            if(null != signCert[0]){
                result.setLogContent("，签名证书SN=" + signCert[0].getSerialNumber().toString(16).toLowerCase());
            }
            return result;

        } catch (Exception e) {
            logger.error("生成用户keyStore类型证书失败", e);
            result.setError(ErrorEnum.GEN_USER_CERT_KEY_STORE_FILE_FAIL);
            return result;
        }

    }

    @Override
    public Result getSysCertBySysNumber(String sysNumber, String certSn) {
        Result result = new Result();
        CustomerSysCertDO customerSysCert = null;
        PublicKey publicKey;
        try {
            customerSysCert = customerDao.getSysCertBySysNumber(sysNumber, certSn);
           if( customerSysCert.getCertStatus() != Constants.CERT_STATUS_NORMAL_1){
               logger.info("系统证书状态不正常");
               result.setError(ErrorEnum.CUSTOMER_SYS_CERT_IS_NOT_NORMAL);
               return result;
           }
            X509Certificate normalCert = CertUtils.getCertFromStr(customerSysCert.getCertInfo());
            publicKey = normalCert.getPublicKey();
        } catch (DataAccessException e) {
            logger.info("查询证书信息结果不为1", e + "certSn : " + certSn);
            result.setError(ErrorEnum.GET_CERT_INFO_BY_SYSNUMBER_IS_EMPTY);
            return result;
        } catch (Exception e) {
            logger.info("获取证书信息异常", e);
            result.setError(ErrorEnum.GET_CERT_INFO_BY_SYSNUMBER_ERROR);
            return result;
        }

        result.setInfo(publicKey);
        return result;
    }

    @Override
    public Result genCertConfirm(String transId, String signSn) {
        Result result = new Result();
        // 获取当前RA服务器签名证书DN
        String raServiceDnName = this.getRAServiceDnName();
        if (StringUtils.isBlank(raServiceDnName)) {
            result.setError(ErrorEnum.GET_RA_SERVICE_DN_NAME_ERROR);
            return result;
        }

        // 获取目标CA证书DN
        String caServiceDnName = this.getCAServiceDnName();
        if (StringUtils.isBlank(caServiceDnName)) {
            result.setError(ErrorEnum.GET_CA_SERVICE_DN_NAME_ERROR);
            return result;
        }
        // 调用sdk发起签发证书响应
        Result certRespResult = certLifeCycleManager.issueCertResp(raServiceDnName, caServiceDnName, Constants.APPLY_USER_TYPE_NORMAL_USER_1, transId);
        if (!certRespResult.isSuccess()) {
            logger.info("签发第三方系统证书响应");
            result.setErrorBean(certRespResult.getErrorBean());
            return result;
        }

        // 更新证书状态
        try {
            customerCertDao.updateCustomerCertStatus(Constants.CERT_STATUS_NORMAL_1, signSn);
        } catch (Exception e) {
            logger.error("更新第三方系统证书状态异常{}", e);
            result.setError(ErrorEnum.UPDATE_USER_CERT_STATUS_EXCEPTION);
            return result;
        }
        return result;
    }

    @Override
    public Result genErrorMsgContent(String transId, int errorCode, String errorMsg) {
        Result result = new Result();
        // 获取当前RA服务器签名证书DN
        String raServiceDnName = this.getRAServiceDnName();
        if (StringUtils.isBlank(raServiceDnName)) {
            result.setError(ErrorEnum.GET_RA_SERVICE_DN_NAME_ERROR);
            return result;
        }

        // 获取目标CA证书DN
        String caServiceDnName = this.getCAServiceDnName();
        if (StringUtils.isBlank(caServiceDnName)) {
            result.setError(ErrorEnum.GET_CA_SERVICE_DN_NAME_ERROR);
            return result;
        }

        // 获取缓存的管理员证书信息
        Map<String, Object> map = OperatorCertCache.getMap();
        map.remove(transId);

        Result errResult = certLifeCycleManager.sendErrorCMPMessage(raServiceDnName, caServiceDnName, Constants.APPLY_USER_TYPE_ADMIN_2, transId, errorCode, errorMsg);
        if (!errResult.isSuccess()) {
            logger.info("错误确认消息出错：" + JsonUtils.object2Json(errResult));
            result.setErrorBean(errResult.getErrorBean());
            return result;
        }

        return result;
    }

    @Override
    public Result revokeCustomerCert(String signSn, String certDn, int revokeType, String revokeReason) {
        Result result = new Result();
        // 获取当前RA服务器签名证书DN
        String raServiceDnName = this.getRAServiceDnName();
        if (StringUtils.isBlank(raServiceDnName)) {
            result.setError(ErrorEnum.GET_RA_SERVICE_DN_NAME_ERROR);
            return result;
        }

        // 获取目标CA证书DN
        String caServiceDnName = this.getCAServiceDnName();
        if (StringUtils.isBlank(caServiceDnName)) {
            result.setError(ErrorEnum.GET_CA_SERVICE_DN_NAME_ERROR);
            return result;
        }

        // 1. 调用SDK，使用CMP协议进行证书的撤销
        Result revokeResult = certLifeCycleManager.revokeUserCert(null,raServiceDnName, caServiceDnName, signSn, certDn, revokeType, revokeReason);
        
        
        if (!revokeResult.isSuccess()) {
            logger.info("CA返回撤销证书错误消息 ==== " + JsonUtils.object2Json(revokeResult));
            result.setErrorBean(revokeResult.getErrorBean());
            return result;
        }

        // 更新当前的证书状态改为已撤销
        try {
            customerCertDao.updateCustomerCertStatus(Constants.CERT_STATUS_REVOKED_3, signSn);
        } catch (Exception e) {
            logger.error("修改第三方系统证书状态异常{}", e);
            result.setError(ErrorEnum.UPDATE_OPERATOR_CERT_STATUS_EXCEPTION);
            return result;
        }

        return result;
    }

    @Override
    public Result getCertMaxValidity(String tempNo) {
        Result result = new Result();
        CertTempDO certTempInfoByTempNo = certTempDao.getCertTempInfoByTempNo(tempNo);
        if (certTempInfoByTempNo == null) {
            result.setError(ErrorEnum.GET_CERT_TEMP_INFO_IS_EMPTY);
            return result;
        }
        int tempMaxValidity = certTempInfoByTempNo.getMaxValidity();

        CaCertDO newCaCertInfo = caCertDao.getNewCaCertInfo();
        if (newCaCertInfo == null) {
            result.setError(ErrorEnum.GET_CA_CERT_INFO_IS_EMPTY);
            return result;
        }
        Timestamp failureTime = newCaCertInfo.getFailureTime();
        long failTime = failureTime.getTime();
        long nowTime = System.currentTimeMillis();
        int caValidity = (int) ((failTime - nowTime) / (1000 * 3600 * 24));
        logger.info("用户模板:" + tempNo + " CA最大有效期:" + caValidity + " 模板最大有效期:" + tempMaxValidity);
        int min = (tempMaxValidity < caValidity ? tempMaxValidity : caValidity);
        Map<String, Integer> map = new HashMap<>();
        map.put("maxValidity", min);
        result.setInfo(map);
        return result;
    }

    @Override
    public Result getCustomerTemplateSelectList(Long customerId) {
        Result result = new Result();
        try {
            List<CertTempDO> customerTemplateSelectList = customerTempDao.getCustomerTemplateSelectList(customerId);
            result.setInfo(customerTemplateSelectList);
        } catch (DAOException e) {
            throw new ServiceException("查询第三方系统模板下拉选择列表异常", e);
        }

        return result;
    }

    @Override
    public Result verifyClientCertDn(String systemFlag, String certDn) {
        //  证书DN的校验规则
        Result result = new Result();
        X500Name x500Name = null;
        try {
            x500Name = DnUtil.getRFC4519X500Name(certDn);
        } catch (Exception e) {
            logger.info("certDn不正确{}", e.getMessage());
            result.setError(ErrorEnum.CERT_APPLY_DN_IS_ERROR);
            return result;
        }
        String certDnFormat = x500Name.toString();
        logger.info("格式化之后的操作员证书DN：" + certDnFormat);

        // 检验DN关键字系统是否支持
        // DN的每个关键字的值不能超过64字节
        //检查是否包含中文逗号
        Result checkResult = CertDnVerifyUtils.checkDn(x500Name);
        if (!checkResult.isSuccess()) {
            result.setError(checkResult.getError());
            return result;
        }
        // 校验DN的特殊项先后顺序 DO 20200817 第三方系统证书签发扩展项目录替换名称也调用这个类 需要加上顺序校验 （原证书主体仅填写CN项 不需要校验）
        checkResult = CertDnVerifyUtils.checkCertDnSort(x500Name);
        if (!checkResult.isSuccess()) {
            logger.info("证书的DN的类型先后顺序有误");
            result.setError(checkResult.getError());
            return result;
        }
        // 校验DN的唯一性  以系统唯一标识为同一个系统
        List<CustomerSysCertDO> customerSysCertDOS = customerCertDao.queryClientCertByCertDn(systemFlag, certDn);
        if (!CollectionUtils.isEmpty(customerSysCertDOS)) {
            logger.info("有多个的证书的DN信息同【" + certDn + "】一致");
            result.setError(ErrorEnum.CLIENT_CERT_DN_HAVE_SAME_IN_CREATED);
            return result;
        }

        return result;
    }

    @Override
    public Result verifyCustomerSysName(Boolean isUpdate, Integer id, String systemName) {

        Result result = new Result();
        //系统标识符校验
        int count = customerDao.getSystemIdentifier(isUpdate, id, systemName);
        if (count >= 1) {
            result.setError(ErrorEnum.SYS_NAME_HAS_EXISTED);
            return result;
        }
        return result;
    }

    @Override
    public Result verifyCustomerSysNumber(String systemNumber) {
        Result result = new Result();
        //系统标识符校验
        int count = customerDao.getSystemIdentifier(systemNumber);
        if (count >= 1) {
            result.setError(ErrorEnum.SYS_ID_HAS_EXISTED);
            return result;
        }
        return result;
    }

    @Override
    public Result verifyUserSystemFlag(String systemNumber) {
        systemNumber = systemNumber == null ? Constants.SYSTEM_FLAG_DEFAULT : systemNumber;
        int count = customerDao.getSystemIdentifier(systemNumber);
        if (1 != count) {
            return Result.failure(ErrorEnum.ILLEGAL_REQUEST_PARAMETER);
        }
        return Result.success();
    }

    /**
     * 校验证书DN中CN项的值 20200714 第三方系统证书主体可能使用模板或者系统baseDn  以下校验无意义
     *
     * @return
     */
    private Result checkCertDnCnValue(String certDn) {
        Result result = new Result();
        Result raBaseDNResult = certApplyService.getRaBaseDn();
        if (!raBaseDNResult.isSuccess()) {
            return raBaseDNResult;
        }
        String raBaseDN = (String) raBaseDNResult.getInfo();
        if (certDn.length() <= 3 || certDn.length() <= (3 + raBaseDN.length())) {
            result.setError(ErrorEnum.OPERATOR_CERT_DN_LENGTH_IS_ERROR);
            return result;
        }
        String dnValue = certDn.substring(3, certDn.length() - raBaseDN.length() - 1);
        logger.info("第三方系统的DN值为 ========== " + dnValue);
        if (dnValue.contains("=")) {
            logger.info("第三方系统的DN中含有特殊符号");
            result.setError(ErrorEnum.OPERATOR_CERT_DN_VALUE_HAVE_ERROR_SYMBOL);
            return result;
        }
       /* if (dnValue.contains(",") || dnValue.contains("=")) {
            logger.info("第三方系统的DN中含有特殊符号");
            result.setError(ErrorEnum.OPERATOR_CERT_DN_VALUE_HAVE_ERROR_SYMBOL);
            return result;
        }*/
/*        Pattern compile = Pattern.compile("\uff0c");
        Matcher matcher = compile.matcher(dnValue);
        if (matcher.find()) {
            logger.info("证书主体中包含中文逗号>>>certDn:" + certDn);
            result.setError(ErrorEnum.CERT_DN_HAVE_CHINESE_COMMA);
            return result;
        }*/
        return result;
    }

    /**
     * 获取系统RA服务器签名证书DN
     *
     * @return
     */
    private String getRAServiceDnName() {
        RaCertDO newRaCertInfo = null;
        try {
            newRaCertInfo = raCertDao.getNewRaCertInfo();
        } catch (Exception e) {
            logger.error("获取RA服务器证书异常{}", e);
            return null;
        }

        if (newRaCertInfo == null) {
            return null;
        }
        String certInfo = newRaCertInfo.getCertInfo();
        X509Certificate certFromStr = CertUtils.getCertFromStr(certInfo);
        String raServiceDnName = CertUtils.getSubjectByX509Cert(certFromStr);
        return raServiceDnName;
    }

    /**
     * 获取系统CA服务器证书DN
     *
     * @return
     */
    private String getCAServiceDnName() {
        CaCertDO newCaCertInfo = null;
        try {
            newCaCertInfo = caCertDao.getNewCaCertInfo();
        } catch (Exception e) {
            logger.error("获取CA服务器证书异常{}", e);
            return null;
        }

        if (newCaCertInfo == null) {
            return null;
        }
        String certInfo = newCaCertInfo.getCertInfo();
        X509Certificate certFromStr = CertUtils.getCertFromStr(certInfo);
        String caServiceDnName = CertUtils.getSubjectByX509Cert(certFromStr);
        return caServiceDnName;
    }

    private Result insertCustomerCertInfo(CustomerCertIssueApplyVO customerCertIssueApplyVO, UserCertInfo customerCertInfo) {
        Result result = new Result();
        String signCert = customerCertInfo.getSignCert();
        if (StringUtils.isBlank(signCert)) {
            logger.info("CA返回的用户证书信息中，签名证书为空");
            result.setError(ErrorEnum.CA_RESPONSE_USER_SIGN_CERT_INFO_IS_EMPTY);
            return result;
        }

        long pairCertIndex = System.nanoTime();

        CustomerSysCertDO signCertDO = new CustomerSysCertDO();

        handleCustomerSysCertDO(pairCertIndex, customerCertIssueApplyVO, signCertDO);

        if (StringUtils.isBlank(customerCertInfo.getEncCert())) {
            signCertDO.setCertType(Constants.CERT_TYPE_SINGLE_1);
        } else {
            signCertDO.setCertType(Constants.CERT_TYPE_SIGN_2);
        }


        CaCertDO newCaCertInfo = caCertDao.getNewCaCertInfo();
        if (newCaCertInfo == null) {
            logger.info("获取CA证书信息为空");
            result.setError(ErrorEnum.GET_CA_CERT_INFO_IS_EMPTY);
            return result;
        }

        signCertDO.setCaCertId(newCaCertInfo.getId());

        X509Certificate signCertStr = CertUtils.getCertFromStr(signCert);
        if (signCertStr == null) {
            logger.info("CA返回的用户证书信息中，签名证书错误");
            result.setError(ErrorEnum.CA_RESPONSE_USER_SIGN_CERT_ERROR);
            return result;
        }
        signCertDO.setCertInfo(signCert);
        signCertDO.setCertSn(signCertStr.getSerialNumber().toString(16).toLowerCase());
        signCertDO.setCertDn(CertUtils.getSubjectByX509Cert(signCertStr));

        Date startDate = signCertStr.getNotBefore();
        Date endDate = signCertStr.getNotAfter();
        signCertDO.setEffectiveTime(new Timestamp(startDate.getTime()));
        signCertDO.setFailureTime(new Timestamp(endDate.getTime()));

        // 第三方系统证书有效期
        int certDays = (int) ((endDate.getTime() - startDate.getTime()) / (1000 * 3600 * 24));
        signCertDO.setCertValidity(certDays);

        // 第三方系统证书中对应的加密公私钥的有效天数
        int priKeyDays = (int) ((newCaCertInfo.getFailureTime().getTime() - startDate.getTime()) / (1000 * 3600 * 24));
        signCertDO.setEncKeyValidity(priKeyDays);

        Date date = new Date();
        signCertDO.setGmtCreate(new Timestamp(date.getTime()));
        signCertDO.setGmtUpdate((new Timestamp(date.getTime())));
        CustomerSysCertDO customerSignCertDO = customerCertDao.insertCustomerCertInfo(signCertDO);

        // 加密
        String encCert = customerCertInfo.getEncCert();
        if (StringUtils.isNotBlank(encCert)) {
            CustomerSysCertDO encCertDO = new CustomerSysCertDO();

            handleCustomerSysCertDO(pairCertIndex, customerCertIssueApplyVO, encCertDO);
            encCertDO.setCertType(Constants.CERT_TYPE_ENC_3);
            encCertDO.setCertStatus(Constants.CERT_STATUS_NO_CONFIRM_0);
            encCertDO.setCaCertId(newCaCertInfo.getId());
            encCertDO.setEffectiveTime(new Timestamp(startDate.getTime()));
            encCertDO.setFailureTime(new Timestamp(endDate.getTime()));
            encCertDO.setCertValidity(certDays);
            encCertDO.setEncKeyValidity(priKeyDays);

            X509Certificate encCertStr = CertUtils.getCertFromStr(encCert);
            if (encCertStr == null) {
                logger.info("CA返回的用户证书信息中，加密证书错误");
                result.setError(ErrorEnum.CA_RESPONSE_USER_ENC_CERT_ERROR);
                return result;
            }

            encCertDO.setCertInfo(encCert);
            encCertDO.setGmtCreate(new Timestamp(date.getTime()));
            encCertDO.setGmtUpdate((new Timestamp(date.getTime())));
            encCertDO.setCertDn(CertUtils.getSubjectByX509Cert(encCertStr));
            encCertDO.setCertSn(encCertStr.getSerialNumber().toString(16).toLowerCase());
            encCertDO.setSignCertSn(customerSignCertDO.getCertSn());
            try {
                customerCertDao.insertCustomerCertInfo(encCertDO);
            } catch (Exception e) {
                logger.info("插入第三方系统证书信息异常");
                customerCertDao.deleteCustomerCert(customerSignCertDO.getId());
            }
        }
        return result;
    }

    private void handleCustomerSysCertDO(long pairCertIndex, CustomerCertIssueApplyVO customerCertIssueApplyVO, CustomerSysCertDO customerSysCertDO) {
        customerSysCertDO.setPairCertIndex(pairCertIndex);
        customerSysCertDO.setCertStatus(Constants.CERT_STATUS_NO_CONFIRM_0);

        customerSysCertDO.setCustomerSysId(customerCertIssueApplyVO.getCustomerSysId());
        customerSysCertDO.setCustomerSysNumber(customerCertIssueApplyVO.getCustomerSysNumber());
        customerSysCertDO.setTempId(customerCertIssueApplyVO.getTempId());
        customerSysCertDO.setTempNo(customerCertIssueApplyVO.getTempNo());
        customerSysCertDO.setSignAlg(customerCertIssueApplyVO.getSignAlg());
        customerSysCertDO.setPrivateKeyLength(customerCertIssueApplyVO.getPrivateKeyLength());
    }


}
