package com.xdja.pki.ra.service.manager.operator;

import com.xdja.ca.utils.DnUtil;
import com.xdja.ca.utils.SdkCertUtils;
import com.xdja.ca.vo.ManagerCertInfo;
import com.xdja.pki.gmssl.crypto.utils.GMSSLRSAKeyUtils;
import com.xdja.pki.ra.core.common.CommonVariable;
import com.xdja.pki.ra.core.commonenum.ErrorEnum;
import com.xdja.pki.ra.core.common.OperatorCertCache;
import com.xdja.pki.ra.core.common.Result;
import com.xdja.pki.ra.core.constant.Constants;
import com.xdja.pki.ra.core.constant.PathConstants;
import com.xdja.pki.ra.core.pkcs7.SignedDataUtils;
import com.xdja.pki.ra.core.util.cert.CertDnVerifyUtils;
import com.xdja.pki.ra.core.util.cert.CertUtils;
import com.xdja.pki.ra.core.util.cert.RandomUtils;
import com.xdja.pki.ra.core.util.file.FileUtils;
import com.xdja.pki.ra.core.util.json.JsonUtils;
import com.xdja.pki.ra.manager.dao.AdminCertDao;
import com.xdja.pki.ra.manager.dao.AdminRoleDao;
import com.xdja.pki.ra.manager.dao.CaCertDao;
import com.xdja.pki.ra.manager.dao.RaCertDao;
import com.xdja.pki.ra.manager.dao.model.AdminCertDO;
import com.xdja.pki.ra.manager.dao.model.AdminRoleDO;
import com.xdja.pki.ra.manager.dao.model.CaCertDO;
import com.xdja.pki.ra.manager.dao.model.RaCertDO;
import com.xdja.pki.ra.manager.dto.AdminCertDTO;
import com.xdja.pki.ra.manager.page.PageInfo;
import com.xdja.pki.ra.manager.sdk.business.CaBusinessManager;
import com.xdja.pki.ra.manager.sdk.cmp.CertLifeCycleManager;
import com.xdja.pki.ra.service.manager.certapply.CertApplyService;
import com.xdja.pki.ra.service.manager.operator.bean.ManagerCertVO;
import com.xdja.pki.ra.service.manager.operator.bean.OperatorCertResp;
import com.xdja.pki.ra.service.manager.operator.bean.OperatorCertVO;
import com.xdja.pki.ra.service.manager.utils.CertContentInfoUtil;
import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.asn1.nist.NISTNamedCurves;
import org.bouncycastle.asn1.sec.SECObjectIdentifiers;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.style.RFC4519Style;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.BigIntegers;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.io.IOException;
import java.math.BigInteger;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.security.cert.X509Certificate;
import java.security.spec.RSAPublicKeySpec;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * RA业务管理员服务层-实现类
 *
 * @author syg
 */
@Service
public class OperatorCertServiceImpl implements OperatorCertService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    AdminCertDao adminCertDao;

    @Autowired
    CertContentInfoUtil certContentInfoUtil;

    @Autowired
    CertLifeCycleManager certLifeCycleManager;

    @Autowired
    CaBusinessManager caBusinessManager;

    @Autowired
    CaCertDao caCertDao;

    @Autowired
    RaCertDao raCertDao;

    @Autowired
    AdminRoleDao adminRoleDao;

    @Autowired
    private CertApplyService certApplyService;

    @Override
    public Result queryOperatorCertList(String certDN, int pageNo, int pageSize) {
        OperatorCertResp operatorCertResp = new OperatorCertResp();
        Result result = new Result();
        PageInfo<AdminCertDTO> adminCertDTOPageInfo = adminCertDao.listPageAdminCert(certDN, pageNo, pageSize);
        if (adminCertDTOPageInfo == null) {
            result.setError(ErrorEnum.QUERY_OPERATOR_CERT_LIST_ERROR);
            return result;
        }
        List<OperatorCertVO> operatorCertVOList = new ArrayList<>();

        List<AdminCertDTO> list = adminCertDTOPageInfo.getList();
        if (CollectionUtils.isEmpty(list)) {
            operatorCertResp.setRecordCount(0);
            operatorCertResp.setPageCount(0);
            operatorCertResp.setDatas(operatorCertVOList);
            result.setInfo(operatorCertResp);
            return result;
        }
        for (AdminCertDTO adminCertDTO : list) {
            OperatorCertVO operatorCertVO = new OperatorCertVO();
            BeanUtils.copyProperties(adminCertDTO, operatorCertVO);
            Timestamp effectiveTime = adminCertDTO.getEffectiveTime();
            Timestamp failureTime = adminCertDTO.getFailureTime();
            operatorCertVO.setEffectiveTime(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(effectiveTime));
            operatorCertVO.setFailureTime(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(failureTime));

            if (new Date().getTime()>failureTime.getTime()){
                operatorCertVO.setCertStatus(Constants.CERT_STATUS_EXPIRED_2);
            }

            operatorCertVOList.add(operatorCertVO);
        }

        operatorCertResp.setDatas(operatorCertVOList);
        operatorCertResp.setPageCount(adminCertDTOPageInfo.getPageCount());
        operatorCertResp.setRecordCount(adminCertDTOPageInfo.getRecordCount());
        result.setInfo(operatorCertResp);
        return result;
    }


    @Override
    public Result getOperatorCertDetail(String certSn, int certType) {
        Result result = new Result();
        AdminCertDO adminCertInfo = adminCertDao.getAdminCertInfo(certSn, certType);
        if (adminCertInfo == null) {
            result.setError(ErrorEnum.GET_OPERATOR_CERT_INFO_EMPTY);
            return result;
        }
        String certInfo = adminCertInfo.getCertInfo();

        Map<String, Object> certContentInfo = null;
        try {
            X509Certificate certificate = CertUtils.getCertFromStr(certInfo);
            certContentInfo = certContentInfoUtil.getCertContentInfo(certificate);
        } catch (Exception e) {
            logger.error("证书详情格式读取异常", e);
            result.setError(ErrorEnum.CERT_DETAIL_FORMAT_ERROR);
            return result;
        }
        result.setInfo(certContentInfo);
        return result;
    }

    @Override
    public Result getOperatorCertInfo(String signSn) {
        Result result = new Result();
        AdminCertDO adminCertInfo = adminCertDao.getAdminCertInfo(signSn, Constants.CERT_TYPE_SIGN_2);
        if (adminCertInfo == null) {
            result.setError(ErrorEnum.GET_OPERATOR_CERT_INFO_EMPTY);
            return result;
        }
        OperatorCertVO operatorCertVO = new OperatorCertVO();
        BeanUtils.copyProperties(adminCertInfo,operatorCertVO);
        int leftValidity = (int) ((adminCertInfo.getFailureTime().getTime()- new Date().getTime()) / (1000 * 3600 * 24));
        operatorCertVO.setCertValidity(leftValidity);
        operatorCertVO.setEffectiveTime(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(adminCertInfo.getEffectiveTime()));
        operatorCertVO.setFailureTime(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(adminCertInfo.getFailureTime()));
        result.setInfo(operatorCertVO);
        return result;
    }

    @Override
    public Result getOperatorCertBase64Info(String signSn) {
        Result result = new Result();
        Map<String, Object> adminCertBase64Info = new HashMap<>();
        try{
            adminCertBase64Info = adminCertDao.getAdminCertBase64Info(signSn);
        }catch (Exception e){
            result.setError(ErrorEnum.GET_OPERATOR_CERT_BASE64_INFO_EMPTY);
            return result;
        }
        if (adminCertBase64Info == null) {
            result.setError(ErrorEnum.GET_OPERATOR_CERT_BASE64_INFO_EMPTY);
            return result;
        }

        String signCertStr = (String) adminCertBase64Info.get("signCert");
        String encCertStr = (String) adminCertBase64Info.get("encCert");


        List<Map<String, Object>> list = new ArrayList<>();
        X509Certificate signCert = CertUtils.getCertFromStr(signCertStr);
        String signTrustCert = null;
        String encTrustCert = null;
        try{
            List<X509Certificate> caTrustCert = CommonVariable.getTrustCaCerts();
            List<X509Certificate> caTrust = new ArrayList<>(caTrustCert);
            caTrust.add(signCert);
            signTrustCert = SignedDataUtils.createCertChainByCerts(caTrust);
            if (!Constants.KEY_ALG_NAME_NIST.equalsIgnoreCase(CommonVariable.getKeyAlgName())) {
                X509Certificate encCert = CertUtils.getCertFromStr(encCertStr);
                caTrust.remove(signCert);
                caTrust.add(encCert);
                encTrustCert = SignedDataUtils.createCertChainByCerts(caTrust);
            }
        }catch (Exception e){
            logger.error("sdk接口-封装证书链异常" ,e);
            result.setError(ErrorEnum.BUILD_TRAIN_CERT_P7b_IS_ERROR);
            return result;
        }

        //审计日志
        result.setLogContent("，证书主体="+ CertUtils.getSubjectByX509Cert(signCert));

        Map<String, Object> signMap = new HashMap<>();
        signMap.put("name", "SignCert");
        signMap.put("suffix", "p7b");
        byte[] signByte = signTrustCert.getBytes();
        signMap.put("buffer", signByte);
        list.add(signMap);

        if (StringUtils.isNotBlank(encTrustCert)) {
            Map<String, Object> encMap = new HashMap<>();
            encMap.put("name", "EncCert");
            encMap.put("suffix", "p7b");
            byte[] encByte = encTrustCert.getBytes();
            encMap.put("buffer", encByte);
            list.add(encMap);
        }

        result.setInfo(list);
        return result;
    }

    @Override
    public Result getRaOperatorCertTemp(){
        Result result = new Result();
        Result raOperatorCertTemp = caBusinessManager.getRaOperatorCertTemp();
        if (!raOperatorCertTemp.isSuccess()){
            return raOperatorCertTemp;
        }
        Map<String,Object> map = (Map<String, Object>) raOperatorCertTemp.getInfo();
//        int tempMaxValidity = (int) map.get("maxDate");
//
//        CaCertDO newCaCertInfo = caCertDao.getNewCaCertInfo();
//        if (newCaCertInfo == null){
//            result.setError(ErrorEnum.GET_CA_CERT_INFO_IS_EMPTY);
//            return result;
//        }
//        Timestamp failureTime = newCaCertInfo.getFailureTime();
//        long failTime = failureTime.getTime();
//        long nowTime = new Date().getTime();
//        int caValidity = (int) ((failTime - nowTime) / (1000 * 3600 * 24));
//
//        int min = (tempMaxValidity < caValidity ? tempMaxValidity : caValidity);
//        map.put("maxDate",min);
        result.setInfo(map);
        return result;
    }

    @Override
    public Result getOperatorCertMaxValidity(int updateKey,String signSn){
        Result result = new Result();
        Result raOperatorCertTemp = caBusinessManager.getRaOperatorCertTemp();
        if (!raOperatorCertTemp.isSuccess()){
            return raOperatorCertTemp;
        }
        Map<String,Object> map = (Map<String, Object>) raOperatorCertTemp.getInfo();
        int tempMaxValidity = (int) map.get("maxDate");

        CaCertDO newCaCertInfo = caCertDao.getNewCaCertInfo();
        if (newCaCertInfo == null){
            result.setError(ErrorEnum.GET_CA_CERT_INFO_IS_EMPTY);
            return result;
        }
        Timestamp failureTime = newCaCertInfo.getFailureTime();
        long failTime = failureTime.getTime();
        long nowTime = new Date().getTime();
        int caValidity = (int) ((failTime - nowTime) / (1000 * 3600 * 24));

        int min = (tempMaxValidity < caValidity ? tempMaxValidity : caValidity);
        logger.info("操作员可选证书有效期中，CA最大有效期:"+caValidity +" 模板最大有效期:"+tempMaxValidity);
        result.setInfo(min);
        return result;
//        if (updateKey == Constants.UPDATE_CERT_KEY_1) {
//            logger.info("操作员可选证书有效期中，CA最大有效期:"+caValidity +" 模板最大有效期:"+tempMaxValidity);
//            result.setInfo(min);
//            return result;
//        }

//        AdminCertDO adminCertInfo = adminCertDao.getAdminCertInfo(signSn, Constants.CERT_TYPE_SIGN_2);
//        if (adminCertInfo == null) {
//            logger.info("选择更新证书密钥对时，传入的签名证书sn{}，未找到对应的操作员证书信息",signSn);
//            result.setError(ErrorEnum.GET_OPERATOR_CERT_INFO_EMPTY);
//            return result;
//        }
//        int operatorCertValidity = (int) ((adminCertInfo.getFailureTime().getTime()- new Date().getTime()) / (1000 * 3600 * 24));
//        logger.info("操作员可选证书有效期中，CA最大有效期:"+caValidity +" 模板最大有效期:"+tempMaxValidity+" 加密密钥最大有效期:"+operatorCertValidity);
//        // 取CA证书有效期、模板有效期、加密密钥有效期的最小值
//        min = tempMaxValidity;
//        min = (caValidity < min) ? caValidity : min;
//        min = (operatorCertValidity < min) ? operatorCertValidity : min;
//        result.setInfo(min);
//        return result;
    }

    @Transactional
    @Override
    public Result insertOperatorCert(AdminCertDTO adminCertDTO){
        Result result = new Result();
        // 获取当前RA服务器签名证书DN
        String raServiceDnName = this.getRAServiceDnName();
        if (StringUtils.isBlank(raServiceDnName)){
            result.setError(ErrorEnum.GET_RA_SERVICE_DN_NAME_ERROR);
            return result;
        }

        CaCertDO newCaCertInfo = caCertDao.getNewCaCertInfo();
        if (newCaCertInfo == null) {
            logger.info("获取CA证书信息为空");
            result.setError(ErrorEnum.GET_CA_CERT_INFO_IS_EMPTY);
            return result;
        }
        String caServiceDnName = newCaCertInfo.getCertDn();

        String transId = RandomUtils.getUUID();

        // 1. 调用SDK，使用CMP协议进行证书的签发
        Result issueResult = certLifeCycleManager.issueManageCert(adminCertDTO.getCardNo(), adminCertDTO.getSignP10(), adminCertDTO.getEncPubKey(), raServiceDnName, caServiceDnName, transId, adminCertDTO.getManagerTempNo(), adminCertDTO.getSignAlg(), adminCertDTO.getCertValidity(), adminCertDTO.getCertDn());
        if (!issueResult.isSuccess()) {
            logger.info("签发证书错误消息 ==== "+ JsonUtils.object2Json(issueResult));
            result.setErrorBean(issueResult.getErrorBean());
            return result;
        }

        // 2. 根据CA签发的管理员证书，保存并授权响应的管理员权限
        ManagerCertInfo managerCertInfo = (ManagerCertInfo) issueResult.getInfo();

        // 2.1 添加admin_cert表数据
        long pairCertIndex = System.nanoTime();

        List<Integer> roles = adminCertDTO.getRoles();
        StringBuffer sb = new StringBuffer();
        for (int role : roles) {
            if (role == Constants.ADMIN_ROLE_OPERATOR_INPUT_3){
                sb.append("录入员、");
            }else if (role == Constants.ADMIN_ROLE_OPERATOR_AUDIT_4){
                sb.append("审核员、");
            }else if (role == Constants.ADMIN_ROLE_OPERATOR_ISSUE_5){
                sb.append("制证员、");
            }else {
                logger.info("操作员角色类型错误");
                result.setError(ErrorEnum.OPERATOR_ROLE_IS_ERROR);
                return result;
            }
        }
        String roleInfo = sb.substring(0,sb.length()-1);

        AdminCertDO adminCertSignCertDO = new AdminCertDO();
        BeanUtils.copyProperties(adminCertDTO,adminCertSignCertDO);
        adminCertSignCertDO.setAdminType(Constants.ADMIN_TYPE_OPERATOR_3);
        adminCertSignCertDO.setCaCertId(newCaCertInfo.getId());
        adminCertSignCertDO.setCertInfo(managerCertInfo.getSignCert());

        adminCertSignCertDO.setPairCertIndex(pairCertIndex);
        adminCertSignCertDO.setCertType(Constants.CERT_TYPE_SIGN_2);
        adminCertSignCertDO.setRoleInfo(roleInfo);

        X509Certificate signCertStr = CertUtils.getCertFromStr(managerCertInfo.getSignCert());
        if (signCertStr == null) {
            logger.info("CA返回的管理员证书信息中，签名证书错误");
            result.setError(ErrorEnum.CA_RESPONSE_USER_SIGN_CERT_ERROR);
            return result;
        }
        adminCertSignCertDO.setCertSn(signCertStr.getSerialNumber().toString(16).toLowerCase());
        adminCertSignCertDO.setCertDn(CertUtils.getSubjectByX509Cert(signCertStr));
        logger.info("============== CA返回签发签名证书sn:"+adminCertSignCertDO.getCertSn());
        adminCertSignCertDO.setEncPubKey(null);

        int publicKeyLength = 0;
        try {
            publicKeyLength = CertUtils.getPublicKeyLength(signCertStr);
            adminCertSignCertDO.setPrivateKeyLength(publicKeyLength);
        } catch (Exception e) {
            logger.error("获取证书公钥长度异常",e);
            result.setError(ErrorEnum.GET_CERT_PUBLIC_KEY_LENGTH_EXCEPTION);
            return result;
        }
        String publicKeyAlg = null;
        try {
            publicKeyAlg = CertUtils.getKeyAlg(signCertStr);
            adminCertSignCertDO.setPublicKeyAlg(publicKeyAlg);
        } catch (Exception e) {
            logger.error("获取证书密钥算法异常", e);
            result.setError(ErrorEnum.GET_CERT_ALG_NAME_LENGTH_EXCEPTION);
            return result;
        }

        Date startDate = signCertStr.getNotBefore();
        Date endDate = signCertStr.getNotAfter();

        adminCertSignCertDO.setEffectiveTime(new Timestamp(startDate.getTime()));
        adminCertSignCertDO.setFailureTime(new Timestamp(endDate.getTime()));

        adminCertSignCertDO.setGmtCreate(new Timestamp(new Date().getTime()));
        adminCertSignCertDO.setGmtUpdate(new Timestamp(new Date().getTime()));
        AdminCertDO operatorSignCertDO = adminCertDao.insertAdminCert(adminCertSignCertDO);


        AdminCertDO adminCertEncCertDO = new AdminCertDO();
        BeanUtils.copyProperties(adminCertDTO,adminCertEncCertDO);
        adminCertEncCertDO.setAdminType(Constants.ADMIN_TYPE_OPERATOR_3);
        adminCertEncCertDO.setCaCertId(newCaCertInfo.getId());
        adminCertEncCertDO.setCertInfo(managerCertInfo.getEncEncCert());

        adminCertEncCertDO.setPairCertIndex(pairCertIndex);
        adminCertEncCertDO.setCertType(Constants.CERT_TYPE_ENC_3);
        adminCertEncCertDO.setRoleInfo(roleInfo);


//        adminCertEncCertDO.setCertSn(signCertStr.getSerialNumber().toString(16).toLowerCase());
        adminCertEncCertDO.setCertDn(CertUtils.getSubjectByX509Cert(signCertStr));
        adminCertEncCertDO.setEncPubKey(adminCertDTO.getEncPubKey());
        adminCertEncCertDO.setPrivateKeyLength(publicKeyLength);
        adminCertEncCertDO.setPublicKeyAlg(publicKeyAlg);

        adminCertEncCertDO.setEffectiveTime(new Timestamp(startDate.getTime()));
        adminCertEncCertDO.setFailureTime(new Timestamp(endDate.getTime()));

        adminCertEncCertDO.setGmtCreate(new Timestamp(new Date().getTime()));
        adminCertEncCertDO.setGmtUpdate(new Timestamp(new Date().getTime()));
        adminCertDao.insertAdminCert(adminCertEncCertDO);

        // 2.2 添加admin_role表数据
        for (int role : roles) {
            AdminRoleDO adminRoleDO = new AdminRoleDO(operatorSignCertDO.getId(), (long) role,new Timestamp(new Date().getTime()),new Timestamp(new Date().getTime()));
            adminRoleDao.insertAdminRole(adminRoleDO);
        }
        ManagerCertVO managerCertVO = new ManagerCertVO(managerCertInfo.getSignCert(),managerCertInfo.getEncEncCert(),transId);
        result.setInfo(managerCertVO);
        return result;
    }

    @Transactional
    @Override
    public Result updateOperatorCert(AdminCertDTO adminCertDTO) {
        Result result = new Result();
        // 获取当前RA服务器签名证书DN
        String raServiceDnName = this.getRAServiceDnName();
        if (StringUtils.isBlank(raServiceDnName)){
            result.setError(ErrorEnum.GET_RA_SERVICE_DN_NAME_ERROR);
            return result;
        }

        CaCertDO newCaCertInfo = caCertDao.getNewCaCertInfo();
        if (newCaCertInfo == null) {
            logger.info("获取CA证书信息为空");
            result.setError(ErrorEnum.GET_CA_CERT_INFO_IS_EMPTY);
            return result;
        }
        String caServiceDnName = newCaCertInfo.getCertDn();

        String transId = RandomUtils.getUUID();

        // 缓存事务id和更新前签名证书sn的对应关系
        Map<String, Object> map = OperatorCertCache.getMap();
        map.put(transId, adminCertDTO.getSignCertSn());
        logger.info("缓存的操作员sn信息 ===== "+JsonUtils.object2Json(map));

        // 1. 调用SDK，使用CMP协议进行证书的更新
        Result updateResult = certLifeCycleManager.updateManageCert(adminCertDTO.getCardNo(), adminCertDTO.getSignP10(), adminCertDTO.getEncPubKey(), raServiceDnName, caServiceDnName, transId, adminCertDTO.getManagerTempNo(), adminCertDTO.getSignAlg(), adminCertDTO.getCertValidity(), adminCertDTO.getCertDn(),adminCertDTO.getSignCertSn(),adminCertDTO.isUpdateKey());
        if (!updateResult.isSuccess()) {
            logger.info("CA返回更新证书错误消息 ==== "+ JsonUtils.object2Json(updateResult));
            result.setErrorBean(updateResult.getErrorBean());
            return result;
        }

        ManagerCertInfo managerCertInfo = (ManagerCertInfo) updateResult.getInfo();
//
//        // 更新前的证书状态改为已撤销
//        try {
//            adminCertDao.updateAdminCertStatus(Constants.CERT_STATUS_REVOKED_3, adminCertDTO.getSignCertSn());
//        } catch (Exception e) {
//            logger.error("修改操作员证书状态异常{}", e);
//            result.setError(ErrorEnum.UPDATE_OPERATOR_CERT_STATUS_EXCEPTION);
//            return result;
//        }

        // 获取之前管理员的基本信息
        AdminCertDO adminCertDO = adminCertDao.getAdminCertInfo(adminCertDTO.getSignCertSn(), Constants.CERT_TYPE_SIGN_2);
        if (adminCertDO == null) {
            result.setError(ErrorEnum.GET_OPERATOR_CERT_INFO_EMPTY);
            return result;
        }

        // 新增信息管理员信息
        long pairCertIndex = System.nanoTime();

        AdminCertDO adminCertSignCertDO = new AdminCertDO();
        BeanUtils.copyProperties(adminCertDTO,adminCertSignCertDO);
        adminCertSignCertDO.setAdminType(Constants.ADMIN_TYPE_OPERATOR_3);
        adminCertSignCertDO.setCaCertId(newCaCertInfo.getId());
        adminCertSignCertDO.setCertInfo(managerCertInfo.getSignCert());

        adminCertSignCertDO.setPairCertIndex(pairCertIndex);
        adminCertSignCertDO.setCertType(Constants.CERT_TYPE_SIGN_2);
        adminCertSignCertDO.setRoleInfo(adminCertDO.getRoleInfo());

        X509Certificate signCertStr = CertUtils.getCertFromStr(managerCertInfo.getSignCert());
        if (signCertStr == null) {
            logger.info("CA返回的管理员证书信息中，签名证书错误");
            result.setError(ErrorEnum.CA_RESPONSE_USER_SIGN_CERT_ERROR);
            return result;
        }
        adminCertSignCertDO.setCertSn(signCertStr.getSerialNumber().toString(16).toLowerCase());
        adminCertSignCertDO.setCertDn(CertUtils.getSubjectByX509Cert(signCertStr));
        adminCertSignCertDO.setEncPubKey(null);

        int publicKeyLength = 0;
        try {
            publicKeyLength = CertUtils.getPublicKeyLength(signCertStr);
            adminCertSignCertDO.setPrivateKeyLength(publicKeyLength);
        } catch (Exception e) {
            logger.error("获取证书公钥长度异常",e);
            result.setError(ErrorEnum.GET_CERT_PUBLIC_KEY_LENGTH_EXCEPTION);
            return result;
        }
        String publicKeyAlg = null;
        try {
            publicKeyAlg = CertUtils.getKeyAlg(signCertStr);
            adminCertSignCertDO.setPublicKeyAlg(publicKeyAlg);
        } catch (Exception e) {
            logger.error("获取证书密钥算法异常", e);
            result.setError(ErrorEnum.GET_CERT_ALG_NAME_LENGTH_EXCEPTION);
            return result;
        }

        Date startDate = signCertStr.getNotBefore();
        Date endDate = signCertStr.getNotAfter();
        adminCertSignCertDO.setEffectiveTime(new Timestamp(startDate.getTime()));
        adminCertSignCertDO.setFailureTime(new Timestamp(endDate.getTime()));

        adminCertSignCertDO.setGmtCreate(new Timestamp(new Date().getTime()));
        adminCertSignCertDO.setGmtUpdate(new Timestamp(new Date().getTime()));
        AdminCertDO operatorSignCertDO = adminCertDao.insertAdminCert(adminCertSignCertDO);


        AdminCertDO adminCertEncCertDO = new AdminCertDO();
        BeanUtils.copyProperties(adminCertDTO,adminCertEncCertDO);
        adminCertEncCertDO.setAdminType(Constants.ADMIN_TYPE_OPERATOR_3);
        adminCertEncCertDO.setCaCertId(newCaCertInfo.getId());
        adminCertEncCertDO.setCertInfo(managerCertInfo.getEncEncCert());

        adminCertEncCertDO.setPairCertIndex(pairCertIndex);
        adminCertEncCertDO.setCertType(Constants.CERT_TYPE_ENC_3);
        adminCertEncCertDO.setRoleInfo(adminCertDO.getRoleInfo());


//        adminCertEncCertDO.setCertSn(signCertStr.getSerialNumber().toString(16).toLowerCase());
        adminCertEncCertDO.setCertDn(CertUtils.getSubjectByX509Cert(signCertStr));
        adminCertEncCertDO.setEncPubKey(adminCertDTO.getEncPubKey());

        adminCertEncCertDO.setPrivateKeyLength(publicKeyLength);
        adminCertEncCertDO.setPublicKeyAlg(publicKeyAlg);

        adminCertEncCertDO.setEffectiveTime(new Timestamp(startDate.getTime()));
        adminCertEncCertDO.setFailureTime(new Timestamp(endDate.getTime()));

        adminCertEncCertDO.setGmtCreate(new Timestamp(new Date().getTime()));
        adminCertEncCertDO.setGmtUpdate(new Timestamp(new Date().getTime()));
        adminCertDao.insertAdminCert(adminCertEncCertDO);

        // 2.2 添加admin_role表数据
        List<AdminRoleDO> adminRoleDOList = adminRoleDao.queryAdminRoleList(adminCertDO.getId());
        if(CollectionUtils.isEmpty(adminRoleDOList)) {
            result.setError(ErrorEnum.QUERY_ADMIN_ROLE_IS_EMPTY);
            return result;
        }
        for (AdminRoleDO adminRoleDO : adminRoleDOList) {
            adminRoleDO.setAdminId(operatorSignCertDO.getId());
            adminRoleDO.setRoleId(adminRoleDO.getRoleId());
            adminRoleDO.setGmtCreate(new Timestamp(new Date().getTime()));
            adminRoleDO.setGmtUpdate(new Timestamp(new Date().getTime()));
            adminRoleDao.insertAdminRole(adminRoleDO);
        }
        ManagerCertVO managerCertVO = new ManagerCertVO(managerCertInfo.getSignCert(),managerCertInfo.getEncEncCert(),transId);
        result.setInfo(managerCertVO);
        return result;
    }

    @Override
    public Result revokeOperatorCert(String signSn, String certDn, int revokeType, String revokeReason) {
        Result result = new Result();
        // 获取当前RA服务器签名证书DN
        String raServiceDnName = this.getRAServiceDnName();
        if (StringUtils.isBlank(raServiceDnName)){
            result.setError(ErrorEnum.GET_RA_SERVICE_DN_NAME_ERROR);
            return result;
        }

        // 获取目标CA证书DN
        String caServiceDnName = this.getCAServiceDnName();
        if (StringUtils.isBlank(caServiceDnName)){
            result.setError(ErrorEnum.GET_CA_SERVICE_DN_NAME_ERROR);
            return result;
        }

        // 1. 调用SDK，使用CMP协议进行证书的撤销
        Result revokeResult = certLifeCycleManager.revokeManageCert(raServiceDnName,caServiceDnName,signSn,certDn,revokeType,revokeReason);
        if (!revokeResult.isSuccess()) {
            logger.info("CA返回撤销证书错误消息 ==== "+ JsonUtils.object2Json(revokeResult));
            result.setErrorBean(revokeResult.getErrorBean());
            return result;
        }

        // 更新前的证书状态改为已撤销
        try {
            adminCertDao.updateAdminCertStatus(Constants.CERT_STATUS_REVOKED_3, signSn);
        } catch (Exception e) {
            logger.error("修改操作员证书状态异常{}", e);
            result.setError(ErrorEnum.UPDATE_OPERATOR_CERT_STATUS_EXCEPTION);
            return result;
        }

        return result;
    }

    @Override
    public Result updateOperatorCertStatus(int certStatus, String certSn) {
        Result result = new Result();
        try {
            adminCertDao.updateAdminCertStatus(certStatus, certSn);
        } catch (Exception e) {
            result.setError(ErrorEnum.UPDATE_OPERATOR_CERT_STATUS_EXCEPTION);
            return result;
        }
        return result;
    }


    @Transactional
    @Override
    public Result genCertConfirm(String transId, String signSn, String encSn, String encCert) {
        Result result = new Result();
        // 获取当前RA服务器签名证书DN
        String raServiceDnName = this.getRAServiceDnName();
        if (StringUtils.isBlank(raServiceDnName)){
            result.setError(ErrorEnum.GET_RA_SERVICE_DN_NAME_ERROR);
            return result;
        }

        // 获取目标CA证书DN
        String caServiceDnName = this.getCAServiceDnName();
        if (StringUtils.isBlank(caServiceDnName)){
            result.setError(ErrorEnum.GET_CA_SERVICE_DN_NAME_ERROR);
            return result;
        }

        // 校验加密公钥是否一致
        String publicKeyStr = null;
        String encPubKey = adminCertDao.getEncPubKeyBySignSn(signSn);
        if (StringUtils.isNotBlank(encPubKey)){
            PublicKey publicKey = null;
            try{
                if (Constants.KEY_ALG_NAME_SM2.equalsIgnoreCase(CommonVariable.getKeyAlgName())){
                    publicKey = SdkCertUtils.convertSM2PublicKey(encPubKey);
                }else if (Constants.KEY_ALG_NAME_NIST.equalsIgnoreCase(CommonVariable.getKeyAlgName())){
                    publicKey = SdkCertUtils.convertECPublicKey(encPubKey, NISTNamedCurves.getName(SECObjectIdentifiers.secp256r1));
                }else {
                    KeyFactory kf = KeyFactory.getInstance("RSA",new BouncyCastleProvider());
                    RSAPublicKeySpec rsaPublicKeySpec = new RSAPublicKeySpec(BigIntegers.fromUnsignedByteArray(Base64.decode(encPubKey)), BigInteger.valueOf(65537L));
                    publicKey = kf.generatePublic(rsaPublicKeySpec);
                }
            }catch (Exception e){
                logger.error("获取证书请求时的公钥信息异常{}",e);
                result.setError(ErrorEnum.GET_CERT_PUBLIC_KEY_EXCEPTION);
                return result;
            }
            publicKeyStr = Base64.toBase64String(publicKey.getEncoded());
            logger.info("证书请求时的公钥信息 ==== " + publicKeyStr);
            if (!Constants.KEY_ALG_NAME_NIST.equalsIgnoreCase(CommonVariable.getKeyAlgName())) {

                logger.info("加密证书明文：" + encCert);
                X509Certificate certificate = CertUtils.getCertFromStr(encCert);
//            SubjectPublicKeyInfo subPubKeyInfo = SubjectPublicKeyInfo.getInstance(certificate.getPublicKey().getEncoded());
//            try {
                String encCertPubKey = Base64.toBase64String(certificate.getPublicKey().getEncoded());
                logger.info("加密证书中的公钥信息 ==== " + encCertPubKey);
//            } catch (IOException e) {
//                logger.error("获取加密证书中的公钥信息异常{}",e);
//                result.setError(ErrorEnum.GET_CERT_PUBLIC_KEY_EXCEPTION);
//                return result;
//            }

                // 判断证书响应中的加密证书公钥和申请时的是否一致
                if (!publicKeyStr.equalsIgnoreCase(encCertPubKey)) {
                    logger.info("证书响应中的加密证书公钥和申请时的不一致");
                    result.setError(ErrorEnum.CERT_CONFIRM_PUB_KEY_NOT_SAME_REQ);
                    return result;
                }
            }
        }


        // 获取缓存的管理员证书信息
        Map<String, Object> map = OperatorCertCache.getMap();
        String oldSignSn = (String) map.get(transId);

        // 调用sdk发起签发证书响应
        Result certRespResult = certLifeCycleManager.issueCertResp(raServiceDnName,caServiceDnName,Constants.APPLY_USER_TYPE_ADMIN_2, transId);
        if (!certRespResult.isSuccess()) {
            map.remove(transId);  // 清除缓存信息
            logger.info("签发管理员证书响应");
            result.setErrorBean(certRespResult.getErrorBean());
            return result;
        }


        // 响应成功后，更新本地管理员老证书状态改为已撤销
        try {
            adminCertDao.updateAdminCertStatus(Constants.CERT_STATUS_REVOKED_3, oldSignSn);
        } catch (Exception e) {
            logger.error("修改操作员证书状态异常{}", e);
            result.setError(ErrorEnum.UPDATE_OPERATOR_CERT_STATUS_EXCEPTION);
            return result;
        }
        map.remove(transId);  // 清除缓存信息

        // 响应成功后，更新本地管理员证书状态
        try {
            adminCertDao.updateAdminCertStatus(Constants.CERT_STATUS_NORMAL_1, signSn);
        } catch (Exception e) {
            logger.error("修改操作员证书状态异常{}", e);
            result.setError(ErrorEnum.UPDATE_OPERATOR_CERT_STATUS_EXCEPTION);
            return result;
        }
        if (!Constants.KEY_ALG_NAME_NIST.equalsIgnoreCase(CommonVariable.getKeyAlgName())) {

            // 更新加密证书sn以及加密证书明文信息
            try {
                adminCertDao.updateAdminEncCertInfoBySignSn(signSn, encSn, encCert);
            } catch (Exception e) {
                logger.error("修改操作员加密证书异常{}", e);
                throw new RuntimeException();
            }
        }
        return result;
    }


    @Override
    public Result verifyOperatorCertApplyDn(String certDn) {
        //  证书DN的校验规则
        Result result = new Result();
        X500Name x500Name = null;
        try{
            x500Name = DnUtil.getRFC4519X500Name(certDn);

            String certDnFormat = x500Name.toString();
            logger.info("格式化之后的操作员证书DN："+certDnFormat);
            // 校验DN的CN值
            Result checkResult = this.checkCertDnCnValue(certDnFormat);
            if (!checkResult.isSuccess()){
                logger.info("证书的DN的特殊符号校验有误");
                result.setError(checkResult.getError());
                return result;
            }
            // 检验DN关键字系统是否支持
            // DN的每个关键字的值不能超过64字节
            //检查是否包含中文逗号
            checkResult = CertDnVerifyUtils.checkDn(x500Name);
            if (!checkResult.isSuccess()){
                result.setError(checkResult.getError());
                return result;
            }
            // 校验DN的特殊项先后顺序 DO bug修复
            checkResult = CertDnVerifyUtils.checkCertDnSort(x500Name);
            if (!checkResult.isSuccess()) {
                logger.info("证书的DN的类型先后顺序有误");
                result.setError(checkResult.getError());
                return result;
            }

           /* // 检验DN的空格  4519其实已经过滤
            checkResult = CertDnVerifyUtils.checkBlankSpace(certDnFormat);
            if (!checkResult.isSuccess()){
                logger.info("证书的DN关键字与值中有空格"+certDnFormat);
                result.setError(checkResult.getError());
                return result;
            }*/

        }catch (Exception e){
            logger.info("certDn不正确{}",e.getMessage());
            result.setError(ErrorEnum.CERT_APPLY_DN_IS_ERROR);
            return result;
        }

       /* // DN的每个关键字的值不能超过64字节
        checkResult = CertDnVerifyUtils.check64(certDnFormat);
        if (!checkResult.isSuccess()){
            logger.info("证书的DN关键字的值校验有误"+certDnFormat);
            result.setError(checkResult.getError());
            return result;
        }*/


        // 校验DN的唯一性  以uKey的cardNo决定是否是同一个用户
//        List<AdminCertDO> adminCertDOList = adminCertDao.queryAdminCertByCertDn(certDn,cardNo);
//        if (!CollectionUtils.isEmpty(adminCertDOList)) {
//            logger.info("有多个的证书的DN信息同【" + certDn + "】一致");
//            result.setError(ErrorEnum.OPERATOR_CERT_DN_HAVE_SAME_IN_CREATED);
//            return result;
//        }

        return result;
    }

    @Override
    public Result genErrorMsgContent(String transId, int errorCode, String errorMsg) {
        Result result = new Result();
        // 获取当前RA服务器签名证书DN
        String raServiceDnName = this.getRAServiceDnName();
        if (StringUtils.isBlank(raServiceDnName)){
            result.setError(ErrorEnum.GET_RA_SERVICE_DN_NAME_ERROR);
            return result;
        }

        // 获取目标CA证书DN
        String caServiceDnName = this.getCAServiceDnName();
        if (StringUtils.isBlank(caServiceDnName)){
            result.setError(ErrorEnum.GET_CA_SERVICE_DN_NAME_ERROR);
            return result;
        }

        // 获取缓存的管理员证书信息
        Map<String, Object> map = OperatorCertCache.getMap();
        map.remove(transId);

        Result errResult = certLifeCycleManager.sendErrorCMPMessage(raServiceDnName,caServiceDnName,Constants.APPLY_USER_TYPE_ADMIN_2, transId, errorCode, errorMsg);
        if (!errResult.isSuccess()){
            logger.info("错误确认消息出错："+ JsonUtils.object2Json(errResult));
            result.setErrorBean(errResult.getErrorBean());
            return result;
        }
        return result;
    }

    /**
     * 校验证书DN中CN项的值
     * @return
     */
    private Result checkCertDnCnValue(String certDn){
        Result result = new Result();
        Result raBaseDNResult = certApplyService.getRaBaseDn();
        if (!raBaseDNResult.isSuccess()) {
            return raBaseDNResult;
        }

        String raBaseDN = (String) raBaseDNResult.getInfo();
        if (certDn.length()<=3 || certDn.length() <= (3 + raBaseDN.length())){
            result.setError(ErrorEnum.OPERATOR_CERT_DN_LENGTH_IS_ERROR);
            return result;
        }
        //原来操作员只能输入CN值，现可多输入，以下判断异常 已在外层整体判断
       /* String dnValue = certDn.substring(3, certDn.length() - raBaseDN.length() - 1);
        logger.info("操作员DN值为 ========== "+ dnValue);
        if (dnValue.contains("=")){
            logger.info("操作员DN中含有特殊符号");
            result.setError(ErrorEnum.OPERATOR_CERT_DN_VALUE_HAVE_ERROR_SYMBOL);
            return result;
        }*/
        /*if (dnValue.contains(",")||dnValue.contains("=")){
            logger.info("操作员DN中含有特殊符号");
            result.setError(ErrorEnum.OPERATOR_CERT_DN_VALUE_HAVE_ERROR_SYMBOL);
            return result;
        }*/
       /* Pattern compile = Pattern.compile("\uff0c");
        Matcher matcher = compile.matcher(dnValue);
        if (matcher.find()){
            logger.info("证书主体中包含中文逗号>>>certDn:"+certDn);
            result.setError(ErrorEnum.CERT_DN_HAVE_CHINESE_COMMA);
            return result;
        }*/
        return result;
    }


    /**
     * 获取系统RA服务器签名证书DN
     * @return
     */
    private String getRAServiceDnName(){
        RaCertDO newRaCertInfo = null;
        String raServiceDnName;
        try {
            newRaCertInfo = raCertDao.getNewRaCertInfo();
            if (newRaCertInfo == null){
                return null;
            }
            String certInfo = newRaCertInfo.getCertInfo();
            X509Certificate certFromStr = CertUtils.getCertFromStr(certInfo);
            raServiceDnName = CertUtils.getSubjectByX509Cert(certFromStr);

        }catch (Exception e){
            logger.error("获取RA服务器证书异常{}",e);
            return null;
        }


        return raServiceDnName;
    }

    /**
     * 获取系统CA服务器证书DN
     * @return
     */
    private String getCAServiceDnName(){
        CaCertDO newCaCertInfo = null;
        try {
            newCaCertInfo = caCertDao.getNewCaCertInfo();
        }catch (Exception e){
            logger.error("获取CA服务器证书异常{}",e);
            return null;
        }

        if (newCaCertInfo == null){
            return null;
        }
        String certInfo = newCaCertInfo.getCertInfo();
        X509Certificate certFromStr = CertUtils.getCertFromStr(certInfo);
        String caServiceDnName = CertUtils.getSubjectByX509Cert(certFromStr);
        return caServiceDnName;
    }

}
