package com.xdja.pki.ra.service.manager.system;

import com.xdja.ca.asn1.NISTObjectIdentifiers;
import com.xdja.ca.asn1.RsaObjectIdentifiers;
import com.xdja.pki.ra.core.common.CertInfo;
import com.xdja.pki.ra.core.common.CommonVariable;
import com.xdja.pki.ra.core.commonenum.ErrorEnum;
import com.xdja.pki.ra.core.common.Result;
import com.xdja.pki.ra.core.config.CaServerConf;
import com.xdja.pki.ra.core.config.Config;
import com.xdja.pki.ra.core.constant.Constants;
import com.xdja.pki.ra.core.constant.PathConstants;
import com.xdja.pki.ra.core.pkcs7.SignedDataUtils;
import com.xdja.pki.ra.core.util.cert.CertUtils;
import com.xdja.pki.ra.core.util.cert.VerifyCert;
import com.xdja.pki.ra.manager.dao.CaCertDao;
import com.xdja.pki.ra.core.util.file.FileUtils;
import com.xdja.pki.ra.manager.dao.CaServerDao;
import com.xdja.pki.ra.manager.dao.InitDao;
import com.xdja.pki.ra.manager.dao.model.CaCertDO;
import com.xdja.pki.ra.manager.sdk.business.CaBusinessManager;
import com.xdja.pki.ra.service.manager.init.InitService;
import com.xdja.pki.ra.service.manager.system.bean.CaCertInfo;
import com.xdja.pki.ra.service.manager.utils.ParmsCommonVerifyUtil;
import org.bouncycastle.asn1.gm.GMObjectIdentifiers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.security.cert.X509Certificate;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.List;


/**
 * @author wly
 */
@Service
public class CaServiceImpl implements CaService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    CaServerDao caServerDao;

    @Autowired
    InitDao initDao;

    @Autowired
    InitService initService;

    @Autowired
    CaCertDao caCertDao;

    @Autowired
    SystemService systemService;

    @Autowired
    CaBusinessManager caBusinessManager;

    @Override
    public Result configCaServer(MultipartFile file, String caServerIp, int caServerPort) {
        Result result = caBusinessManager.testCaServerConnect(file, caServerIp, caServerPort);
        if(!result.isSuccess()){
            logger.error("测试CA连通性失败");
            return result;
        }

        //判断操作步骤是否正确
        Result stepResult = initService.getOperateStep();
        logger.info("当前初始化步骤为========" + stepResult.getInfo());
        if (!stepResult.getInfo().equals(Constants.HSM_SERVER_PAGE)) {
            logger.info("初始化步骤数错误");
            result.setError(ErrorEnum.INIT_STEP_ERROR);
            return result;
        }

        //修改CA服务信息
        Result updateCAResult = updateConfigCaInfo(file, caServerIp, caServerPort,false);
        if (!updateCAResult.isSuccess()) {
            return updateCAResult;
        }

        //修改步骤数
        return initService.updateInitStep(Constants.CA_SERVER_PAGE);
    }

    @Override
    public Result updateLessCAServer(String caServerIp, int caServerPort) {

        Result result = new Result();
        boolean isIpv4 = ParmsCommonVerifyUtil.isIpv4(caServerIp);
        if (!isIpv4) {
            logger.error("IP地址格式错误！");
            return Result.failure(ErrorEnum.ILLEGAL_REQUEST_PARAMETER);
        }
        //修改CA服务配置文件
        try {
            Config config = systemService.getConfigFile(Constants.CONFIG_JSON_FILE_NAME);
            CaServerConf caServerConf = config.getCaServerConf();
            caServerConf.setCaServerIp(caServerIp);
            caServerConf.setCaServerPort(caServerPort);
            systemService.updateConfigFile(config);
        } catch (Exception e) {
            logger.error("配置CA服务操作config.json异常", e);
            result.setError(ErrorEnum.CONFIG_JSON_FILE_OPERATION_ERROR);
            return result;
        }
        CommonVariable.setCaServiceIp(caServerIp);
        CommonVariable.setCaServicePort(caServerPort);
        return result;
    }

    @Override
    public Result updateConfigCaServer(MultipartFile file, String caServerIp, int caServerPort) {

        Result result = caBusinessManager.testCaServerConnect(file, caServerIp, caServerPort);
        if(!result.isSuccess()){
            logger.error("测试CA连通性失败");
            return result;
        }

        //修改CA服务信息
        return updateConfigCaInfo(file, caServerIp, caServerPort,true);
    }

    /**
     * 修改CA服务信息 用于配置/更新CA服务
     *
     * isUpdate : true 为更新CA配置，需要判断秘钥算法是否一致 ；false 不需要判断
     *
     */
    public Result updateConfigCaInfo(MultipartFile file, String caServerIp, int caServerPort, boolean isUpdate) {
        Result result = new Result();
        boolean isIpv4 = ParmsCommonVerifyUtil.isIpv4(caServerIp);
        if (!isIpv4) {
            logger.error("IP地址格式错误！");
            return Result.failure(ErrorEnum.ILLEGAL_REQUEST_PARAMETER);
        }

        //获取文件名
        String trustCertName = file.getOriginalFilename();
        String fileName = trustCertName.substring(trustCertName.lastIndexOf("\\") + 1);

        //修改CA服务配置文件
        //修改CA服务信息
        Config config;
        try {
            config = systemService.getConfigFile(Constants.CONFIG_JSON_FILE_NAME);
            CaServerConf caServerConf = config.getCaServerConf();
            if(null == caServerConf){
                caServerConf = new CaServerConf();
            }
            caServerConf.setCaServerIp(caServerIp);
            caServerConf.setCaServerPort(caServerPort);
            caServerConf.setTrustCertName(fileName);
            config.setCaServerConf(caServerConf);
        } catch (Exception e) {
            logger.error("配置CA服务操作config.json异常", e);
            result.setError(ErrorEnum.CONFIG_JSON_FILE_OPERATION_ERROR);
            return result;
        }
        //file的bytes
        byte[] fileBytes = new byte[0];
        try {
            fileBytes = file.getBytes();
        } catch (Exception e) {
            logger.info("文件转换bytes异常");
            result.setError(ErrorEnum.FILE_TO_BYTES_ERROR);
            return result;
        }

        //验证证书链CA证书
        Result verifyResult = VerifyCert.verifyP7bCertList(fileBytes);
        if (!verifyResult.isSuccess()) {
            result.setError(verifyResult.getError());
            return result;
        }

        //读取解析证书链
        List<X509Certificate> certs = null;
        try {
//            byte[] base64Bytes = Base64.decode(fileBytes);
            certs = SignedDataUtils.resolveCertChain(fileBytes);
        } catch (Exception e) {
            logger.error("配置CA服务解析证书链异常", e);
            result.setError(ErrorEnum.CONVERT_CERT_ERROR);
            return result;
        }

        //获取p7b的base64值
        String p7bContent = null;
        try {
            p7bContent = new String(fileBytes);
        } catch (Exception e) {
            logger.error("配置CA服务读取证书链base64值异常", e);
            result.setError(ErrorEnum.CERT_P7B_INFO_READ_ERROR);
            return result;
        }

        // 从导入的CA信任证书链中获取第一个为当前RA系统签发的上级CA证书
        X509Certificate cert = certs.get(0);
        String certPem = null;
        try {
            certPem = CertUtils.writeObject(cert);
        } catch (Exception e) {
            logger.error("读取CA服务证书pem值异常", e);
            result.setError(ErrorEnum.CERT_SERVER_INFO_READ_ERROR);
            return result;
        }
        //获取证书的秘钥算法
        String keyAlg;
        try {
            keyAlg = CertUtils.getKeyAlg(cert);
            //如果系统配置SWXA的密码机 则不支持NIST
            if(Constants.HSM_SERVER_SWXA_2 == CommonVariable.getIsHsm() && Constants.KEY_ALG_NAME_NIST.equalsIgnoreCase(keyAlg)){
                logger.error("三未信安密码机不支持NIST256算法");
                result.setError(ErrorEnum.SWXA_HSM_NOT_SUPPORT_NIST);
                return result;
            }

            //更新CA证书，判断新证书秘钥算法和老证书是否一致
            logger.info("=============CommonVariable.getKeyAlgName()"+ CommonVariable.getKeyAlgName());
            logger.info("=============certInfo.getKeyAlg()"+keyAlg);
            if(isUpdate && !CommonVariable.getKeyAlgName() .equalsIgnoreCase(keyAlg)){
                logger.info("导入CA证书秘钥算法和原系统不一致");
                result.setError(ErrorEnum.CA_ALGORITHM_IS_NOT_SAME_OLD);
                return result;
            }
            //写到配置文件

            config.setKeyAlgName(keyAlg);
            systemService.updateConfigFile(config);
        } catch (Exception e) {
            logger.error("秘钥算法写到配置文件异常", e);
            result.setError(ErrorEnum.CONFIG_JSON_FILE_OPERATION_ERROR);
            return result;
        }

        //将CA证书存到指定路径
        try {
            FileUtils.saveFile(certPem, PathConstants.CA_SERVICE_CERT_FILE_PATH);
        } catch (Exception e) {
            logger.info("证书保存在指定路径中错误", e);
            result.setInfo(ErrorEnum.SAVE_CA_CERT_IS_ERROR);
            return result;
        }
        //当前时间戳
        String date = new SimpleDateFormat("yyyyMMddHHmmss").format(Calendar.getInstance().getTime());

        //将CA证书存到历史CA证书路径  该文件夹用作通道认证是
        try {
            String path = PathConstants.SUPER_CA_CERTS_FILE_PATH + "/" + "caCert" + "-" + date + ".cer";
            FileUtils.saveFile(certPem, path);
        } catch (Exception e) {
            logger.info("证书保存在指定路径中错误", e);
            result.setInfo(ErrorEnum.SAVE_CA_CERT_IS_ERROR);
            return result;
        }

        //将CA证书链存到指定路径
        try {
            FileUtils.saveFile(p7bContent, PathConstants.CA_TRUST_SERVICE_CERT_FILE_PATH);
        } catch (Exception e) {
            logger.info("证书链保存在指定路径中错误", e);
            result.setInfo(ErrorEnum.SAVE_CA_CERT_IS_ERROR);
            return result;
        }
        CaCertDO caCertDO = new CaCertDO();
        //证书主题
        caCertDO.setCertDn(CertUtils.getSubjectByX509Cert(cert));
        //证书序列号
        caCertDO.setCertSn(cert.getSerialNumber().toString(16).toLowerCase());
        //公钥算法
        caCertDO.setPublicKeyAlg(keyAlg);
        //私钥长度
        int publicKeyLength = 0;
        try {
            publicKeyLength = CertUtils.getPublicKeyLength(cert);
            caCertDO.setPrivateKeyLength(publicKeyLength);
        } catch (Exception e) {
            logger.error("获取证书公钥长度异常", e);
            result.setError(ErrorEnum.GET_CERT_PUBLIC_KEY_LENGTH_EXCEPTION);
            return result;
        }
        //签名算法
        caCertDO.setSignAlg(cert.getSigAlgName());
        //证书内容
        caCertDO.setCertInfo(certPem);
        //证书链
        caCertDO.setCertP7b(p7bContent);
        //证书生效时间
        caCertDO.setEffectiveTime(new Timestamp(cert.getNotBefore().getTime()));
        //证书失效时间
        caCertDO.setFailureTime(new Timestamp(cert.getNotAfter().getTime()));
        caCertDO.setGmtCreate(new Timestamp(System.currentTimeMillis()));
        caCertDO.setGmtUpdate(new Timestamp(System.currentTimeMillis()));
        //存数据库-------------
        try {
            caServerDao.insertCaCertInfo(caCertDO);
        } catch (Exception e) {
            logger.error("CA基本信息存ca_cert异常", e);
            result.setError(ErrorEnum.INSERT_CA_CERT_INFO_ERROR);
            return result;
        }

        // 缓存CA配置信息
        CommonVariable.setCaServiceIp(caServerIp);
        CommonVariable.setCaServicePort(caServerPort);
        CommonVariable.setCaServiceCert(cert);
        CommonVariable.setKeyAlgName(keyAlg);
        CommonVariable.setTrustCaCerts(null);
        CommonVariable.setCaCerts(null);

        return result;
    }



    /**
     * 导入证书链 验证 获取信息
     *
     * isUpdate : true 为更新CA配置，需要判断秘钥算法是否一致 ；false 不需要判断
     *
     */
    @Override
    public Result getVerifyCaCertInfo(MultipartFile file, boolean isUpdate) {
        Result result = new Result();

        //file的bytes
        byte[] fileBytes = new byte[0];
        try {
            fileBytes = file.getBytes();
        } catch (Exception e) {
            logger.info("文件转换bytes异常");
            result.setError(ErrorEnum.FILE_TO_BYTES_ERROR);
            return result;
        }

        //验证证书链CA证书
        Result verifyResult = VerifyCert.verifyP7bCertList(fileBytes);
        if (!verifyResult.isSuccess()) {
            result.setError(verifyResult.getError());
            return result;
        }

        //读取解析证书链
        List<X509Certificate> certs = null;
        try {
            certs = SignedDataUtils.resolveCertChain(fileBytes);
        } catch (Exception e) {
            logger.error("配置CA服务解析证书链异常", e);
            result.setError(ErrorEnum.CONVERT_CERT_ERROR);
            return result;
        }

        // 从导入的CA信任证书链中获取第一个为当前RA系统签发的上级CA证书
        X509Certificate cert = certs.get(0);

        //获取证书的秘钥算法
        String keyAlg;
        try {
            keyAlg = CertUtils.getKeyAlg(cert);
            //更新CA证书，判断新证书秘钥算法和老证书是否一致
            logger.info("=============CommonVariable.getKeyAlgName()"+ CommonVariable.getKeyAlgName());
            logger.info("=============certInfo.getKeyAlg()"+keyAlg);
            if(isUpdate && !CommonVariable.getKeyAlgName() .equalsIgnoreCase(keyAlg)){
                logger.info("导入CA证书秘钥算法和原系统不一致");
                result.setError(ErrorEnum.CA_ALGORITHM_IS_NOT_SAME_OLD);
                return result;
            }
        } catch (Exception e) {
            logger.error("获取证书密钥算法异常", e);
            result.setError(ErrorEnum.GET_CERT_ALG_NAME_LENGTH_EXCEPTION);
            return result;
        }

        CaCertInfo caCertInfo = getCAInfo(cert);
        result.setInfo(caCertInfo);
        return result;
    }

    @Override
    public Result getCaConfigInfo() {

        Result result = new Result();
        Config config = null;
        try {
            config = systemService.getConfigFile(Constants.CONFIG_JSON_FILE_NAME);
        } catch (Exception e) {
            result.setError(ErrorEnum.CONFIG_JSON_FILE_OPERATION_ERROR);
            return result;
        }

        CaServerConf caServerConf = config.getCaServerConf();
        if (null == caServerConf) {
            logger.error("获取CA服务详情为空");
            result.setError(ErrorEnum.GET_CA_SERVER_INFO_IS_EMPTY);
            return result;
        }
        X509Certificate caCert = CommonVariable.getCaServiceCert();
        CaCertInfo caCertInfo = getCAInfo(caCert);
        caCertInfo.setCaServerConf(caServerConf);

        result.setInfo(caCertInfo);

        return result;
    }

    private CaCertInfo getCAInfo(X509Certificate caCert){
        String SIGN_ALGO_SM3WITHSM2 = "SM3WithSM2";
        String SIGN_ALGO_SHA256WithECDSA = "sha256WithECDSA";
        String SIGN_ALGO_SHA256WithRSA = "sha256WithRSA";
        String SIGN_ALGO_SHA1WithRSA = "sha-1WithRSA";

        //证书主题
        String certDn = CertUtils.getSubjectByX509Cert(caCert);
        String sigAlgOID = caCert.getSigAlgOID();
        String sigAlgName = caCert.getSigAlgName();
        String sm3withsm2OID = GMObjectIdentifiers.sm2sign_with_sm3.getId();
        if(sigAlgOID.equals(sm3withsm2OID)) {
            sigAlgName = SIGN_ALGO_SM3WITHSM2;
        }
        if(sigAlgOID.equals(NISTObjectIdentifiers.nistSignAlgorithm.getId())){
            sigAlgName = SIGN_ALGO_SHA256WithECDSA;
        }
        if(sigAlgOID.equals(RsaObjectIdentifiers.sha256WithRSA.getId())){
            sigAlgName = SIGN_ALGO_SHA256WithRSA;
        }
        if(sigAlgOID.equals(RsaObjectIdentifiers.sha1WithRSA.getId())){
            sigAlgName = SIGN_ALGO_SHA1WithRSA;
        }
        CaCertInfo caCertInfo = new CaCertInfo();
        caCertInfo.setCertDN(certDn);
        caCertInfo.setSignAlgName(sigAlgName);
        //证书生效时间
        caCertInfo.setNotBefore(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(caCert.getNotBefore()));
        //证书失效时间
        caCertInfo.setNotAfter(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(caCert.getNotAfter()));

        return caCertInfo;
    }

    @Override
    public CaCertDO getCaConfigInfoByTime() {

        //获取当前创建时间最大的一条记录的id
        long id = caServerDao.getCaCertId();
        //旧的CA证书信息
        CaCertDO oldCaCertDO = caServerDao.getCaCert(id);

        return oldCaCertDO;
    }


    @Override
    public Result getNewCaCertInfo() {
        Result result = new Result();
        CaCertDO newCaCertInfo = caCertDao.getNewCaCertInfo();
        if (newCaCertInfo == null) {
            result.setError(ErrorEnum.GET_CA_CERT_INFO_IS_EMPTY);
            return result;
        }
        result.setInfo(newCaCertInfo);
        return result;
    }

    /**
     * 获取系统CA服务器证书DN
     * @return
     */
    @Override
    public String getCAServiceDnName(){
        CaCertDO newCaCertInfo = null;
        try {
            newCaCertInfo = caCertDao.getNewCaCertInfo();
        }catch (Exception e){
            logger.error("获取CA服务器证书异常{}",e);
            return null;
        }

        if (newCaCertInfo == null){
            return null;
        }
        String certInfo = newCaCertInfo.getCertInfo();
        X509Certificate certFromStr = CertUtils.getCertFromStr(certInfo);
        String caServiceDnName = CertUtils.getSubjectByX509Cert(certFromStr);
        return caServiceDnName;
    }

}
