package com.xdja.pki.ra.service.manager.system;

import com.xdja.ca.sdk.CmpApi;
import com.xdja.ca.utils.DnUtil;
import com.xdja.pki.gmssl.crypto.utils.GMSSLSM2KeyUtils;
import com.xdja.pki.gmssl.tomcat.utils.GMSSLTomcatUtils;
import com.xdja.pki.gmssl.x509.utils.GMSSLP10Utils;
import com.xdja.pki.ra.core.common.CommonVariable;
import com.xdja.pki.ra.core.commonenum.ErrorEnum;
import com.xdja.pki.ra.core.common.Result;
import com.xdja.pki.ra.core.config.Config;
import com.xdja.pki.ra.core.config.RaServerConfig;
import com.xdja.pki.ra.core.constant.Constants;
import com.xdja.pki.ra.core.constant.PathConstants;
import com.xdja.pki.ra.core.pkcs7.SignedDataUtils;
import com.xdja.pki.ra.core.util.cert.*;
import com.xdja.pki.ra.core.util.file.FileUtils;
import com.xdja.pki.ra.core.util.file.ZipUtils;
import com.xdja.pki.ra.core.util.spring.SpringUtils;
import com.xdja.pki.ra.manager.dao.CaCertDao;
import com.xdja.pki.ra.manager.dao.RaCertDao;
import com.xdja.pki.ra.manager.dao.model.CaCertDO;
import com.xdja.pki.ra.manager.dao.model.RaCertDO;
import com.xdja.pki.ra.manager.dto.RaServerCertDTO;
import com.xdja.pki.ra.service.manager.baseuser.bean.EncryptUserInfo;
import com.xdja.pki.ra.service.manager.init.InitService;
import com.xdja.pki.ra.service.manager.init.bean.HsmResp;
import com.xdja.pki.ra.service.manager.utils.CertContentInfoUtil;
import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.math.BigInteger;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.cert.X509Certificate;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author cl
 */
@Service
public class RaServerImpl implements RaServer {
    @Autowired
    CaCertDao caCertDao;

    @Autowired
    RaCertDao raCertDao;

    @Autowired
    CertContentInfoUtil certContentInfoUtil;

    @Autowired
    InitService initService;

    @Autowired
    SystemService systemService;

    private String tomcatPath = System.getProperty("catalina.home");

    @Value("${ra.system.https.port}")
    private String httpsPort;

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Override
    public Result updateRaServerConfig(Integer keyIndex, String priKeyPwd, MultipartFile certChain, MultipartFile enc) {
        Result result = Result.success();
        //密钥索引、密钥访问控制码不修改 需要从配置文件中获取
        int keyDex = 0;
        String priPwd = null;
        //更新RA服务器配置
        //if (Constants.KEY_ALG_NAME_SM2.equalsIgnoreCase(CommonVariable.getKeyAlgName())) {
        if (Constants.HSM_SERVER_XDJA_1 == CommonVariable.getIsHsm() && Constants.KEY_ALG_NAME_SM2.equalsIgnoreCase(CommonVariable.getKeyAlgName())) {
            try {
                Config config =  (Config) SpringUtils.getBean("configFile");
                RaServerConfig raServerConfig = config.getRaServerConfig();
                //都不为空
                if (keyIndex != null && !StringUtils.isBlank(priKeyPwd)) {
                    keyDex = keyIndex;
                    priPwd = priKeyPwd;
                    //都为空
                } else if (keyIndex == null && StringUtils.isBlank(priKeyPwd)) {
                    keyDex = raServerConfig.getKeyIndex();
                    priPwd = HsmUtils.getDecrypt(raServerConfig.getPriKeyPwd());
                } else {
                    result.setError(ErrorEnum.ILLEGAL_REQUEST_PARAMETER);
                    return result;
                }
            } catch (Exception e) {
                logger.info("查询配置文件失败", e);
                result.setError(ErrorEnum.CONFIG_JSON_FILE_OPERATION_ERROR);
                return result;
            }
            result = raServerSaveByHsm(keyDex, priPwd, certChain, enc, Constants.SYSTEM_UPDATE_RA_TYPE);
            result.setLogContent("，密钥索引=" + keyDex + "，密钥访问控制码=" + priPwd);
        } else {
            byte[] signCertChainBytes = null;
            byte[] encEncCertBytes = null;
            try {
                signCertChainBytes = certChain.getBytes();
                encEncCertBytes = enc.getBytes();
            } catch (IOException e) {
                logger.error("获取输入文件异常", e);
                result.setError(ErrorEnum.ILLEGAL_REQUEST_PARAMETER);
                return result;
            }
            if (signCertChainBytes == null || encEncCertBytes == null) {
                result.setError(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
                return result;
            }
            result = raServerSaveByBC(certChain.getOriginalFilename(), signCertChainBytes, enc.getOriginalFilename(), encEncCertBytes, Constants.SYSTEM_UPDATE_RA_TYPE);
        }
        return result;
    }

    @Override
    public Result initRaServerConfig(Integer keyIndex, String priKeyPwd, MultipartFile certChain, MultipartFile enc) {
        Result result = Result.success();
        //判断操作步骤是否正确
        Result stepResult = initService.getOperateStep();
        logger.info("当前初始化步骤为========" + stepResult.getInfo());
        if (!stepResult.getInfo().equals(Constants.CA_SERVER_PAGE)) {
            logger.info("初始化步骤数错误");
            result.setError(ErrorEnum.INIT_STEP_ERROR);
            return result;
        }
        // 目前迭代实现的功能是  如果是sm2的 初始化配置默认都是用硬的  RSA的默认使用软的配置RA服务器证书  20190916
        // 此处的判断需要后续可能需要根据CommonVariable.isUseHsm()来判断软硬件使用情况   Y
        if (Constants.HSM_SERVER_XDJA_1 == CommonVariable.getIsHsm() && Constants.KEY_ALG_NAME_SM2.equalsIgnoreCase(CommonVariable.getKeyAlgName())) {
            result = this.raServerSaveByHsm(keyIndex, priKeyPwd, certChain, enc, Constants.INIT_UPDATE_RA_TYPE);
        } else {
            byte[] signCertChainBytes = null;
            byte[] encEncCertBytes = null;
            try {
                signCertChainBytes = certChain.getBytes();
                encEncCertBytes = enc.getBytes();
            } catch (IOException e) {
                logger.error("获取输入文件异常", e);
                result.setError(ErrorEnum.ILLEGAL_REQUEST_PARAMETER);
                return result;
            }
            if (signCertChainBytes == null || encEncCertBytes == null) {
                result.setError(ErrorEnum.MISSING_REQUIRED_PARAMETERS);
                return result;
            }
            result = this.raServerSaveByBC(certChain.getOriginalFilename(), signCertChainBytes, enc.getOriginalFilename(), encEncCertBytes, true);
        }

        return result;
    }


    @Override
    public Result getRaServerCertDetails(int type) {
        Result result = Result.success();
        RaServerCertDTO raCertInfo = null;
        //数据库获取RA服务器签名或加密证书
        try {
            raCertInfo = raCertDao.getNewRaCertInfo(type);
        } catch (Exception e) {
            logger.info("获取RA服务器证书为空", e);
            result.setError(ErrorEnum.GET_RA_SERVER_CERT_INFO_EMPTY);
            return result;
        }
        X509Certificate cert = CertUtils.getCertFromStr(raCertInfo.getCertInfo());
        Map<String, Object> certContentInfo;
        try {
            certContentInfo = certContentInfoUtil.getCertContentInfo(cert);
        } catch (Exception e) {
            logger.info("证书详情格式读取异常", e);
            result.setError(ErrorEnum.CERT_DETAIL_FORMAT_ERROR);
            return result;
        }
        result.setInfo(certContentInfo);
        return result;
    }


    @Override
    public Result getRaServerCertInfo() {
        Result result = Result.success();
        RaCertDO newRaCertInfo = null;
        //获取RA最新服务器证书
        try {
            newRaCertInfo = raCertDao.getNewRaCertInfo();
            result.setInfo(newRaCertInfo);
        } catch (Exception e) {
            logger.info("获取RA服务器最新签名证书失败", e);
            result.setError(ErrorEnum.GET_RA_SERVER_CERT_INFO_EMPTY);
            return result;
        }
        return result;
    }

    @Override
    public Result getRaServerConfigInfo() {
        Result result = Result.success();
        RaServerConfig raServerConfig = null;
        try {
            Config config = systemService.getConfigFile(Constants.CONFIG_JSON_FILE_NAME);
            raServerConfig = config.getRaServerConfig();
        } catch (Exception e) {
            logger.info("读取配置文件错误", e);
            result.setError(ErrorEnum.CONFIG_JSON_FILE_OPERATION_ERROR);
            return result;
        }
        result.setInfo(raServerConfig);
        return result;
    }


    /**
     * 通过密码机验证并保存RA服务器信息
     *
     * @param keyIndex
     * @param priKeyPwd
     * @param certChain
     * @param enc
     * @param initType
     * @return
     */
    private Result raServerSaveByHsm(Integer keyIndex, String priKeyPwd, MultipartFile certChain, MultipartFile enc, boolean initType) {
        logger.info(" 通过密码机验证并保存RA服务器信息 keyIndex:" + keyIndex +", priKeyPwd : " + priKeyPwd);
        Result result = Result.success();
        boolean pwdBoolean = false;
        try {
            pwdBoolean = GMSSLSM2KeyUtils.getPrivateKeyAccessRightFromYunHsm(keyIndex, priKeyPwd);
        } catch (Exception e) {
            logger.info("异常：密钥访问控制码错误", e);
            result.setError(ErrorEnum.RA_SERVER_PRIKEYPWD_ERROR);
            return result;
        }

        if (!pwdBoolean) {
            logger.info("密钥访问控制码错误");
            result.setError(ErrorEnum.RA_SERVER_PRIKEYPWD_ERROR);
            return result;
        }
        //获取CA根证书
        CaCertDO newCaCertInfo = caCertDao.getNewCaCertInfo();
        if (newCaCertInfo == null) {
            logger.info("获取CA证书信息为空");
            result.setError(ErrorEnum.GET_CA_CERT_INFO_IS_EMPTY);
            return result;
        }
        X509Certificate caRootCert = CertUtils.getCertFromStr(newCaCertInfo.getCertInfo());

        //获取加密证书
        String encCertB64String = null;
        X509Certificate encCert = null;
        try {
            encCertB64String = CertUtils.getEncCertByEnvelopDataByHsm(keyIndex, priKeyPwd, enc);
            encCert = CertUtils.getCertFromStr(encCertB64String);
        } catch (Exception e) {
            logger.info("解析信封失败", e);
            result.setError(ErrorEnum.GET_RA_SERVER_ENC_CERT_ERROR);
            return result;
        }

        //certChain的bytes
        byte[] fileBytes = null;
        try {
            fileBytes = certChain.getBytes();
        } catch (Exception e) {
            logger.info("文件转换bytes异常");
            result.setError(ErrorEnum.FILE_TO_BYTES_ERROR);
            return result;
        }

        //验证证书链
        Result verifyResult = VerifyCert.verifyP7bCertList(fileBytes);
        if (!verifyResult.isSuccess()) {
            result.setError(verifyResult.getError());
            return result;
        }

        //读取解析证书链
        List<X509Certificate> certs = null;
        try {
            certs = SignedDataUtils.resolveCertChain(fileBytes);
            if (certs.size() < 2) {
                logger.error("RA服务器证书链大小错误");
                result.setError(ErrorEnum.CONVERT_CERT_ERROR);
                return result;
            }
        } catch (Exception e) {
            logger.error("RA服务解析证书链异常", e);
            result.setError(ErrorEnum.CONVERT_CERT_ERROR);
            return result;
        }

        //取第一张证书为用户证书,获得用户证书的pem值,即证书内容。
        X509Certificate signCert = certs.get(0);
        logger.info("RA的签名证书: " + signCert);

        // 从RA的签名服务器证书链中，截取获取CA证书链
        certs.remove(0);

        //获取p7b的base64值
        String p7bContent = null;
        try {
            p7bContent = Base64.toBase64String(fileBytes);
        } catch (Exception e) {
            logger.error("配置RA服务读取证书链base64值异常", e);
            result.setError(ErrorEnum.CERT_P7B_INFO_READ_ERROR);
            return result;
        }

        //判断是否为当前CA签发
        try {
            //判断DN是否正确
            String signDN = CertUtils.getIssuerByX509Cert(signCert);
            String encDN = CertUtils.getIssuerByX509Cert(encCert);
            String caDN = CertUtils.getSubjectByX509Cert(caRootCert);
            logger.info("signDN: " + signDN + "  encDN: " + encDN + "  caDN: " + caDN);
            if (signDN.equals(caDN) ? (encDN.equals(caDN) ? false : true) : true) {
                logger.info("DN不一致：非当前CA签发的服务器证书");
                result.setError(ErrorEnum.CERT_P7B_VERIFY_ERROR);
                return result;
            }
            PublicKey publicKey = caRootCert.getPublicKey();

            //签名证书
            boolean signVerify = HsmUtils.verifyCertByYunHsm(signCert, publicKey);
            if (!signVerify) {
                logger.info("签名证书验签：非当前CA签发的服务器证书");
                result.setError(ErrorEnum.CERT_P7B_VERIFY_ERROR);
                return result;
            }

            //加密证书
            boolean encVerify = HsmUtils.verifyCertByYunHsm(encCert, publicKey);
            if (!encVerify) {
                logger.info("加密证书验签：非当前CA签发的服务器证书");
                result.setError(ErrorEnum.GET_RA_SERVER_ENC_CERT_ERROR);
                return result;
            }
        } catch (Exception e) {
            logger.info("密码机异常", e);
            result.setError(ErrorEnum.CERT_P7B_VERIFY_ERROR);
            return result;
        }

        String signCertPem = null;
        String encCertPem = null;
        try {
            signCertPem = CertUtils.writeObject(signCert);
            encCertPem = CertUtils.writeObject(encCert);
        } catch (Exception e) {
            logger.error("读取RA服务证书pem值异常", e);
            result.setError(ErrorEnum.READ_RA_SERVER_CERT_PEM_ERROR);
            return result;
        }
        try {
            String stringPublicKey = Base64.toBase64String(signCert.getPublicKey().getEncoded());
            PublicKey signPublicKey = null;
            //密码机中获取公钥
            try {
                signPublicKey = GMSSLSM2KeyUtils.getSignPublicKeyByYunhsm(keyIndex);
            } catch (Exception e) {
                logger.error("密钥索引错误", e);
                result.setError(ErrorEnum.RA_SERVER_KEYINDEX_ERROR);
                return result;
            }
            String stringSignPublicKey = Base64.toBase64String(signPublicKey.getEncoded());
            if (!stringPublicKey.equals(stringSignPublicKey)) {
                logger.info("证书中的公钥和申请时的公钥不一致");
                result.setError(ErrorEnum.CERT_PUB_KEY_NOT_SAME_REQ_PUB_KEY);
                return result;
            }
        } catch (Exception e) {
            logger.error("获取公钥异常", e);
            result.setError(ErrorEnum.GET_PUB_KEY_INFO_EXCEPTION);
            return result;
        }

        RaCertDO raSignCertDO = new RaCertDO();
        RaCertDO raEncCertDO = new RaCertDO();
        long pairCertIndex = System.nanoTime();
        raSignCertDO.setPairCertIndex(pairCertIndex);
        raSignCertDO.setCaCertId(newCaCertInfo.getId());
        raSignCertDO.setCertDn(CertUtils.getSubjectByX509Cert(signCert));
        raSignCertDO.setCertSn(signCert.getSerialNumber().toString(16).toLowerCase());
        raSignCertDO.setPublicKeyAlg(signCert.getPublicKey().getAlgorithm());
        raSignCertDO.setCertType(Constants.CERT_TYPE_SIGN_2);
        try {
            int publicKeyLength = CertUtils.getPublicKeyLength(signCert);
            raSignCertDO.setPrivateKeyLength(publicKeyLength);
        } catch (Exception e) {
            logger.error("获取证书公钥长度异常", e);
            result.setError(ErrorEnum.GET_CERT_PUBLIC_KEY_LENGTH_EXCEPTION);
            return result;
        }
        //存数据库-------------
        try {

            raSignCertDO.setSignAlg(signCert.getSigAlgName());
            raSignCertDO.setHsmKeyIndex(keyIndex);
            //DO 20200914 不存了
//            String pwd = priKeyPwd;
//            if (Constants.DATA_IS_ENCRYPTION_1.equals(CommonVariable.getIsEncrypt())) {
//                //更新证书才可能会进来
//                pwd = HsmUtils.getEncrypt(priKeyPwd);
//            }
//            raSignCertDO.setHsmPriKeyPin(pwd);
            raSignCertDO.setCertInfo(signCertPem);
            raSignCertDO.setCertP7b(p7bContent);
            raSignCertDO.setGmtCreate(new Timestamp(System.currentTimeMillis()));
            raSignCertDO.setGmtUpdate(new Timestamp(System.currentTimeMillis()));
            BeanUtils.copyProperties(raSignCertDO, raEncCertDO);
            raEncCertDO.setCertP7b(null);
            raEncCertDO.setCertDn(CertUtils.getSubjectByX509Cert(encCert));
            raEncCertDO.setCertSn(encCert.getSerialNumber().toString(16).toLowerCase());
            raEncCertDO.setCertType(Constants.CERT_TYPE_ENC_3);
            raEncCertDO.setSignAlg(encCert.getSigAlgName());
            raEncCertDO.setCertInfo(encCertPem);

            raCertDao.insertRaCertInfo(raSignCertDO);
            raCertDao.insertRaCertInfo(raEncCertDO);
        } catch (Exception e) {
            logger.error("RA服务器证书存ra_cert异常", e);
            result.setError(ErrorEnum.INSERT_RA_CERT_INFO_ERROR);
            return result;
        }

        try {
            FileUtils.saveFile(signCertPem, PathConstants.RA_SERVICE_CERT_FILE_PATH);
        } catch (Exception e) {
            logger.info("证书保存在指定路径中错误", e);
            result.setError(ErrorEnum.SAVE_RA_CERT_IS_ERROR);
            return result;
        }

        //启用https
        try {
            GMSSLTomcatUtils.openHttpsPortByYunHsm(certs, signCert, encCert, keyIndex, priKeyPwd, tomcatPath, Integer.parseInt(httpsPort));
        } catch (Exception e) {
            logger.error("启用https异常", e);
            result.setError(ErrorEnum.START_TOMCAT_HTTPS);
            return result;
        }

        // 修改配置文件
        try {
            //获取配置文件
            Config config = systemService.getConfigFile(Constants.CONFIG_JSON_FILE_NAME);
            //修改初始化步骤
            if (initType) {
                config.setInitStep(Constants.RA_SERVER_PAGE);
            }
            RaServerConfig raServerConfig = config.getRaServerConfig();
            if (null == raServerConfig) {
                raServerConfig = new RaServerConfig();
            }
            String signCertName = certChain.getOriginalFilename();
            raServerConfig.setSignCertName(signCertName.substring(signCertName.lastIndexOf("\\") + 1));
            String encCertName = enc.getOriginalFilename();
            raServerConfig.setEncCertName(encCertName.substring(encCertName.lastIndexOf("\\") + 1));
            if (keyIndex != 0 && !StringUtils.isBlank(priKeyPwd)) {
                raServerConfig.setKeyIndex(keyIndex);
                raServerConfig.setPriKeyPwd(priKeyPwd);
                CommonVariable.setKeyPwd(priKeyPwd);
            }
            config.setRaServerConfig(raServerConfig);
            config.setSigAlgName(signCert.getSigAlgName());
            if(!initType){
                config = HsmUtils.cryptConfigRaPwd(true, config);
            }
            systemService.updateConfigFile(config);
        } catch (Exception e) {
            logger.info("修改初始化配置文件失败", e);
            result.setError(ErrorEnum.CONFIG_JSON_FILE_OPERATION_ERROR);
            return result;
        }

        // 缓存RA配置信息
        CommonVariable.setRaSingSn(signCert.getSerialNumber().toString(16).toLowerCase());
        CommonVariable.setKeyIndex(keyIndex);
     //   CommonVariable.setKeyPwd(priKeyPwd);
        CommonVariable.setSigAlgName(signCert.getSigAlgName());
        return result;
    }

    /**
     * 通过BC验证并保存RA服务器信息
     *
     * @param signCertName
     * @param signCertChain
     * @param encCertName
     * @param encEncCert
     * @param initType
     * @return
     */
    private Result raServerSaveByBC(String signCertName, byte[] signCertChain, String encCertName, byte[] encEncCert, boolean initType) {
        Result result = Result.success();
        //获取CA根证书
        CaCertDO newCaCertInfo = caCertDao.getNewCaCertInfo();
        if (newCaCertInfo == null) {
            logger.info("获取CA证书信息为空");
            result.setError(ErrorEnum.GET_CA_CERT_INFO_IS_EMPTY);
            return result;
        }
        X509Certificate caRootCert = CertUtils.getCertFromStr(newCaCertInfo.getCertInfo());

        //获取加密证书
        String encCertB64String;
        X509Certificate encCert;
        PrivateKey encPriKey;
        PrivateKey signPriKey;
        try {
            boolean isEncPwd = false;
            if(initType){
                isEncPwd = true;
            }
            encPriKey = KeyStoreUtils.readServerPrivateKey(false, PathConstants.SOFT_ALG_FOLDER_PATH_TEMP, isEncPwd);
            encCertB64String = CertUtils.getDataFromEnvelopDataByBc(encPriKey.getEncoded(), encEncCert, CommonVariable.getKeyAlgName());
            encCert = CertUtils.getCertFromStr(encCertB64String);

            signPriKey = KeyStoreUtils.readServerPrivateKey(true, PathConstants.SOFT_ALG_FOLDER_PATH_TEMP, isEncPwd);
        } catch (Exception e) {
            logger.info("解析信封失败", e);
            result.setError(ErrorEnum.GET_RA_SERVER_ENC_CERT_ERROR);
            return result;
        }

        //验证证书链CA证书
        Result verifyResult = VerifyCert.verifyP7bCertList(signCertChain);
        if (!verifyResult.isSuccess()) {
            result.setError(verifyResult.getError());
            return result;
        }

        //读取解析证书链
        List<X509Certificate> certs = null;
        try {
            certs = SignedDataUtils.resolveCertChain(signCertChain);
            if (certs.size() < 2) {
                logger.error("RA服务器证书链大小错误");
                result.setError(ErrorEnum.CONVERT_CERT_ERROR);
                return result;
            }
        } catch (Exception e) {
            logger.error("RA服务解析证书链异常", e);
            result.setError(ErrorEnum.CONVERT_CERT_ERROR);
            return result;
        }

        //取第一张证书为用户证书,获得用户证书的pem值,即证书内容。
        X509Certificate signCert = certs.get(0);
        logger.info("RA的签名证书: " + signCert);


        logger.info("截取前信任链的大小为：" + certs.size());
        // 从RA的签名服务器证书链中，截取获取CA证书链
        certs.remove(0);
        logger.info("截取后信任链的大小为：" + certs.size());

        //获取p7b的base64值
        String p7bContent = null;
        try {
            p7bContent = Base64.toBase64String(signCertChain);
        } catch (Exception e) {
            logger.error("配置RA服务读取证书链base64值异常", e);
            result.setError(ErrorEnum.CERT_P7B_INFO_READ_ERROR);
            return result;
        }

        //判断是否为当前CA签发
        try {
            //判断DN是否正确
            String signDN = signCert.getIssuerDN().getName();
            String encDN = encCert.getIssuerDN().getName();
            String caDN = caRootCert.getSubjectDN().getName();
            BigInteger serialNumber = caRootCert.getSerialNumber();
            logger.info("signDN: " + signDN + "  encDN: " + encDN + "  caDN: " + caDN);
            if (signDN.equals(caDN) ? (encDN.equals(caDN) ? false : true) : true) {
                logger.info("DN不一致：非当前CA签发的服务器证书");
                result.setError(ErrorEnum.CERT_P7B_VERIFY_ERROR);
                return result;
            }

            PublicKey publicKey = caRootCert.getPublicKey();
            //签名证书
            boolean signVerify = HsmUtils.verifyCertByBC(signCert.getSigAlgName(), publicKey, signCert.getSignature(), signCert.getTBSCertificate());
            if (!signVerify) {
                logger.info("签名证书验签：非当前CA签发的服务器证书");
                result.setError(ErrorEnum.CERT_P7B_VERIFY_ERROR);
                return result;
            }
            //加密证书
            boolean encVerify = HsmUtils.verifyCertByBC(encCert.getSigAlgName(), publicKey, encCert.getSignature(), encCert.getTBSCertificate());
            if (!encVerify) {
                logger.info("加密证书验签：非当前CA签发的服务器证书");
                result.setError(ErrorEnum.GET_RA_SERVER_ENC_CERT_ERROR);
                return result;
            }
        } catch (Exception e) {
            logger.info("密码机异常", e);
            result.setError(ErrorEnum.CERT_P7B_VERIFY_ERROR);
            return result;
        }

        String signCertPem = null;
        String encCertPem = null;
        try {
            signCertPem = CertUtils.writeObject(signCert);
            encCertPem = CertUtils.writeObject(encCert);
        } catch (Exception e) {
            logger.error("读取RA服务证书pem值异常", e);
            result.setError(ErrorEnum.READ_RA_SERVER_CERT_PEM_ERROR);
            return result;
        }
        try {
            // 验证导入证书是否是最后一次申请时产生的公钥对应的证书
            byte[] signCertPubKey = signCert.getPublicKey().getEncoded();
            logger.debug("1:" + Base64.toBase64String(signCertPubKey));
            String signP10 = FileUtils.readByBytes(PathConstants.SOFT_ALG_FOLDER_PATH_TEMP + "CertReq.p10");
            PKCS10CertificationRequest pkcs10CertificationRequest = GMSSLP10Utils.decodeP10(signP10);
            byte[] p10PubKey = pkcs10CertificationRequest.getSubjectPublicKeyInfo().getEncoded();
            logger.debug("2:" + Base64.toBase64String(p10PubKey));

            byte[] encPubKey = FileUtils.readByBinary(PathConstants.SOFT_ALG_FOLDER_PATH_TEMP + "EncPubKey.pem");
            SubjectPublicKeyInfo subjectPublicKeyInfo = SubjectPublicKeyInfo.getInstance(Base64.decode(encPubKey));
            logger.debug("3:" + Base64.toBase64String(subjectPublicKeyInfo.getEncoded()));
            byte[] encCertPubKey = encCert.getPublicKey().getEncoded();
            logger.debug("4:" + Base64.toBase64String(encCertPubKey));

            if (!Arrays.equals(signCertPubKey, p10PubKey) || !Arrays.equals(encCertPubKey, subjectPublicKeyInfo.getEncoded())) {
                logger.info("证书中的公钥和申请时的公钥不一致");
                result.setError(ErrorEnum.CERT_PUB_KEY_NOT_SAME_REQ_PUB_KEY);
                return result;
            }
        } catch (Exception e) {
            logger.error("获取公钥异常", e);
            result.setError(ErrorEnum.GET_PUB_KEY_INFO_EXCEPTION);
            return result;
        }

        RaCertDO raSignCertDO = new RaCertDO();
        RaCertDO raEncCertDO = new RaCertDO();
        long pairCertIndex = System.nanoTime();
        raSignCertDO.setPairCertIndex(pairCertIndex);
        raSignCertDO.setCaCertId(newCaCertInfo.getId());
        raSignCertDO.setCertDn(CertUtils.getSubjectByX509Cert(signCert));
        raSignCertDO.setCertSn(signCert.getSerialNumber().toString(16).toLowerCase());
        raSignCertDO.setPublicKeyAlg(signCert.getPublicKey().getAlgorithm());
        raSignCertDO.setCertType(Constants.CERT_TYPE_SIGN_2);
        try {
            int publicKeyLength = CertUtils.getPublicKeyLength(signCert);
            raSignCertDO.setPrivateKeyLength(publicKeyLength);
        } catch (Exception e) {
            logger.error("获取证书公钥长度异常", e);
            result.setError(ErrorEnum.GET_CERT_PUBLIC_KEY_LENGTH_EXCEPTION);
            return result;
        }
        raSignCertDO.setSignAlg(signCert.getSigAlgName());
        raSignCertDO.setCertInfo(signCertPem);
        raSignCertDO.setCertP7b(p7bContent);
        raSignCertDO.setGmtCreate(new Timestamp(System.currentTimeMillis()));
        raSignCertDO.setGmtUpdate(new Timestamp(System.currentTimeMillis()));
        BeanUtils.copyProperties(raSignCertDO, raEncCertDO);
        raEncCertDO.setCertP7b(null);
        raEncCertDO.setCertDn(CertUtils.getSubjectByX509Cert(encCert));
        raEncCertDO.setCertSn(encCert.getSerialNumber().toString(16).toLowerCase());
        raEncCertDO.setCertType(Constants.CERT_TYPE_ENC_3);
        raEncCertDO.setSignAlg(encCert.getSigAlgName());
        raEncCertDO.setCertInfo(encCertPem);
        try {
            FileUtils.saveFile(signCertPem, PathConstants.RA_SERVICE_CERT_FILE_PATH);

            // 当导入RA证书正常后，将temp文件夹中的临时签名和加密私钥转移到正式的目录下
 //           ScriptUtils.executeScript("/bin/cp " + PathConstants.SOFT_ALG_FOLDER_PATH_TEMP + "SignPriKey.p12" + " " + PathConstants.SOFT_ALG_FOLDER_PATH + "SignPriKey.p12");
//            String signPriKey = FileUtils.readByBytes(PathConstants.SOFT_ALG_FOLDER_PATH_TEMP + "SignPriKey.jks");
//            FileUtils.saveFile(signPriKey, PathConstants.SOFT_ALG_FOLDER_PATH + "SignPriKey.jks");
 //           ScriptUtils.executeScript("/bin/cp " + PathConstants.SOFT_ALG_FOLDER_PATH_TEMP + "EncPriKey.p12" + " " + PathConstants.SOFT_ALG_FOLDER_PATH + "EncPriKey.p12");
//            String encPriKey = FileUtils.readByBytes(PathConstants.SOFT_ALG_FOLDER_PATH_TEMP + "EncPriKey.jks");
//            FileUtils.saveFile(encPriKey, PathConstants.SOFT_ALG_FOLDER_PATH + "EncPriKey.jks");
            /**
             * DO 20200820 放入RA的服务器证书 以便取公钥
             */
            //这个时候缓存或者路径下还是旧的服务器证书
            boolean isEncPwd = true;
            if(!initType) {
                isEncPwd = false;
            }
            // 签名私钥 p12存储
            KeyStoreUtils.saveServerPrivateKey(true, PathConstants.SOFT_ALG_FOLDER_PATH, signPriKey, new X509Certificate[]{signCert}, isEncPwd);
            // 加密私钥 p12存储
            KeyStoreUtils.saveServerPrivateKey(false, PathConstants.SOFT_ALG_FOLDER_PATH, encPriKey, new X509Certificate[]{encCert}, isEncPwd);
        } catch (Exception e) {
            logger.info("证书和对应私钥保存在指定路径中错误", e);
            result.setError(ErrorEnum.SAVE_RA_CERT_IS_ERROR);
            return result;
        }

        try {
            raCertDao.insertRaCertInfo(raSignCertDO);
            raCertDao.insertRaCertInfo(raEncCertDO);
        } catch (Exception e) {
            logger.error("RA服务器证书存ra_cert异常", e);
            result.setError(ErrorEnum.INSERT_RA_CERT_INFO_ERROR);
            return result;
        }
        //启用https
        try {
            if (CommonVariable.getKeyAlgName().equalsIgnoreCase(Constants.KEY_ALG_NAME_SM2)) {
                GMSSLTomcatUtils.openHttpsPortByBC(certs, signCert, encCert, signPriKey, encPriKey, tomcatPath, Integer.parseInt(httpsPort));
            } else if (CommonVariable.getKeyAlgName().equalsIgnoreCase(Constants.KEY_ALG_NAME_RSA)) {
                GMSSLTomcatUtils.openHttpsPortByJKSWithRSA(certs, signCert, encCert, signPriKey, encPriKey, tomcatPath, Integer.parseInt(httpsPort));
            } else {
                GMSSLTomcatUtils.openHttpsPortByJKSWithNist(certs, signCert, encCert, signPriKey, encPriKey, tomcatPath, Integer.parseInt(httpsPort));
            }

        } catch (Exception e) {
            logger.error("启用https异常", e);
            result.setError(ErrorEnum.START_TOMCAT_HTTPS);
            return result;
        }

        RaServerConfig raServerConfig = null;
        try {//获取配置文件
            Config config = systemService.getConfigFile(Constants.CONFIG_JSON_FILE_NAME);
            //修改初始化步骤
            if (initType) {
                config.setInitStep(Constants.RA_SERVER_PAGE);
            }else {
                if(Constants.DATA_IS_ENCRYPTION_1.equals(CommonVariable.getIsEncrypt()) && Constants.HSM_SERVER_XDJA_1 != CommonVariable.getIsHsm()) {
                    //DO 如果是配了信大捷安密码机的软通道 不需要重新加密
                    //DO 旧的服务器证书私钥解密 新的证书公钥加密
                    //byte[] key = Base64.decode(config.getEncryptKey());
                    //使用私钥解密对称密钥
                    //key = HsmUtils.privateKeyDecryptDate(oldRaEncPrivateKey, key);
                    //使用公钥加密 缓存内的明文对称密钥
                    byte[] key = HsmUtils.publicKeyEncryptDate(encCert.getPublicKey(), Base64.decode(CommonVariable.getEncryptKey()));
                    config.setEncryptKey(Base64.toBase64String(key));
                }
            }
            raServerConfig = config.getRaServerConfig();
            if (null == raServerConfig) {
                raServerConfig = new RaServerConfig();
            }
            raServerConfig.setSignCertName(signCertName.substring(signCertName.lastIndexOf("\\") + 1));
            raServerConfig.setEncCertName(encCertName.substring(encCertName.lastIndexOf("\\") + 1));
            config.setRaServerConfig(raServerConfig);
            config.setSigAlgName(signCert.getSigAlgName());

            systemService.updateConfigFile(config);
        } catch (Exception e) {
            logger.info("修改初始化配置文件失败", e);
            result.setError(ErrorEnum.CONFIG_JSON_FILE_OPERATION_ERROR);
            return result;

        }
        // 缓存RA配置信息
        CommonVariable.setRaSingSn(signCert.getSerialNumber().toString(16).toLowerCase());
        CommonVariable.setRaSignPriKey(signPriKey.getEncoded());
        CommonVariable.setRaEncPriKey(encPriKey);
        CommonVariable.setSigAlgName(signCert.getSigAlgName());
        CommonVariable.setRaEncPublicKey(encCert.getPublicKey());
        return result;
    }


    @Override
    public Result isUseHsmInfo() {

        Result result = new Result();

        String keyAlg = CommonVariable.getKeyAlgName();
        Integer isHsm = CommonVariable.getIsHsm();
        logger.info("获取当前【系统密钥算法】类型：" + keyAlg);
        if (Constants.HSM_SERVER_SWXA_2 == isHsm && Constants.KEY_ALG_NAME_NIST.equalsIgnoreCase(keyAlg)) {
            logger.error("三未信安密码机不支持NIST256算法");
            result.setError(ErrorEnum.SWXA_HSM_NOT_SUPPORT_NIST);
            return result;
        }
        HsmResp hsmResp = new HsmResp();
        hsmResp.setHsmType(isHsm);

        if (Constants.HSM_SERVER_XDJA_1 == isHsm && Constants.KEY_ALG_NAME_SM2.equalsIgnoreCase(keyAlg)) {
            /**
             * 跳转到配秘钥索引 私钥访问控制码页面
             */
            hsmResp.setJump(true);
        }

        if (!Constants.KEY_ALG_NAME_RSA.equalsIgnoreCase(keyAlg)) {
            hsmResp.setKeyLength(256);
        } else {
            hsmResp.setKeyLength(2048);
        }
        result.setInfo(hsmResp);
        return result;

    }

    @Override
    public Result genRaServerCertP10(String certDn, int keyLength, String keyAlg, boolean isInit) throws Exception {
        Result result = new Result();
        //签名公私钥对
        KeyPair signKeyPair = KeyUtils.genKeyPair(keyAlg, keyLength);
        PublicKey signPubKey = signKeyPair.getPublic();
        PrivateKey signPriKey = signKeyPair.getPrivate();
        //加密公私钥对
        KeyPair encKeyPair = KeyUtils.genKeyPair(keyAlg, keyLength);
        PublicKey encPubKey = encKeyPair.getPublic();
        PrivateKey encPriKey = encKeyPair.getPrivate();

        // 校验DN正确性
        Result verifyRaServerCertDn = this.verifyRaServerCertDn(certDn);
        if (!verifyRaServerCertDn.isSuccess()) {
            result.setError(verifyRaServerCertDn.getError());
            return result;
        }
        X500Name x500Name = DnUtil.getRFC4519X500Name(certDn);
        PKCS10CertificationRequest p10 = null;
        if (Constants.KEY_ALG_NAME_SM2.equals(keyAlg)) {
            p10 = P10Utils.createP10ByAlgType(x500Name, signPubKey, signPriKey, Constants.SIGN_ALG_NAME_SM3_WHIT_SM2);
        } else if (Constants.KEY_ALG_NAME_RSA.equals(keyAlg)) {
            p10 = P10Utils.createP10ByAlgType(x500Name, signPubKey, signPriKey, Constants.SIGN_ALG_NAME_SHA256_WHIT_RSA);
        } else {
            //do ECC-NIEST-P256算法算法 256
            p10 = P10Utils.createP10ByAlgType(x500Name, signPubKey, signPriKey, Constants.SIGN_ALG_NAME_SHA256_WHIT_ECDSA);
        }
        // 签名公钥保存为P10形式
        String p10Name = PathConstants.SOFT_ALG_FOLDER_PATH_TEMP + "CertReq.p10";
        FileUtils.saveFile(Base64.toBase64String(p10.getEncoded()), p10Name);
        boolean isEncPwd = false;
        if(isInit){
            isEncPwd = true;
        }
        // 签名私钥 jks存储
        KeyStoreUtils.saveServerPrivateKey(true, PathConstants.SOFT_ALG_FOLDER_PATH_TEMP, signPriKey, new X509Certificate[]{CommonVariable.getCaServiceCert()},isEncPwd);
//        FileUtils.saveFile(Base64.toBase64String(signPriKey.getEncoded()),signPriKeyName);

        // 加密公钥 带oid的pem文件存储
        String encPubKeyName = PathConstants.SOFT_ALG_FOLDER_PATH_TEMP + "EncPubKey.pem";
        FileUtils.saveFile(Base64.toBase64String(encPubKey.getEncoded()), encPubKeyName);

        // 加密私钥 jks存储
        KeyStoreUtils.saveServerPrivateKey(false, PathConstants.SOFT_ALG_FOLDER_PATH_TEMP, encPriKey, new X509Certificate[]{CommonVariable.getCaServiceCert()}, isEncPwd);
//        FileUtils.saveFile(Base64.toBase64String(encPriKey.getEncoded()),encPriKeyName);


        // 通过P10申请书的形式
        List<Map<String, Object>> list = new ArrayList<>();

        Map<String, Object> signMap = new HashMap<>();
        signMap.put("name", "RACertReq");
        signMap.put("suffix", "p10");
        signMap.put("buffer", Base64.encode(p10.getEncoded()));
        list.add(signMap);

        Map<String, Object> encMap = new HashMap<>();
        encMap.put("name", "RAEncPubKey");
        encMap.put("suffix", "pem");
        encMap.put("buffer", Base64.encode(encPubKey.getEncoded()));
        list.add(encMap);

        byte[] certReq = ZipUtils.generateZipByte(list);
        String fileName = "RACertReq_" + System.currentTimeMillis() + ".zip";
        FileUtils.saveFile(certReq, PathConstants.SOFT_ALG_FOLDER_PATH_TEMP + fileName);
        result.setInfo(fileName);
        return result;
    }


    @Override
    public Result downloadRaServerCertReq(String zipFileName) {
        Result result = new Result();
        byte[] certReq = FileUtils.readByBinary(PathConstants.SOFT_ALG_FOLDER_PATH_TEMP + zipFileName);
        if (certReq == null) {
            logger.info("获取服务器证书申请文件夹为空");
            result.setError(ErrorEnum.GET_RA_CERT_REQ_FILE_IS_EMPTY);
            return result;
        }

        result.setInfo(certReq);
        return result;
    }

    /**
     * 获取密钥对
     *
     * @param keyLength
     * @param alg
     * @return
     * @throws Exception
     */
    public static KeyPair getKeyPair(int keyLength, String alg) throws Exception {
        /** RSA算法要求有一个可信任的随机数源 */
        SecureRandom secureRandom = new SecureRandom();

        /** 为RSA算法创建一个KeyPairGenerator对象 */
        KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance(alg);

        /** 利用上面的随机数据源初始化这个KeyPairGenerator对象 */
        keyPairGenerator.initialize(keyLength, secureRandom);
        //keyPairGenerator.initialize(KEYSIZE);

        /** 生成密匙对 */
        KeyPair keyPair = keyPairGenerator.generateKeyPair();

        return keyPair;
    }


    /**
     * 校验RA服务证书DN
     *
     * @param certDn
     * @return
     */
    private Result verifyRaServerCertDn(String certDn) {
        //  证书DN的校验规则
        Result result = new Result();
        X500Name x500Name = null;
        try {
            x500Name = DnUtil.getRFC4519X500Name(certDn);
            String certDnFormat = x500Name.toString();
            logger.info("格式化之后的RA服务器证书DN：" + certDnFormat);

            // 检验DN关键字系统是否支持
            // DN的每个关键字的值不能超过64字节
            //检查是否包含中文逗号
            Result checkResult = CertDnVerifyUtils.checkDn(x500Name);
            if (!checkResult.isSuccess()) {
                result.setError(checkResult.getError());
                return result;
            }
        } catch (Exception e) {
            logger.info("certDn不正确{}", e.getMessage());
            result.setError(ErrorEnum.CERT_APPLY_DN_IS_ERROR);
            return result;
        }

        // DN的每个关键字的值不能超过64字节
       /* checkResult = CertDnVerifyUtils.check64(certDnFormat);
        if (!checkResult.isSuccess()) {
            logger.info("证书的DN关键字的值校验有误" + certDnFormat);
            result.setError(checkResult.getError());
            return result;
        } */
        // 校验DN的特殊符号  等号和逗号
       /* checkResult = CertDnVerifyUtils.checkCertDnSymbol(x500Name);
        if (!checkResult.isSuccess()) {
            logger.info("证书的DN的特殊符号校验有误");
            result.setError(checkResult.getError());
            return result;
        }

        // 检验DN的空格 无意义
        checkResult = CertDnVerifyUtils.checkBlankSpace(certDnFormat);
        if (!checkResult.isSuccess()) {
            logger.info("证书的DN关键字与值中有空格" + certDnFormat);
            result.setError(checkResult.getError());
            return result;
        }
         */

        return result;
    }

    /**
     * 获取系统RA服务器签名证书DN
     *
     * @return
     */
    @Override
    public String getRAServiceDnName() {
        RaCertDO newRaCertInfo = null;
        try {
            newRaCertInfo = raCertDao.getNewRaCertInfo();
        } catch (Exception e) {
            logger.error("获取RA服务器证书异常{}", e);
            return null;
        }

        if (newRaCertInfo == null) {
            return null;
        }
        String certInfo = newRaCertInfo.getCertInfo();
        X509Certificate certFromStr = CertUtils.getCertFromStr(certInfo);
        String raServiceDnName = CertUtils.getSubjectByX509Cert(certFromStr);
        return raServiceDnName;
    }
}