package com.xdja.pki.ra.service.manager.tbox;

import com.xdja.pki.ra.core.common.Result;
import com.xdja.pki.ra.core.commonenum.ErrorEnum;
import com.xdja.pki.ra.core.constant.Constants;
import com.xdja.pki.ra.core.util.file.HexUtils;
import com.xdja.pki.ra.core.util.json.JsonUtils;
import com.xdja.pki.ra.core.util.time.DateUtils;
import com.xdja.pki.ra.manager.dao.*;
import com.xdja.pki.ra.manager.dao.model.BaseUserDO;
import com.xdja.pki.ra.manager.dao.model.CertTempDO;
import com.xdja.pki.ra.manager.dao.model.DeviceKeyDO;
import com.xdja.pki.ra.manager.dao.model.UserCertDO;
import com.xdja.pki.ra.manager.dto.DeviceUserDTO;
import com.xdja.pki.ra.manager.dto.IssueApplyDTO;
import com.xdja.pki.ra.manager.dto.UpdateApplyDTO;
import com.xdja.pki.ra.service.manager.baseuser.bean.EncryptUserInfo;
import com.xdja.pki.ra.service.manager.certapply.CertApplyService;
import com.xdja.pki.ra.service.manager.certapply.IssueApplyService;
import com.xdja.pki.ra.service.manager.certapply.UpdateApplyService;
import com.xdja.pki.ra.service.manager.deviceuser.DeviceUserService;
import com.xdja.pki.ra.service.manager.deviceuser.bean.DeviceUserInfo;
import com.xdja.pki.ra.service.manager.deviceuser.bean.UserInfo;
import com.xdja.pki.ra.service.manager.tbox.bean.SharedKeyInfo;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.security.SecureRandom;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;

/**
 * Tbox设备服务实现类
 *
 * @author syg
 */
@Service
public class TboxDeviceServiceImpl implements TboxDeviceService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private static final String LOCK_ISSUER_APPLY_NO="issuerApplyNo";
    private static final String LOCK_UPDER_APPLY_NO="updateApplyNo";

    @Autowired
    DeviceUserService deviceUserService;

    @Autowired
    DeviceUserDao deviceUserDao;

    @Autowired
    DeviceKeyDao deviceKeyDao;

    @Autowired
    BaseUserDao baseUserDao;

    @Autowired
    IssueApplyService issueApplyService;

    @Autowired
    UpdateApplyService updateApplyService;

    @Autowired
    CertTempDao certTempDao;

    @Autowired
    private Environment env;

    @Autowired
    CaCertDao caCertDao;

    @Autowired
    UserCertDao userCertDao;

    @Autowired
    CertApplyService certApplyService;

    /**
     * 合并事务添加 @Transactional 20200925 DO WLY
     * @param deviceNo
     * @param deviceType
     * @param deviceName
     * @param deviceDesc
     * @return
     */
    @Transactional
    @Override
    public Result registerTboxDeviceUser(String deviceNo, int deviceType, String deviceName, String deviceDesc) {
        Result result = new Result();
        DeviceUserInfo deviceUserInfo = new DeviceUserInfo();
        deviceUserInfo.setUserName(deviceName);
        deviceUserInfo.setLicenseType(deviceType);
        deviceUserInfo.setDeviceDesc(deviceDesc);
        deviceUserInfo.setLicenseNumber(deviceNo);

       // UserInfo contactUserInfo = new UserInfo("v2x", 1, 0, 4, "123456", "17637123456", "xxx@xdja.com", "郑州市金水区杨金路139号智慧公园天利H座");
        UserInfo contactUserInfo = new UserInfo();
        // 已注册校验Tbox标识和用户类型
        String licenseNumberEncrypt;
        try {
            licenseNumberEncrypt = EncryptUserInfo.getEncryptString(deviceUserInfo.getLicenseNumber());
        } catch (Exception e) {
            result.setError(ErrorEnum.DECRYPT_ENCRYPT_INFO_ERROR);
            return result;
        }
        List<DeviceUserDTO> deviceUserDTOList = deviceUserDao.queryDeviceUserByLicenseType(licenseNumberEncrypt, deviceType);
        // 如果该设备已经注册过，则不再注册该设备
        if (!CollectionUtils.isEmpty(deviceUserDTOList)) {
            logger.info("该设备已注册 ============ deviceNo:" + deviceNo + " deviceType:" + deviceType + " size:" + deviceUserDTOList.size());
            DeviceUserDTO deviceUserDTO = deviceUserDTOList.get(0);
            int status = deviceUserDTO.getStatus();

            // 此时如果设备状态为已停用，则返回错误码，告知设备
            if (status == Constants.USER_STATUS_STOP_1) {
                result.setError(ErrorEnum.TBOX_DEVICE_STATUS_IS_STOP);
                return result;
            }
        } else {
            //T-BOX用户注册加标识
            //DO 0403 4、RA侧接口，T-BOX通过线上接口注册时，将T-BOX标识为“V2X”，同时T-BOX只有权限查询和操作自己设备用户证书的权限；
            deviceUserInfo.setSystemFlag(Constants.SYSTEM_FLAG_V2X);
            Result registerResult = deviceUserService.registerDeviceUser(deviceUserInfo, contactUserInfo,true);
            if (!registerResult.isSuccess()) {
                logger.info("设备注册失败：" + JsonUtils.object2Json(registerResult));
                result.setError(registerResult.getError());
                return result;
            }
        }

        // 如果设备状态是正常的，则生成一个共享密钥返回给设备
        byte[] sharedKey = new byte[16];
        SecureRandom secureRandom = new SecureRandom();
        secureRandom.nextBytes(sharedKey);

        String sharedKeyId = String.valueOf(System.currentTimeMillis());
        String sharedKeyValue = HexUtils.bytesToHexFun1(sharedKey);
        deviceKeyDao.addDeviceSharedKey(new DeviceKeyDO(deviceNo, sharedKeyId, sharedKeyValue));

        result.setInfo(new SharedKeyInfo(sharedKeyId, sharedKeyValue));
        return result;
    }

    @Override
    public Result getTboxDeviceInfo(String deviceNo, String keyId) {


        return null;
    }

//    @Transactional
    @Override
    public Result insertIssueCertApply(String deviceNo, String userCertDn) {
        Result result = new Result();
        // 判断注册设备状态
        BaseUserDO baseUser;
        try {
            String encryptDeviceNo = EncryptUserInfo.getEncryptString(deviceNo);
            baseUser = baseUserDao.queryUserByUserType(encryptDeviceNo, Constants.USER_TYPE_DEVICE_3);
        } catch (Exception e) {
            result.setError(ErrorEnum.ONLINE_DEVICE_INFO_NOT_EXIST);
            return result;
        }
        if (baseUser == null) {
            result.setError(ErrorEnum.ONLINE_DEVICE_INFO_NOT_EXIST);
            return result;
        }

        Integer deviceStatus = baseUser.getStatus();
        if (Constants.USER_STATUS_STOP_1 == deviceStatus) {
            result.setError(ErrorEnum.TBOX_DEVICE_STATUS_IS_STOP);
            return result;
        }
        // 获取Tbox的默认模板信息  判断TBox模板状态
        String tboxTemplateNo = null;
        try {
            tboxTemplateNo = env.getProperty("tbox.cert.template.no");
        } catch (Exception e) {
            result.setError(ErrorEnum.GET_TBOX_TEMPLATE_EXCEPTION);
            return result;
        }
        if (StringUtils.isBlank(tboxTemplateNo)) {
            result.setError(ErrorEnum.TBOX_TEMPLATE_NO_NOT_EXIST);
            return result;
        }
        logger.info(" =========== tbox证书模板编号:" + tboxTemplateNo);

        // 判断tbox模板状态
        CertTempDO certTempDO = certTempDao.getCertTempInfoByTempNo(tboxTemplateNo);
        if (certTempDO == null) {
            result.setError(ErrorEnum.TBOX_TEMPLATE_NO_NOT_EXIST);
            return result;
        }
        int tempStatus = certTempDO.getTempStatus();
        int tempBound = certTempDO.getTempBound();
        if (Constants.TEMP_STATUS_STOP_3 == tempStatus || Constants.TEMP_UNBOUND_STATUS_2 == tempBound) {
            result.setError(ErrorEnum.TBOX_TEMPLATE_IS_STOP_OR_UNBOUND);
            return result;
        }

        // 自动发起Tbox的证书签发申请
        IssueApplyDTO issueApplyDTO = new IssueApplyDTO();
        issueApplyDTO.setUserId(baseUser.getId());
        issueApplyDTO.setTempNo(tboxTemplateNo);
        issueApplyDTO.setCertDn(userCertDn);  // 自定义证书DN
        issueApplyDTO.setSignAlg(certTempDO.getSignAlg());
        issueApplyDTO.setPrivateKeyLength(certTempDO.getPrivateKeyLength());
        issueApplyDTO.setCertValidity(certTempDO.getMaxValidity());
        issueApplyDTO.setApplyReason("Tbox设备在线发起设备注册和签发证书申请");
        issueApplyDTO.setTempId(certTempDO.getId());
        issueApplyDTO.setCertPatterm(certTempDO.getCertPatterm());
        //申请编号
//        String applyNo = DateUtils.getCurrDate(DateUtils.FORMAT_FIVE) + (int) (Math.random() * 900000) + 100000;
        String applyNo;
        synchronized (LOCK_ISSUER_APPLY_NO){
            applyNo  = String.valueOf(System.nanoTime())+ (ThreadLocalRandom.current().nextLong(0,9999999) + 90000000);
        }
        issueApplyDTO.setApplyNo(applyNo);
        //DO 签发申请
        Result issueApplyResult = issueApplyService.insertCertIssueApply(issueApplyDTO, certTempDO);
        if (!issueApplyResult.isSuccess()) {
            result.setError(issueApplyResult.getError());
            return result;
        }
        issueApplyDTO.setApplyId((Long) issueApplyResult.getInfo());
        //DO  操作记录
        certApplyService.insertCertApplyRecord(Constants.CERT_APPLY_TYPE_ISSUE_1, Constants.OPERATE_TYPE_ENTRY_1, applyNo, null, Constants.CERT_APPLY_STATUS_NOT_CHECK_1, issueApplyDTO.getApplyReason(), Constants.CERT_APPLY_OPERATE_TYPE_SUBMIT_SUCCESS_1, false,true);
        // V2X添加自动审核记录
        certApplyService.insertCertApplyRecord(Constants.CERT_APPLY_TYPE_ISSUE_1, Constants.OPERATE_TYPE_CHECK_3, applyNo, null, Constants.CERT_APPLY_STATUS_NOT_ISSUE_3, "自动审核类模板-审核成功", Constants.CERT_APPLY_OPERATE_TYPE_CHECK_SUCCESS_4, true,true);

        result.setInfo(issueApplyDTO);

        return result;
    }

    @Transactional
    @Override
    public Result insertUpdateCertApply(String deviceNo, String userCertDn, String signSn, Integer certValidity, boolean isUpdateKey) {
        Result result = new Result();

        // 判断注册设备状态
        BaseUserDO baseUser;
        try {
            baseUser = baseUserDao.queryUserByUserType(EncryptUserInfo.getEncryptString(deviceNo), Constants.USER_TYPE_DEVICE_3);
        } catch (Exception e) {
            result.setError(ErrorEnum.ONLINE_DEVICE_INFO_NOT_EXIST);
            return result;
        }
        if (baseUser == null) {
            result.setError(ErrorEnum.ONLINE_DEVICE_INFO_NOT_EXIST);
            return result;
        }

        Integer deviceStatus = baseUser.getStatus();
        if (Constants.USER_STATUS_STOP_1 == deviceStatus) {
            result.setError(ErrorEnum.TBOX_DEVICE_STATUS_IS_STOP);
            return result;
        }

        //判断deviceNo是否属于T-BOX
        if (!Constants.SYSTEM_FLAG_V2X.equalsIgnoreCase(baseUser.getSystemFlag())) {
            result.setError(ErrorEnum.DEVICE_NO_IS_NOT_TBOX_SYSTEM);
            return result;
        }

        //判断deviceNo和signSn是否对应，是否属于同一用户
        UserCertDO userCertDO = userCertDao.getUserCertBaseInfo(signSn);
        if (userCertDO == null || (userCertDO.getUserId().longValue() != baseUser.getId().longValue())) {
            result.setError(ErrorEnum.ONLINE_DEVICE_NO_GET_CERT_ERROR);
            return result;
        }

        long time = userCertDO.getFailureTime().getTime() - System.currentTimeMillis();
        if (Constants.CERT_STATUS_NORMAL_1 != userCertDO.getCertStatus() || time < 0) {
            logger.info("待更新证书不是正常状态");
            result.setError(ErrorEnum.USER_CERT_IS_NOT_NORMAL_STATUS);
            return result;
        }

        // 使用签发时候使用的模板
        logger.info(" =========== 证书签发时使用模板编号:" + userCertDO.getTempNo());
        // 判断tbox模板状态
        CertTempDO certTempDO = certTempDao.getCertTempInfoByTempNo(userCertDO.getTempNo());
        if (certTempDO == null) {
            result.setError(ErrorEnum.TBOX_TEMPLATE_NO_NOT_EXIST);
            return result;
        }
        int tempStatus = certTempDO.getTempStatus();
        int tempBound = certTempDO.getTempBound();
        if (Constants.TEMP_STATUS_STOP_3 == tempStatus || Constants.TEMP_UNBOUND_STATUS_2 == tempBound) {
            result.setError(ErrorEnum.OLD_TEMPLATE_IS_STOP_OR_UNBOUND);
            return result;
        }

        // 自动发起Tbox的证书更新申请
        UpdateApplyDTO updateApplyDTO = new UpdateApplyDTO();
        updateApplyDTO.setUserId(baseUser.getId());
        updateApplyDTO.setTempNo(userCertDO.getTempNo());
        updateApplyDTO.setCertDn(userCertDO.getCertDn());  // 自定义证书DN
        updateApplyDTO.setSignAlg(certTempDO.getSignAlg());
        updateApplyDTO.setPrivateKeyLength(certTempDO.getPrivateKeyLength());
        updateApplyDTO.setCertValidity(certValidity == 0 || certValidity > certTempDO.getMaxValidity() ? certTempDO.getMaxValidity() : certValidity);//证书有效期为0或大于最大有效期则按系统计算的有效期为准
        updateApplyDTO.setApplyReason("Tbox设备在线发起更新证书申请");
        updateApplyDTO.setUpdateKey(isUpdateKey);
        updateApplyDTO.setSignSn(signSn);
        updateApplyDTO.setUpdateValidity(certValidity == 0 ? false : true);
        updateApplyDTO.setCertPatterm(certTempDO.getCertPatterm());
        updateApplyDTO.setTempId(certTempDO.getId());
        //申请编号
//        String applyNo = DateUtils.getCurrDate(DateUtils.FORMAT_FIVE) + (int) (Math.random() * 900000) + 100000;
        String applyNo;
        synchronized (LOCK_UPDER_APPLY_NO){
            applyNo = String.valueOf(System.nanoTime())+ (ThreadLocalRandom.current().nextLong(0,9999999) + 90000000);
        }
        updateApplyDTO.setApplyNo(applyNo);
        //DO 更新申请
        Result updateApplyResult = updateApplyService.insertCertUpdateApply(updateApplyDTO, userCertDO, certTempDO,false);
        if (!updateApplyResult.isSuccess()) {
            result.setError(updateApplyResult.getError());
            return result;
        }
        updateApplyDTO.setApplyId((Long) updateApplyResult.getInfo());
        //DO  操作记录
        certApplyService.insertCertApplyRecord(Constants.CERT_APPLY_TYPE_UPDATE_2, Constants.OPERATE_TYPE_ENTRY_1, applyNo, null, Constants.CERT_APPLY_STATUS_NOT_CHECK_1, updateApplyDTO.getApplyReason(), Constants.CERT_APPLY_OPERATE_TYPE_SUBMIT_SUCCESS_1, false, true);
        //V2X添加自动审核记录
        certApplyService.insertCertApplyRecord(Constants.CERT_APPLY_TYPE_UPDATE_2, Constants.OPERATE_TYPE_CHECK_3, applyNo, null, Constants.CERT_APPLY_STATUS_NOT_ISSUE_3, "自动审核类模板-审核成功", Constants.CERT_APPLY_OPERATE_TYPE_CHECK_SUCCESS_4, true, true);

        result.setInfo(updateApplyDTO);
        return result;
    }
}
