package com.xdja.pki.ra.service.manager.utils;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.security.NoSuchProviderException;
import java.security.Security;
import java.security.cert.CertificateException;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;
import javax.security.auth.x500.X500Principal;

import com.xdja.ca.asn1.NISTObjectIdentifiers;
import com.xdja.ca.asn1.RsaObjectIdentifiers;
import com.xdja.ca.utils.DnUtil;
import com.xdja.pki.gmssl.asn1.x509.SubjectInformationAccess;
import com.xdja.pki.ra.core.util.cert.CertUtils;
import com.xdja.pki.ra.core.util.cert.IdentifyCode;
import com.xdja.pki.ra.core.util.cert.SM2Extensions;
import com.xdja.pki.ra.core.util.time.DateUtils;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1GeneralizedTime;
import org.bouncycastle.asn1.ASN1Integer;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.ASN1Set;
import org.bouncycastle.asn1.DERBitString;
import org.bouncycastle.asn1.DERIA5String;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.DERPrintableString;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.DLSequence;
import org.bouncycastle.asn1.gm.GMObjectIdentifiers;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.style.BCStyle;
import org.bouncycastle.asn1.x500.style.RFC4519Style;
import org.bouncycastle.asn1.x509.AccessDescription;
import org.bouncycastle.asn1.x509.Attribute;
import org.bouncycastle.asn1.x509.AuthorityInformationAccess;
import org.bouncycastle.asn1.x509.AuthorityKeyIdentifier;
import org.bouncycastle.asn1.x509.BasicConstraints;
import org.bouncycastle.asn1.x509.CRLDistPoint;
import org.bouncycastle.asn1.x509.CertificatePolicies;
import org.bouncycastle.asn1.x509.DistributionPoint;
import org.bouncycastle.asn1.x509.DistributionPointName;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.Extensions;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.bouncycastle.asn1.x509.GeneralSubtree;
import org.bouncycastle.asn1.x509.NameConstraints;
import org.bouncycastle.asn1.x509.PolicyConstraints;
import org.bouncycastle.asn1.x509.PolicyInformation;
import org.bouncycastle.asn1.x509.PolicyMappings;
import org.bouncycastle.asn1.x509.PolicyQualifierId;
import org.bouncycastle.asn1.x509.PolicyQualifierInfo;
import org.bouncycastle.asn1.x509.PrivateKeyUsagePeriod;
import org.bouncycastle.asn1.x509.SubjectDirectoryAttributes;
import org.bouncycastle.asn1.x509.SubjectKeyIdentifier;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateHolder;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.Strings;
import org.bouncycastle.util.encoders.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Component;
/**
 * 证书详情解析工具类
 * @author zjr
 *
 */
@Component
public class CertContentInfoUtil {
	private Logger logger = LoggerFactory.getLogger(this.getClass());

	@Autowired
	private  Environment env;
	private static final String SIGN_ALGO_SM3WITHSM2 = "SM3WithSM2";
	private static final String SIGN_ALGO_SHA256WithECDSA = "sha256WithECDSA";
	private static final String SIGN_ALGO_SHA256WithRSA = "sha256WithRSA";
	private static final String SIGN_ALGO_SHA1WithRSA = "sha-1WithRSA";

	static {
		Security.addProvider(new BouncyCastleProvider());
	}

	public Map<String, Object> getCertContentInfo(X509Certificate certificate) throws IOException, CertificateException, NoSuchProviderException{

		Map<String, Object> certContentInfo = new HashMap<String, Object>();
		Map<String, Object> extendsInfo = new HashMap<String, Object>();

		SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		// 版本号
		int version = certificate.getVersion();
		certContentInfo.put("version", "V" + String.valueOf(version));

		// 序列号
		String serialNumber = certificate.getSerialNumber().toString(16);
		certContentInfo.put("sn", serialNumber);
		// 签名算法
		String sigAlgName = certificate.getSigAlgName();
		String sigAlgOID = certificate.getSigAlgOID();
		String sm3withsm2OID = GMObjectIdentifiers.sm2sign_with_sm3.getId();
		if(sigAlgOID.equals(sm3withsm2OID)) {
			sigAlgName = SIGN_ALGO_SM3WITHSM2;
		}
		if(sigAlgOID.equals(NISTObjectIdentifiers.nistSignAlgorithm.getId())){
			sigAlgName = SIGN_ALGO_SHA256WithECDSA;
		}
		if(sigAlgOID.equals(RsaObjectIdentifiers.sha256WithRSA.getId())){
			sigAlgName = SIGN_ALGO_SHA256WithRSA;
		}
		if(sigAlgOID.equals(RsaObjectIdentifiers.sha1WithRSA.getId())){
			sigAlgName = SIGN_ALGO_SHA1WithRSA;
		}
		certContentInfo.put("sigAlgName", sigAlgName);
		// 颁发者
		String issuerDN = CertUtils.getIssuerByX509Cert(certificate);
		certContentInfo.put("issuer", issuerDN);
		// 证书主体
		String subjectname = CertUtils.getSubjectByX509Cert(certificate);
		certContentInfo.put("subject", subjectname);
		// 生效时间
		String notBefore = simpleDateFormat.format(certificate.getNotBefore());
		certContentInfo.put("notbefore", notBefore);
		// 失效时间
		String notAfter = simpleDateFormat.format(certificate.getNotAfter());
		certContentInfo  .put("notafter", notAfter);
		// 主体公钥信息
		SubjectPublicKeyInfo subPubKeyInfo = SubjectPublicKeyInfo.getInstance(certificate.getPublicKey().getEncoded());
		certContentInfo.put("subjectPublicKeyInfo",  Strings.fromByteArray(Hex.encode(subPubKeyInfo.getEncoded())));
//		System.out.println(
//				"publicKey : " + Strings.fromByteArray(Hex.encode(subPubKeyInfo.getPublicKeyData().getBytes())));
//		certContentInfo.put("publicKey", Strings.fromByteArray(Hex.encode(subPubKeyInfo.getPublicKeyData().getBytes())));
//		String algorithm = certificate.getPublicKey().getAlgorithm();
//		System.out.println("algorithm : " + algorithm);
//		certContentInfo.put("algorithm", algorithm);
		// 颁发者唯一标识符
		boolean[] issuerUniqueID = certificate.getIssuerUniqueID();
		if(null != issuerUniqueID) {
			certContentInfo.put("issueUniqueId", booleanToBitString(issuerUniqueID).toString());
		}

		// 主体唯一标识符
		boolean[] subjectUniqueID = certificate.getSubjectUniqueID();
		if(null != subjectUniqueID) {
			certContentInfo.put("subjectUniqueId", booleanToBitString(subjectUniqueID).toString());
		}
		// 证书的扩展信息
		logger.info("<<===========证书的扩展信息==============>>");
		X509CertificateHolder certificateHoder = new JcaX509CertificateHolder(certificate);
		Extensions extensions = certificateHoder.getExtensions();

		//获取授权密钥标识符
		getAuthorityKeyIdentifier(extendsInfo,extensions);

		//获取使用者秘钥标识符
		getSubjectKeyIdentifier(extendsInfo,extensions);

		//获取密钥用途
		List<String> usages = getKeyUsage(certificate.getKeyUsage());
		if (null != usages) {
			logger.info("usages : " + usages.toString());
			extendsInfo.put("keyUsage", usages.toString());
		}
		//获取密钥扩展用途
		List<String> extendedKeyUsages = getExtendedKeyUsages(certificate.getExtendedKeyUsage());
		if (null != extendedKeyUsages) {
			logger.info("extendedKeyUsages :" + extendedKeyUsages.toString());
			extendsInfo.put("extendedKeyUsage", extendedKeyUsages.toString());
		}
		//获取私有秘钥使用周期
		getPrivateKeyUsagePeriod(extendsInfo,extensions);
		//获取证书策略
		getcertificatePolicies(extendsInfo,extensions);
		//获取证书策略映射
		getPolicyMapping(extendsInfo,extensions);
		//获取主体替换名称
		getSubjectAltName(extendsInfo,certificate);
		//获取主体目录属性
		getSubjectDirectoryAttributes(extendsInfo,extensions);
		//获取颁发者替换名称
		getIssuerAltName(extendsInfo,certificate);
		//获取基本限制
		getBasicConstraints(extendsInfo,extensions);
		//获取名称限制
		getNameConstraints(extendsInfo,extensions);
		//获取策略限制
		getPolicyConstraints(extendsInfo,extensions);
		//获取限制所有策略
		getInhibitAnyPolicy(extendsInfo,extensions);
		//获取CRL分发点
		getCRLDistPoint(extendsInfo,extensions);
		//获取最新CRL
		getFreshestCRL(extendsInfo,extensions);
		//获取机构信息访问
		getAuthorityInfoAccess(extendsInfo,extensions);
		//获取主体信息访问
		getSubjectInformationAccess(extendsInfo,extensions);
		//获取个人身份标识
		getIdentifyCode(extendsInfo,extensions);
		//获取个人社会保险号
		getInsuranceNumber(extendsInfo,extensions);
		//获取企业工商注册号
		getICRegistrationNumber(extendsInfo,extensions);
		//获取企业组织机构代码
		getOrganizationCode(extendsInfo,extensions);
		//获取企业税号
		getTaxationNumber(extendsInfo,extensions);

		certContentInfo.put("extendsInfo", extendsInfo);
		return certContentInfo;
	}

	/**
	 * 获取授权秘钥标识符
	 * @param extendsInfo
	 * @param extensions
	 */
	private void getAuthorityKeyIdentifier(Map<String, Object> extendsInfo,Extensions extensions) {
		AuthorityKeyIdentifier authorityKeyIdentifier = AuthorityKeyIdentifier.fromExtensions(extensions);
		if(null != authorityKeyIdentifier) {
			List<String> authorityKeyIdentifierList = new ArrayList<String>();
			String extendName = CertExtendNameFinder.getExtendName(Extension.authorityKeyIdentifier);
			if(null != authorityKeyIdentifier.getKeyIdentifier()) {
				authorityKeyIdentifierList.add( "KeyID=" + Strings.fromByteArray(Hex.encode(authorityKeyIdentifier.getKeyIdentifier())));
			}
			GeneralNames authorityCertIssuer = authorityKeyIdentifier.getAuthorityCertIssuer();
			if(null != authorityCertIssuer) {
				GeneralName[] names = authorityCertIssuer.getNames();
				StringBuffer stringbuffer = new StringBuffer();
				for(GeneralName gneralname : names) {
					Integer tag = gneralname.getTagNo();
					ASN1Encodable name = gneralname.getName();
					X500Name x500Name = DnUtil.getRFC4519X500Name(name);
					X500Principal x500Principal = null;
					try {
						x500Principal = new X500Principal(x500Name.getEncoded());
					} catch (IOException e) {
						logger.error("x500Principal转换异常，",e);
					}
					String generalName = GeneralNameFinder.getGeneralName(tag);
					stringbuffer.append(generalName + "=" + x500Principal.getName() + ",");
				}
				authorityKeyIdentifierList.add("Certificate Issuer:" + stringbuffer.deleteCharAt(stringbuffer.length()-1).toString());
			}
			if(null != authorityKeyIdentifier.getAuthorityCertSerialNumber()) {
				BigInteger authorityCertSerialNumber  = authorityKeyIdentifier.getAuthorityCertSerialNumber();
				authorityKeyIdentifierList.add("Certificate SerialNumber=" + authorityCertSerialNumber.toString(16));
			}
			logger.info("authorityKeyIdentifier: " + authorityKeyIdentifierList.toString());
			extendsInfo.put(extendName, authorityKeyIdentifierList.toString());
		}
	}

	/**
	 * 获取使用者秘钥标识符
	 * @param extendsInfo
	 * @param extensions
	 */
	private void getSubjectKeyIdentifier(Map<String, Object> extendsInfo, Extensions extensions) {
		SubjectKeyIdentifier subjectKeyIdentifier = SubjectKeyIdentifier.fromExtensions(extensions);
		if(null != subjectKeyIdentifier) {
			String extendName = CertExtendNameFinder.getExtendName(Extension.subjectKeyIdentifier);
			logger.info("subjectKeyIdentifier" + Strings.fromByteArray(Hex.encode(subjectKeyIdentifier.getKeyIdentifier())));
			extendsInfo.put(extendName, Strings.fromByteArray(Hex.encode(subjectKeyIdentifier.getKeyIdentifier())));
		}
	}

	/**
	 * 获取私钥使用周期
	 * @param extendsInfo
	 * @param extensions
	 * @throws ParseException
	 */
	private void getPrivateKeyUsagePeriod(Map<String, Object> extendsInfo, Extensions extensions){
		String extendName = CertExtendNameFinder.getExtendName(Extension.privateKeyUsagePeriod);
		ASN1Encodable extvalue = extensions.getExtensionParsedValue(Extension.privateKeyUsagePeriod);
		PrivateKeyUsagePeriod privateKeyUsagePeriod = null;
		if (extvalue != null) {
			privateKeyUsagePeriod = PrivateKeyUsagePeriod.getInstance(extvalue);

			SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy年MM月dd日 HH:mm:ss");
			try {
				String notBefore = simpleDateFormat.format(privateKeyUsagePeriod.getNotBefore().getDate());
				String notAfter = simpleDateFormat.format(privateKeyUsagePeriod.getNotAfter().getDate());
				String privkeyPeriod = "私钥使用周期从  " + notBefore + "到 " + notAfter;
				logger.info(privkeyPeriod);
				extendsInfo.put(extendName, privkeyPeriod);
			} catch (ParseException e) {
				logger.error("时间格式转换异常，",e);
			}
		}
	}

	/**
	 * 获取证书策略
	 * @param extendsInfo
	 * @param extensions
	 * @throws IOException
	 */
	private void  getcertificatePolicies(Map<String, Object> extendsInfo, Extensions extensions) throws IOException{
		CertificatePolicies certificatePolicies = CertificatePolicies.fromExtensions(extensions);
		if(null != certificatePolicies) {
			StringBuffer stbuf = new StringBuffer();

			String extendName = CertExtendNameFinder.getExtendName(Extension.certificatePolicies);
			PolicyInformation[] policyInformations = certificatePolicies.getPolicyInformation();

			for(PolicyInformation policyInformation : policyInformations) {
				stbuf.append("Policy Identifier=");

				ASN1ObjectIdentifier policyIdentifier = policyInformation.getPolicyIdentifier();
				stbuf.append(policyIdentifier.toString());

				ASN1Sequence policyQualifiers = policyInformation.getPolicyQualifiers();

				if(null != policyQualifiers) {
					stbuf.append("\r\n [");
					String prefix = ", ";
					for(int i=0;i<policyQualifiers.size();i++) {
						stbuf.append("Policy Qualifier Info:");
						ASN1Encodable policyQualifier = policyQualifiers.getObjectAt(i);
						PolicyQualifierInfo policyQualifierInfo = PolicyQualifierInfo.getInstance(policyQualifier);
						ASN1ObjectIdentifier policyQualifierId = policyQualifierInfo.getPolicyQualifierId();
						ASN1Encodable qualifier = policyQualifierInfo.getQualifier();

						if(null != policyQualifierId) {
							if(PolicyQualifierId.id_qt_cps.equals(policyQualifierId)) {
								stbuf.append("Policy Qualifier Id=CPS");
								if(null != qualifier) {
									stbuf.append(",Qualifier:");
									stbuf.append(qualifier.toString());
								}
							}else  if(PolicyQualifierId.id_qt_unotice.equals(policyQualifierId)) {
								stbuf.append("Policy Qualifier Id=UNOTICE");
								if(null != qualifier) {
									stbuf.append(",Qualifier:Notice Text=");
									stbuf.append(qualifier.toString());
								}
							}
						}
						stbuf.append(prefix);
					}
					stbuf.deleteCharAt(stbuf.length()-prefix.length());
					stbuf.append("]");
					stbuf.append(" \r\n");
				}
			}

			extendsInfo.put(extendName, stbuf.toString());
		}
	}

	/**
	 * 获取证书策略映射
	 * @param extendsInfo
	 * @param extensions
	 * @throws IOException
	 */
	private void getPolicyMapping(Map<String, Object> extendsInfo,Extensions extensions) throws IOException{
		ASN1Encodable extensionParsedValue = extensions.getExtensionParsedValue(Extension.policyMappings);
		PolicyMappings policyMappings = PolicyMappings.getInstance(extensionParsedValue);
		if(null != policyMappings) {
			ASN1Primitive policyMappingsPrimitive = policyMappings.toASN1Primitive();
			if(policyMappingsPrimitive instanceof ASN1Sequence) {
				ASN1Sequence asn1Sequence = (ASN1Sequence) policyMappingsPrimitive;
				if(null != asn1Sequence) {
					String extendName = CertExtendNameFinder.getExtendName(Extension.policyMappings);
					List<Map<String,String>> policyMappList = new ArrayList<Map<String,String>>();
					for(int i=0;i<asn1Sequence.size();i++) {
						ASN1Encodable asn1enable = asn1Sequence.getObjectAt(i);
						ASN1Primitive asn1Primitive = asn1enable.toASN1Primitive();
						if(asn1Primitive instanceof DLSequence) {
							Map<String,String> parm = new HashMap<String, String>();
							ASN1Sequence policyMappingsSequence = (ASN1Sequence) asn1Primitive;
							parm.put("Issuer Domain", policyMappingsSequence.getObjectAt(0).toASN1Primitive().toString());
							parm.put("Subject Domain", policyMappingsSequence.getObjectAt(1).toASN1Primitive().toString());
							policyMappList.add(parm);
						}
					}
					extendsInfo.put(extendName, policyMappList.toString());
				}
			}
		}
	}

	/**
	 * 获取主体替换名称
	 * @param extendsInfo
	 * @param extendsInfo
	 * @throws CertificateParsingException
	 */
	private void getSubjectAltName(Map<String, Object> extendsInfo, X509Certificate certificate) throws CertificateParsingException {
		Collection<List<?>> subjectAlternativeNames = certificate.getSubjectAlternativeNames();
		if(null != subjectAlternativeNames && !subjectAlternativeNames.isEmpty()) {
			String extendName = CertExtendNameFinder.getExtendName(Extension.subjectAlternativeName);
			List<String> subjectAltNameList = getGeneralNameList(subjectAlternativeNames);
			extendsInfo.put(extendName, subjectAltNameList.toString());
		}
	}

	/**
	 * 获取主体目录属性
	 * @param extendsInfo
	 * @param extensions
	 * @throws UnsupportedEncodingException
	 */
	private void getSubjectDirectoryAttributes(Map<String, Object> extendsInfo, Extensions extensions) throws UnsupportedEncodingException{
		ASN1Encodable extensionParsedValue = extensions.getExtensionParsedValue(Extension.subjectDirectoryAttributes);
		SubjectDirectoryAttributes subjectDirectoryAttributes = SubjectDirectoryAttributes.getInstance(extensionParsedValue);
		if(null != subjectDirectoryAttributes) {
			Vector attributes = subjectDirectoryAttributes.getAttributes();
			if(!attributes.isEmpty()) {
				String extendName = CertExtendNameFinder.getExtendName(Extension.subjectDirectoryAttributes);
				StringBuffer strbf = new StringBuffer();
				String prefix = ", ";
				strbf.append("[");
				for(int i=0; i<attributes.size(); i++) {
					Attribute attribute = (Attribute) attributes.get(i);
					ASN1Set set = attribute.getAttrValues();
					String attributeId = attribute.getAttrType().getId();
					if(attributeId.equals(BCStyle.DATE_OF_BIRTH.getId())) {
						ASN1GeneralizedTime time = ASN1GeneralizedTime.getInstance(set.getObjectAt(0));
						Date date = null;
						try {
							date = time.getDate();
						} catch (ParseException e) {
							logger.error("gmt时间格式转换异常，",e);
							continue;
						}
						String dateStr = DateUtils.dateToString(date,DateUtils.FORMAT_ONE);
						strbf.append("dateOfBirth="+dateStr + prefix);
					}else if(attributeId.equals(BCStyle.T.getId())) {
						ASN1Encodable aSN1Encodable = set.getObjectAt(0);
						ASN1OctetString octeString = null;
						String tile = null;
						if(aSN1Encodable instanceof DEROctetString) {
							octeString = DEROctetString.getInstance(aSN1Encodable);
							tile = new String(octeString.getOctets(), "UTF-8");
						}else if(aSN1Encodable instanceof DERPrintableString) {
							DERPrintableString dERPrintableString = DERPrintableString.getInstance(aSN1Encodable);
							tile = dERPrintableString.toString();
						}
						strbf.append("Title="+ tile + prefix);
					}
					else {
						ASN1Encodable aSN1Encodable = set.getObjectAt(0);
						ASN1OctetString octeString = null;
						String value = null;
						if(aSN1Encodable instanceof DEROctetString) {
							octeString = DEROctetString.getInstance(aSN1Encodable);
							value = new String(octeString.getOctets(), "UTF-8");
						}else if(aSN1Encodable instanceof DERPrintableString) {
							DERPrintableString dERPrintableString = DERPrintableString.getInstance(aSN1Encodable);
							value = dERPrintableString.toString();
						}
						String oidToDisplayName = BCStyle.INSTANCE.oidToDisplayName(attribute.getAttrType());
						strbf.append(oidToDisplayName + "=" + value + prefix);
					}
				}
				strbf.deleteCharAt(strbf.length()-prefix.length());
				strbf.append("]");
				extendsInfo.put(extendName, strbf.toString());
			}
		}
	}

	/**
	 * 获取颁发者替换名称
	 * @param extendsInfo
	 * @param certificate
	 * @throws CertificateParsingException
	 */
	private void getIssuerAltName(Map<String, Object> extendsInfo, X509Certificate certificate) throws CertificateParsingException {
		Collection<List<?>> issuerAlternativeNames = certificate.getIssuerAlternativeNames();
		if(null != issuerAlternativeNames && !issuerAlternativeNames.isEmpty()) {
			String extendName = CertExtendNameFinder.getExtendName(Extension.issuerAlternativeName);
			List<String> issuerAltNameList = getGeneralNameList(issuerAlternativeNames);
			extendsInfo.put(extendName, issuerAltNameList.toString());
		}
	}

	/**
	 * 获取基本限制
	 * @param extendsInfo
	 * @param extensions
	 */
	private void getBasicConstraints(Map<String, Object> extendsInfo, Extensions extensions) {
		BasicConstraints basicConstraints = BasicConstraints.fromExtensions(extensions);
		if(null != basicConstraints) {
			String extendName = CertExtendNameFinder.getExtendName(Extension.basicConstraints);
			boolean isCa = basicConstraints.isCA();
			StringBuffer strbuf = new StringBuffer();
			strbuf.append("Subject Type=" + (isCa ? "CA" : "End Entity") + ", ");
			strbuf.append("Path Length Constraint=" +
					(null != basicConstraints.getPathLenConstraint() ? basicConstraints.getPathLenConstraint().toString(16) : "None"));
//			System.out.println("basicConstraints: " + strbuf.toString());
			extendsInfo.put(extendName, strbuf.toString());
		}
	}

	/**
	 * 获取CRL分发点
	 * @param extendsInfo
	 * @param extensions
	 * @throws IOException
	 * @throws UnsupportedEncodingException
	 */
	private void getCRLDistPoint(Map<String, Object> extendsInfo, Extensions extensions) throws UnsupportedEncodingException {
//		ASN1Encodable extvalue = extensions.getExtensionParsedValue(Extension.cRLDistributionPoints);
//		CRLDistPoint cRLDistPoint = CRLDistPoint.getInstance(extvalue);
//		if(null != cRLDistPoint) {
//			DistributionPoint[] distributionPoints = cRLDistPoint.getDistributionPoints();
//			if(null != distributionPoints && distributionPoints.length > 0) {
//				String extendName = CertExtendNameFinder.getExtendName(Extension.cRLDistributionPoints);
//				List<String> CRLDistPointList = new ArrayList<String>();
//				for(DistributionPoint distributionPoint : distributionPoints) {
//
//					CRLDistPointList.add(new String(distributionPoint.getEncoded("DER"),"UTF-8"));
//				}
//				extendsInfo.put(extendName, CRLDistPointList.toString());
//			}
//		}

		ASN1Encodable extvalue = extensions.getExtensionParsedValue(Extension.cRLDistributionPoints);
		CRLDistPoint cRLDistPoint = CRLDistPoint.getInstance(extvalue);
		if(null != cRLDistPoint) {
			DistributionPoint[] distributionPoints = cRLDistPoint.getDistributionPoints();
			if(null != distributionPoints && distributionPoints.length > 0) {
				String extendName = CertExtendNameFinder.getExtendName(Extension.cRLDistributionPoints);
				List<String> CRLDistPointList = new ArrayList<String>();
				for(DistributionPoint distributionPoint : distributionPoints) {
					StringBuffer parmap = new StringBuffer();
					String CRLDistPointKey = "";
					String CRLDistPointValue = "";
					DistributionPointName distributionPointName = distributionPoint.getDistributionPoint();
					int type = distributionPointName.getType();
					ASN1Encodable distributionPointNameEncodable = distributionPointName.getName();
					if(distributionPointNameEncodable instanceof GeneralNames) {
						GeneralNames generalNames = (GeneralNames)distributionPointNameEncodable;
						GeneralName[] names = generalNames.getNames();
						for(GeneralName name : names) {

							String generalName = GeneralNameFinder.getGeneralName(name.getTagNo());
							DERIA5String dERIA5String =	(DERIA5String)name.getName();
							CRLDistPointValue = generalName + "=" + new String(dERIA5String.getOctets(), "UTF-8");
						}
					}

					if(type == 0) {
						CRLDistPointKey = "Full Name";
					}else if(type == 1) {
						CRLDistPointKey = "nameRelativeToCRLIssuer";
					}
					parmap.append(CRLDistPointKey).append(":").append(CRLDistPointValue);


					CRLDistPointList.add(parmap.toString());
				}
				extendsInfo.put(extendName, CRLDistPointList.toString());
			}
		}
	}

	/**
	 * 获取最新CRL
	 * @param extendsInfo
	 * @param extensions
	 * @throws UnsupportedEncodingException
	 */
	private void getFreshestCRL(Map<String, Object> extendsInfo, Extensions extensions) throws UnsupportedEncodingException{
//		ASN1Encodable extvalue = extensions.getExtensionParsedValue(Extension.freshestCRL);
//		CRLDistPoint cRLDistPoint = CRLDistPoint.getInstance(extvalue);
//		if(null != cRLDistPoint) {
//			DistributionPoint[] distributionPoints = cRLDistPoint.getDistributionPoints();
//			if(null != distributionPoints && distributionPoints.length > 0) {
//				String extendName = CertExtendNameFinder.getExtendName(Extension.freshestCRL);
//				List<String> CRLDistPointList = new ArrayList<String>();
//				for(DistributionPoint distributionPoint : distributionPoints) {
//					CRLDistPointList.add(distributionPoint.toString());
//				}
//				extendsInfo.put(extendName, CRLDistPointList.toString());
//			}
//		}

		ASN1Encodable extvalue = extensions.getExtensionParsedValue(Extension.freshestCRL);
		CRLDistPoint cRLDistPoint = CRLDistPoint.getInstance(extvalue);
		if(null != cRLDistPoint) {
			DistributionPoint[] distributionPoints = cRLDistPoint.getDistributionPoints();
			if(null != distributionPoints && distributionPoints.length > 0) {
				String extendName = CertExtendNameFinder.getExtendName(Extension.freshestCRL);
				List<String> CRLDistPointList = new ArrayList<String>();
				for(DistributionPoint distributionPoint : distributionPoints) {
					StringBuffer parmap = new StringBuffer();
					String CRLDistPointKey = "";
					String CRLDistPointValue = "";
					DistributionPointName distributionPointName = distributionPoint.getDistributionPoint();
					int type = distributionPointName.getType();
					ASN1Encodable distributionPointNameEncodable = distributionPointName.getName();
					if(distributionPointNameEncodable instanceof GeneralNames) {
						GeneralNames generalNames = (GeneralNames)distributionPointNameEncodable;
						GeneralName[] names = generalNames.getNames();
						for(GeneralName name : names) {

							String generalName = GeneralNameFinder.getGeneralName(name.getTagNo());
							DERIA5String dERIA5String =	(DERIA5String)name.getName();
							CRLDistPointValue = generalName + "=" + new String(dERIA5String.getOctets(), "UTF-8");
						}
					}

					if(type == 0) {
						CRLDistPointKey = "Full Name";
					}else if(type == 1) {
						CRLDistPointKey = "nameRelativeToCRLIssuer";
					}
					parmap.append(CRLDistPointKey).append(":").append(CRLDistPointValue);


					CRLDistPointList.add(parmap.toString());
				}
				extendsInfo.put(extendName, CRLDistPointList.toString());
			}
		}
	}

	/**
	 * 获取机构信息访问
	 * @param extendsInfo
	 * @param extensions
	 * @throws UnsupportedEncodingException
	 */
	private void getAuthorityInfoAccess(Map<String, Object> extendsInfo, Extensions extensions) throws UnsupportedEncodingException {
		AuthorityInformationAccess authorityInformationAccess = AuthorityInformationAccess.fromExtensions(extensions);
		if(null != authorityInformationAccess) {
			AccessDescription[] accessDescriptions = authorityInformationAccess.getAccessDescriptions();
			if(null != accessDescriptions && accessDescriptions.length > 0) {
				String extendName = CertExtendNameFinder.getExtendName(Extension.authorityInfoAccess);
				List<Map<String, Object>> authorityInfoAccessList = new ArrayList<Map<String,Object>>();
				for(AccessDescription accessDescription : accessDescriptions) {
					Map<String, Object> authorityInfoMap = new HashMap<String, Object>();
					ASN1ObjectIdentifier accessMethod = accessDescription.getAccessMethod();
					authorityInfoMap.put("Access Method", accessMethod.toString());
					GeneralName accessLocation = accessDescription.getAccessLocation();
					DERIA5String dERIA5String = (DERIA5String)accessLocation.getName();
					authorityInfoMap.put("Access Location", new String(dERIA5String.getOctets(), "UTF-8"));
					authorityInfoAccessList.add(authorityInfoMap);
				}
				extendsInfo.put(extendName, authorityInfoAccessList.toString());
			}
		}
	}

	/**
	 * 获取主体信息访问
	 * @param extendsInfo
	 * @param extensions
	 * @throws UnsupportedEncodingException
	 */
	private void getSubjectInformationAccess(Map<String, Object> extendsInfo,Extensions extensions) throws UnsupportedEncodingException {
		SubjectInformationAccess subjectInformationAccess = SubjectInformationAccess.fromExtensions(extensions);
		if(null != subjectInformationAccess) {
			AccessDescription[] accessDescriptions = subjectInformationAccess.getAccessDescriptions();
			if(null != accessDescriptions && accessDescriptions.length > 0) {
				String extendName = CertExtendNameFinder.getExtendName(Extension.subjectInfoAccess);
				List<Map<String, Object>> authorityInfoAccessList = new ArrayList<Map<String,Object>>();
				for(AccessDescription accessDescription : accessDescriptions) {
					Map<String, Object> authorityInfoMap = new HashMap<String, Object>();
					ASN1ObjectIdentifier accessMethod = accessDescription.getAccessMethod();
					authorityInfoMap.put("Access Method", accessMethod.toString());
					GeneralName accessLocation = accessDescription.getAccessLocation();
					String generalName = GeneralNameFinder.getGeneralName(accessLocation.getTagNo());
					DERIA5String dERIA5String = (DERIA5String)accessLocation.getName();
					authorityInfoMap.put("Access Location", generalName + "=" + new String(dERIA5String.getOctets(), "UTF-8"));
					authorityInfoAccessList.add(authorityInfoMap);
				}
				extendsInfo.put(extendName, authorityInfoAccessList.toString());
			}
		}
	}

	/**
	 * 获取名称限制
	 * @param extendsInfo
	 * @param extensions
	 */
	private void getNameConstraints(Map<String, Object> extendsInfo, Extensions extensions) {
//		ASN1Encodable extvalue = extensions.getExtensionParsedValue(Extension.nameConstraints);
//		NameConstraints nameConstraints = NameConstraints.getInstance(extvalue);
//		if (null != nameConstraints) {
//			String extendName = CertExtendNameFinder.getExtendName(Extension.nameConstraints);
//			PKIXNameConstraintValidator validator = new PKIXNameConstraintValidator();
//			GeneralSubtree[] excludedSubtrees = nameConstraints.getExcludedSubtrees();
//			GeneralSubtree[] permittedSubtrees = nameConstraints.getPermittedSubtrees();
//
//			if (permittedSubtrees != null) {
//				validator.intersectPermittedSubtree(permittedSubtrees);
//			}
//			if (excludedSubtrees != null) {
//				for (GeneralSubtree subtree : excludedSubtrees) {
//					validator.addExcludedSubtree(subtree);
//				}
//			}
//			extendsInfo.put(extendName,validator.toString());
//
//		}

		ASN1Encodable extensionParsedValue = extensions.getExtensionParsedValue(Extension.nameConstraints);
		NameConstraints nameConstraints = NameConstraints.getInstance(extensionParsedValue);
		if(null != nameConstraints) {
			String extendName = CertExtendNameFinder.getExtendName(Extension.nameConstraints);
			StringBuffer strbuf = new StringBuffer();
			GeneralSubtree[] permittedSubtrees = nameConstraints.getPermittedSubtrees();
			GeneralSubtree[] excludedSubtrees = nameConstraints.getExcludedSubtrees();
			if(null != permittedSubtrees && permittedSubtrees.length > 0) {
				strbuf.append("Permitted:");
				List<String> altNameList = new ArrayList<String>();
				for(GeneralSubtree permitted : permittedSubtrees) {
					GeneralName base = permitted.getBase();
					String generalName = GeneralNameFinder.getGeneralName(base.getTagNo());
					altNameList.add(generalName + "=" + base.getName());
				}
				strbuf.append(altNameList.toString());
			}
			if(null != excludedSubtrees && excludedSubtrees.length > 0) {
				strbuf.append("; Excluded:");
				List<String> altNameList = new ArrayList<String>();
				for(GeneralSubtree excluded : excludedSubtrees) {
					GeneralName base = excluded.getBase();
					String generalName = GeneralNameFinder.getGeneralName(base.getTagNo());
					altNameList.add(generalName + "=" + base.getName());
				}
				strbuf.append(altNameList.toString());
			}
			extendsInfo.put(extendName, strbuf.toString());
		}
	}

	/**
	 * 获取策略限制
	 * @param extendsInfo
	 * @param extensions
	 */
	private void getPolicyConstraints(Map<String, Object> extendsInfo, Extensions extensions) {
		PolicyConstraints policyConstraints = PolicyConstraints.fromExtensions(extensions);
		if(null != policyConstraints) {
			String extendName = CertExtendNameFinder.getExtendName(Extension.policyConstraints);
			List<String> policyConstraintStr = new ArrayList<String>();
			BigInteger inhibitPolicyMapping = policyConstraints.getInhibitPolicyMapping();
			BigInteger requireExplicitPolicyMapping = policyConstraints.getRequireExplicitPolicyMapping();
			if(null != inhibitPolicyMapping) {
				policyConstraintStr.add("inhibitPolicyMapping=" + inhibitPolicyMapping.toString(16));
			}
			if(null != requireExplicitPolicyMapping) {
				policyConstraintStr.add("requireExplicitPolicy=" + requireExplicitPolicyMapping.toString(16));
			}
			extendsInfo.put(extendName, policyConstraintStr.toString());
		}
	}

	/**
	 * 获取限制所有策略
	 * @param extendsInfo
	 * @param extensions
	 */
	private void getInhibitAnyPolicy(Map<String, Object> extendsInfo, Extensions extensions) {
		Extension inhibitAnyPolicyExtens = extensions.getExtension(Extension.inhibitAnyPolicy);
		if(null != inhibitAnyPolicyExtens) {
			String extendName = CertExtendNameFinder.getExtendName(Extension.inhibitAnyPolicy);
			int extValue = ASN1Integer.getInstance(inhibitAnyPolicyExtens.getParsedValue()).getValue().intValue();
			extendsInfo.put(extendName, extValue < 0 ? "0" : String.valueOf(extValue));
		}
	}

	/**
	 * 获取个人身份标识
	 * @param extendsInfo
	 * @param extensions
	 * @throws IOException
	 */
	private void getIdentifyCode(Map<String, Object> extendsInfo, Extensions extensions) throws IOException{
		Extension extvalue = extensions.getExtension(SM2Extensions.identifyCode);
		if(null != extvalue) {
			String extendName = CertExtendNameFinder.getExtendName(SM2Extensions.identifyCode);
			ASN1Encodable parsedValue = extvalue.getParsedValue();
			IdentifyCode identifyCode = IdentifyCode.getInstance(parsedValue);
			Map<String,String> map = new HashMap<String, String>();
			map.put(IdentifyCodeEnum.convert(identifyCode.getType()).desc, identifyCode.getIdentifyCode().toString());
			List<Map<String,String>> identifyCodeList = new ArrayList<Map<String,String>>();
			identifyCodeList.add(map);
			extendsInfo.put(extendName, identifyCodeList.toString());
		}
	}

	/**
	 * 获取个人社会保险号
	 * @param extendsInfo
	 * @param extensions
	 * @throws IOException void
	 */
	private void getInsuranceNumber(Map<String, Object> extendsInfo, Extensions extensions) throws IOException {
		ASN1Encodable extvalue = extensions.getExtension(SM2Extensions.insuranceNumber);
		if(null != extvalue) {
			String extendName = CertExtendNameFinder.getExtendName(SM2Extensions.insuranceNumber);
			ASN1Primitive asn1Primitive = extvalue.toASN1Primitive();
			if(asn1Primitive instanceof DERSequence) {
				ASN1Sequence asn1Sequence = (ASN1Sequence) asn1Primitive;
				DEROctetString derOctetString = (DEROctetString)asn1Sequence.getObjectAt(asn1Sequence.size()-1);
				logger.info("InsuranceNumber:"+Strings.fromByteArray(derOctetString.getOctets()));
				extendsInfo.put(extendName, Strings.fromByteArray(derOctetString.getOctets()));
			}
		}
	}

	/**
	 * 获取企业工商注册号
	 * @param extendsInfo
	 * @param extensions void
	 */
	private void getICRegistrationNumber(Map<String, Object> extendsInfo, Extensions extensions) {
		ASN1Encodable extvalue = extensions.getExtension(SM2Extensions.iCRegistrationNumber);
		if(null != extvalue) {
			String extendName = CertExtendNameFinder.getExtendName(SM2Extensions.iCRegistrationNumber);
			ASN1Primitive asn1Primitive = extvalue.toASN1Primitive();
			if(asn1Primitive instanceof DERSequence) {
				ASN1Sequence asn1Sequence = (ASN1Sequence) asn1Primitive;
				DEROctetString derOctetString = (DEROctetString)asn1Sequence.getObjectAt(asn1Sequence.size()-1);

				logger.info("ICRegistrationNumber:"+Strings.fromByteArray(derOctetString.getOctets()));
				extendsInfo.put(extendName, Strings.fromByteArray(derOctetString.getOctets()));
			}
		}
	}

	/**
	 * 企业组织机构代码
	 * @param extendsInfo
	 * @param extensions void
	 */
	private void getOrganizationCode(Map<String, Object> extendsInfo, Extensions extensions) {
		ASN1Encodable extvalue = extensions.getExtension(SM2Extensions.organizationCode);
		if(null != extvalue) {
			String extendName = CertExtendNameFinder.getExtendName(SM2Extensions.organizationCode);
			ASN1Primitive asn1Primitive = extvalue.toASN1Primitive();
			if(asn1Primitive instanceof DERSequence) {
				ASN1Sequence asn1Sequence = (ASN1Sequence) asn1Primitive;
				DEROctetString derOctetString = (DEROctetString)asn1Sequence.getObjectAt(asn1Sequence.size()-1);
				logger.info("OrganizationCode:"+Strings.fromByteArray(derOctetString.getOctets()));
				extendsInfo.put(extendName, Strings.fromByteArray(derOctetString.getOctets()));
			}
		}
	}

	/**
	 * 获取企业税号
	 * @param extendsInfo
	 * @param extensions void
	 */
	private void getTaxationNumber(Map<String, Object> extendsInfo, Extensions extensions) {
		ASN1Encodable extvalue = extensions.getExtension(SM2Extensions.taxationNumber);
		if(null != extvalue) {
			String extendName = CertExtendNameFinder.getExtendName(SM2Extensions.taxationNumber);
			ASN1Primitive asn1Primitive = extvalue.toASN1Primitive();
			if(asn1Primitive instanceof DERSequence) {
				ASN1Sequence asn1Sequence = (ASN1Sequence) asn1Primitive;
				DEROctetString derOctetString = (DEROctetString)asn1Sequence.getObjectAt(asn1Sequence.size()-1);
				logger.info("TaxationNumber:"+Strings.fromByteArray(derOctetString.getOctets()));
				extendsInfo.put(extendName, Strings.fromByteArray(derOctetString.getOctets()));
			}
		}
	}

	/**
	 * 获取名称列表
	 * @param alternativeNames
	 * @return
	 */
	private List<String> getGeneralNameList(Collection<List<?>> alternativeNames){
		List<String> altNameList = new ArrayList<String>();
		for(List<?> alternativeName : alternativeNames) {
			Integer tag = (Integer)alternativeName.get(0);
			String generalName = GeneralNameFinder.getGeneralName(tag);
			Object value = alternativeName.get(1);
			altNameList.add(generalName + "=" + value);
		}
		return altNameList;
	}

	/**
	 * 获取密钥用途
	 * @param keyUsages
	 * @return
	 */
	private List<String> getKeyUsage(boolean[] keyUsages){
		if(null == keyUsages) {
			return null;
		}
		List<String> usages = new ArrayList<String>();

		// digitalSignature        (0),
		if (keyUsages[0]) {
			usages.add("digitalSignature");
		}
		// nonRepudiation          (1),
		if (keyUsages[1]) {
			usages.add("nonRepudiation");
		}
		// keyEncipherment         (2),
		if (keyUsages[2]) {
			usages.add("keyEncipherment");
		}
		// dataEncipherment        (3),
		if (keyUsages[3]) {
			usages.add("dataEncipherment");
		}
		// keyAgreement            (4),
		if (keyUsages[4]) {
			usages.add("keyAgreement");
		}
		// keyCertSign             (5),
		if (keyUsages[5]) {
			usages.add("keyCertSign");
		}
		// cRLSign                 (6),
		if (keyUsages[6]) {
			usages.add("cRLSign");
		}
		// encipherOnly            (7),
		if (keyUsages[7]) {
			usages.add("encipherOnly");
		}
		// decipherOnly
		if (keyUsages[8]) {
			usages.add("decipherOnly");
		}
		return usages;
	}

	/**
	 * 获取密钥扩展用途
	 * @param extendedKeyUsage
	 * @return
	 */
	private List<String>  getExtendedKeyUsages(List<String> extendedKeyUsage){
		if(null == extendedKeyUsage) {
			return null;
		}
		List<String> extendUsagesList = new ArrayList<String>();
		for(String extendKeyUsage : extendedKeyUsage) {
			String property = "extendedkeyusage_" + extendKeyUsage;
			String extendUsage = env.getProperty(property);
			extendUsagesList.add(extendUsage == null ? extendKeyUsage : extendUsage);
		}
		return extendUsagesList;
	}

	/**
	 * 颁发者唯一标识符/主体唯一标识符 boolean转BitString
	 * @param id
	 * @return DERBitString
	 */
	private DERBitString booleanToBitString(boolean[] id)
	{
		byte[] bytes = new byte[(id.length + 7) / 8];
		for (int i = 0; i != id.length; i++)
		{
			bytes[i / 8] |= (id[i]) ? (1 << ((7 - (i % 8)))) : 0;
		}
		int pad = id.length % 8;
		if (pad == 0)
		{
			return new DERBitString(bytes);
		}
		else
		{
			return new DERBitString(bytes, 8 - pad);
		}
	}
}